<?php

declare(strict_types=1);

namespace Drupal\search_api_opensearch_semantic\Plugin\views\filter;

use Drupal\Core\Form\FormStateInterface;
use Drupal\search_api\Plugin\views\filter\SearchApiFilterTrait;
use Drupal\search_api_opensearch_semantic\Enum\SearchTypeEnum;
use Drupal\views\Attribute\ViewsFilter;

/**
 * Defines a filter for adding a fulltext semantic search to the view.
 *
 * @todo Create generic config options to store semantic and fulltext plugins.
 *   that way we can share the config across both the filter plugins. This will
 *   simplify buildForm() method as well.
 */
#[ViewsFilter(
  id: self::PLUGIN_ID,
)]
class SearchApiSemantic extends SearchApiSemanticSearchFilterBase {

  use SearchApiFilterTrait;

  public const string PLUGIN_ID = 'search_api_semantic';

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::buildOptionsForm($form, $form_state);
    // We don't need fulltext_search fields on the semantic filter.
    unset($form['fields']);
  }

  /**
   * {@inheritdoc}
   */
  public function query(): void {
    while (\is_array($this->value)) {
      $this->value = $this->value ? reset($this->value) : '';
    }
    // Catch empty strings entered by the user, but not "0".
    if ($this->value === '') {
      return;
    }
    $fields = $this->options['semantic_field'];
    $query = $this->getQuery();

    if ($this->options['parse_mode']) {
      /** @var \Drupal\search_api\ParseMode\ParseModeInterface $parse_mode */
      $parse_mode = $this->getParseModeManager()
        ->createInstance($this->options['parse_mode']);
      $query->setParseMode($parse_mode);
    }

    $query->setFulltextFields([$fields]);
    $query->setOption('min_score', $this->options['min_score']);
    $query->setOption('search_type', SearchTypeEnum::Semantic);
    $query->setOption('semantic_field', $this->options['semantic_field']);
    $query->keys($this->value);
  }

}
