<?php

namespace Drupal\search_api_opensearch_semantic\SearchAPI\Query;

use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_opensearch\SearchAPI\Query\SearchParamBuilder;
use Drupal\search_api_opensearch_semantic\Enum\SearchTypeEnum;
use Drupal\search_api_opensearch_semantic\Trait\SemanticIndexTrait;
use MakinaCorpus\Lucene\Query;

/**
 * Provides a search param builder for semantic search queries.
 */
class SemanticSearchParamBuilder extends SearchParamBuilder {

  use SemanticIndexTrait;

  public function __construct(
    protected SearchParamBuilder $inner,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function buildSearchParams(QueryInterface $query, array $indexFields, array $settings): array {
    $index = $query->getIndex();
    if ($this->isSemanticIndex($index->id()) === FALSE) {
      return $this->inner->buildSearchParams($query, $indexFields, $settings);
    }
    // Full text search.
    $keys = $query->getKeys();
    if (empty($keys)) {
      return [];
    }

    // Ensure $keys are an array.
    if (\is_string($keys)) {
      $keys = [$keys];
    }

    // Query string.
    $luceneQuery = $this->inner->buildSearchString($keys, $settings['fuzziness']);
    $params = [];
    if (!$luceneQuery->isEmpty()) {
      $params = match($query->getOption('search_type')) {
        SearchTypeEnum::Semantic => $this->buildNeuralParams($query, $luceneQuery),
        SearchTypeEnum::Hybrid => $this->buildHybridParams($query, $luceneQuery),
        default => $this->inner->buildSearchParams($query, $indexFields, $settings)
      };
    }

    return $params;
  }

  /**
   * Prepare query params for the hybrid search.
   */
  private function buildHybridParams(QueryInterface $query, Query $luceneQuery): array {
    $params = [];
    $index = $query->getIndex();
    // Prepare params for the hybrid search.
    $neuralQueries = [];
    $semanticField = $query->getOption('semantic_field', NULL);
    if ($semanticField !== NULL) {
      $neuralQueries['neural'][$semanticField] = [
        'query_text' => (string) $luceneQuery,
        'model_id' => $index->getThirdPartySetting('search_api_opensearch_semantic', 'model_id'),
      ];
    }

    $textQueries = [];
    $fulltextFields = array_reverse($query->getOption('fulltext_fields', NULL));
    $textQueries['bool']['should'] = [];
    foreach ($fulltextFields as $fieldId) {
      $textQueries['bool']['should'][] = ['match' => [$fieldId => (string) $luceneQuery]];
    }

    // Note: Following query sequence matters. When we create the
    // nlp-search-pipeline, we need to pass two weights in array. The first
    // weight considered to first query.
    // For example: If you gave [0.3, 0.7] weight while creating search pipeline
    // with normalization-processor, so 0.3 weight is passed for text query and
    // 0.7 for the neural query, in our case.
    $params['hybrid']['queries'] = [$textQueries, $neuralQueries];

    // For pagination_depth refer following link.
    // https://opensearch.org/blog/navigating-pagination-in-hybrid-queries-with-the-pagination_depth-parameter/
    // For now, keeping arbitrary big number.
    $params['hybrid']['pagination_depth'] = 100;
    return $params;
  }

  /**
   * Prepare query param for the neural search.
   */
  private function buildNeuralParams(QueryInterface $query, Query $luceneQuery): array {
    $params = [];
    $index = $query->getIndex();
    $semanticField = $query->getOption('semantic_field', NULL);
    if ($semanticField !== NULL) {
      $params['neural'][$semanticField] = [
        'query_text' => (string) $luceneQuery,
        'model_id' => $index->getThirdPartySetting('search_api_opensearch_semantic', 'model_id'),
        'min_score' => $query->getOption('min_score') ?? 0,
      ];
    }
    return $params;
  }

}
