<?php

declare(strict_types=1);

namespace Drupal\search_api_pinbyphrase\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Entity\View as ViewEntity;

/**
 * Configure Search API PinByPhrase settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'search_api_pinbyphrase_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['search_api_pinbyphrase.settings'];
  }

  /**
   * {@inheritdoc}
   *
   * @param string|\Drupal\search_api\IndexInterface|null $search_api_index
   *   Index parameter from the route. We treat it only for display context;
   *   the underlying configuration is global for all indexes.
   */
  public function buildForm(array $form, FormStateInterface $form_state, $search_api_index = NULL): array {
    $config = $this->config('search_api_pinbyphrase.settings');

    $stored_items = $config->get('pinned_items') ?? [];
    $allowed_views = $config->get('allowed_views') ?? [];

    if (!is_array($stored_items)) {
      $stored_items = [];
    }
    if (!is_array($allowed_views)) {
      $allowed_views = [];
    }

    // Initialize dynamic row count from form_state or config.
    $row_count = $form_state->get('pinned_items_count');
    if ($row_count === NULL) {
      $row_count = max(count($stored_items), 1);
      $form_state->set('pinned_items_count', $row_count);
    }

    // If we are rebuilding after add/remove, prefer submitted values for the
    // pinned_items defaults so the user doesn't lose input.
    if ($form_state->hasValue('pinned_items')) {
      $stored_items = $form_state->getValue('pinned_items');
      if (!is_array($stored_items)) {
        $stored_items = [];
      }
    }

    $form['#tree'] = TRUE;

    // Display index context, but do not store config per index.
    if ($search_api_index !== NULL) {
      if (is_object($search_api_index) && method_exists($search_api_index, 'label')) {
        $index_label = $search_api_index->label();
      }
      else {
        $index_label = (string) $search_api_index;
      }

      $form['index_info'] = [
        '#type' => 'item',
        '#title' => $this->t('Index'),
        '#markup' => $this->t('You are configuring the <strong>@index</strong> index. Settings are global and apply to all indexes.', [
          '@index' => $index_label,
        ]),
      ];
    }

    // ----------------------------------------------------------------------
    // Views restriction: only act on selected Search API views.
    // ----------------------------------------------------------------------
    $view_options = [];
    /** @var \Drupal\views\Entity\View[] $view_entities */
    $view_entities = \Drupal::entityTypeManager()
      ->getStorage('view')
      ->loadMultiple();

    foreach ($view_entities as $view_entity) {
      if (!$view_entity instanceof ViewEntity) {
        continue;
      }

      $base_table = (string) $view_entity->get('base_table');
      if (!str_starts_with($base_table, 'search_api_index_')) {
        continue;
      }

      $view_options[$view_entity->id()] = $this->t('@label (@id)', [
        '@label' => $view_entity->label(),
        '@id' => $view_entity->id(),
      ]);
    }

    $form['allowed_views'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Views where PinByPhrase is active'),
      '#description' => $this->t('Select the Search API views where pinned phrases should be applied. If none are selected, PinByPhrase will not alter any results.'),
      '#options' => $view_options,
      '#default_value' => $allowed_views,
    ];

    // ----------------------------------------------------------------------
    // Dynamic pinned items: rows of phrase / nid / langcode.
    // ----------------------------------------------------------------------
    $form['pinned_items_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'search-api-pinbyphrase-pinned-items-wrapper'],
    ];

    $form['pinned_items_wrapper']['pinned_items'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Exact phrase'),
        $this->t('Node ID'),
        $this->t('Language code'),
      ],
      '#empty' => $this->t('No pinned phrases configured.'),
    ];

    for ($delta = 0; $delta < $row_count; $delta++) {
      $phrase = $stored_items[$delta]['phrase'] ?? '';
      $nid = $stored_items[$delta]['nid'] ?? '';
      $langcode = $stored_items[$delta]['langcode'] ?? '';

      $form['pinned_items_wrapper']['pinned_items'][$delta]['phrase'] = [
        '#type' => 'textfield',
        '#default_value' => $phrase,
        '#size' => 40,
        '#maxlength' => 255,
        '#placeholder' => $this->t('Exact phrase to match'),
      ];

      $form['pinned_items_wrapper']['pinned_items'][$delta]['nid'] = [
        '#type' => 'number',
        '#default_value' => $nid,
        '#min' => 1,
        '#step' => 1,
        '#placeholder' => $this->t('Node ID'),
      ];

      $form['pinned_items_wrapper']['pinned_items'][$delta]['langcode'] = [
        '#type' => 'textfield',
        '#default_value' => $langcode,
        '#size' => 10,
        '#maxlength' => 12,
        '#placeholder' => $this->t('e.g. en, fr'),
      ];
    }

    // Add/remove buttons for dynamic rows.
    $form['pinned_items_wrapper']['actions'] = [
      '#type' => 'actions',
    ];

    $form['pinned_items_wrapper']['actions']['add_row'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add one more'),
      '#submit' => ['::addRow'],
      '#ajax' => [
        'callback' => '::ajaxPinnedItemsCallback',
        'wrapper' => 'search-api-pinbyphrase-pinned-items-wrapper',
      ],
    ];

    if ($row_count > 1) {
      $form['pinned_items_wrapper']['actions']['remove_row'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove one'),
        '#submit' => ['::removeRow'],
        '#ajax' => [
          'callback' => '::ajaxPinnedItemsCallback',
          'wrapper' => 'search-api-pinbyphrase-pinned-items-wrapper',
        ],
      ];
    }

    // Avoid caching the form structure because row count is dynamic.
    $form_state->setCached(FALSE);

    // Main submit.
    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save configuration'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * AJAX callback to refresh the pinned_items wrapper.
   */
  public function ajaxPinnedItemsCallback(array &$form, FormStateInterface $form_state): array {
    return $form['pinned_items_wrapper'];
  }

  /**
   * Submit callback for the "Add one more" button.
   */
  public function addRow(array &$form, FormStateInterface $form_state): void {
    $row_count = (int) $form_state->get('pinned_items_count');
    $row_count++;
    $form_state->set('pinned_items_count', $row_count);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Submit callback for the "Remove one" button.
   */
  public function removeRow(array &$form, FormStateInterface $form_state): void {
    $row_count = (int) $form_state->get('pinned_items_count');
    if ($row_count > 1) {
      $row_count--;
      $form_state->set('pinned_items_count', $row_count);
    }
    $form_state->setRebuild(TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValue(['pinned_items_wrapper', 'pinned_items']) ?? [];

    foreach ($values as $delta => $row) {
      $phrase = trim((string) ($row['phrase'] ?? ''));
      $nid = $row['nid'] ?? NULL;

      // Entirely empty row is allowed.
      if ($phrase === '' && ($nid === NULL || $nid === '')) {
        continue;
      }

      if ($phrase === '') {
        $form_state->setErrorByName("pinned_items_wrapper][pinned_items][$delta][phrase", $this->t('Phrase is required when a Node ID is provided.'));
      }

      if ($nid === NULL || $nid === '') {
        $form_state->setErrorByName("pinned_items_wrapper][pinned_items][$delta][nid", $this->t('Node ID is required when a phrase is provided.'));
      }
      elseif (!is_numeric($nid) || (int) $nid <= 0) {
        $form_state->setErrorByName("pinned_items_wrapper][pinned_items][$delta][nid", $this->t('Node ID must be a positive number.'));
      }
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValue(['pinned_items_wrapper', 'pinned_items']) ?? [];
    $pinned_items = [];

    foreach ($values as $row) {
      $phrase = trim((string) ($row['phrase'] ?? ''));
      $nid = $row['nid'] ?? NULL;

      if ($phrase === '' || $nid === NULL || $nid === '') {
        continue;
      }

      $langcode = trim((string) ($row['langcode'] ?? ''));

      $pinned_items[] = [
        'phrase' => $phrase,
        'nid' => (int) $nid,
        'langcode' => $langcode,
      ];
    }

    // Normalize allowed_views from checkboxes.
    $allowed_views = $form_state->getValue('allowed_views') ?? [];
    if (!is_array($allowed_views)) {
      $allowed_views = [];
    }
    $allowed_views = array_values(array_filter($allowed_views));

    $this->config('search_api_pinbyphrase.settings')
      ->set('pinned_items', $pinned_items)
      ->set('allowed_views', $allowed_views)
      ->save();

    parent::submitForm($form, $form_state);
  }

}

