<?php

declare(strict_types=1);

namespace Drupal\search_api_pinbyphrase\EventSubscriber;

use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\node\NodeInterface;
use Drupal\search_api_solr\Event\PostExtractResultsEvent;
use Drupal\search_api_solr\Event\SearchApiSolrEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Pins configured nodes to the top of matching search results.
 */
final class PinItemSubscriber implements EventSubscriberInterface {

  /**
   * The config storage service.
   *
   * @var \Drupal\Core\Config\StorageInterface
   */
  protected StorageInterface $configStorage;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * Constructs a new PinItemSubscriber.
   *
   * @param \Drupal\Core\Config\StorageInterface $config_storage
   *   The config storage.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   */
  public function __construct(StorageInterface $config_storage, LanguageManagerInterface $language_manager) {
    $this->configStorage = $config_storage;
    $this->languageManager = $language_manager;
  }

  /**
   * Reacts on Search API Solr results after they have been extracted.
   *
   * @param \Drupal\search_api_solr\Event\PostExtractResultsEvent $event
   *   The event object.
   */
  public function onPostExtractResults(PostExtractResultsEvent $event): void {
    $config = $this->configStorage->read('search_api_pinbyphrase.settings') ?? [];

    // --------------------------------------------------------------------
    // Limit processing to specific Search API views, if configured.
    // --------------------------------------------------------------------
    $allowed_views = $config['allowed_views'] ?? [];
    if (!is_array($allowed_views)) {
      $allowed_views = [];
    }

    if ($allowed_views) {
      $search_api_query = $event->getSearchApiQuery();
      $view = $search_api_query->getOption('search_api_view');

      // Queries not originating from a Search API view (e.g. autocomplete)
      // have no 'search_api_view' option, so skip them.
      if (!$view) {
        return;
      }

      $view_id = $view->id();
      if (!in_array($view_id, $allowed_views, TRUE)) {
        return;
      }
    }

    if (empty($config['pinned_items']) || !is_array($config['pinned_items'])) {
      return;
    }

    $pinned_items_config = $config['pinned_items'];

    if (!isset($search_api_query)) {
      $search_api_query = $event->getSearchApiQuery();
    }

    // Determine the current search phrase (keys) from the Search API query.
    $keys = $search_api_query->getOriginalKeys();

    if (is_array($keys)) {
      if (isset($keys['keys'])) {
        $keys = $keys['keys'];
      }
      else {
        $keys = reset($keys) ?: '';
      }
    }

    $keys = (string) $keys;
    $keys = trim($keys);
    if ($keys === '') {
      return;
    }
    $keys = mb_strtolower($keys);

    // Determine current language via language manager.
    $current_langcode = $this->languageManager->getCurrentLanguage()->getId();

    // Find configuration rows that match the current search phrase + language.
    $matching_config_rows = [];
    foreach ($pinned_items_config as $row) {
      $phrase = isset($row['phrase']) ? (string) $row['phrase'] : '';
      $phrase = mb_strtolower(trim($phrase));
      if ($phrase === '' || $phrase !== $keys) {
        continue;
      }

      $row_langcode = $row['langcode'] ?? '';
      if ($row_langcode !== '' && $current_langcode !== NULL && $row_langcode !== $current_langcode) {
        continue;
      }

      $nid = isset($row['nid']) ? (int) $row['nid'] : 0;
      if ($nid <= 0) {
        continue;
      }

      $matching_config_rows[] = [
        'phrase' => $phrase,
        'nid' => $nid,
        'langcode' => (string) $row_langcode,
      ];
    }

    if (!$matching_config_rows) {
      return;
    }

    $result_set = $event->getSearchApiResultSet();
    $result_items = $result_set->getResultItems();

    if (!$result_items) {
      return;
    }

    $pinned = [];

    // For each configured node, try to find and pin the corresponding result.
    foreach ($matching_config_rows as $config_row) {
      $nid = $config_row['nid'];
      $row_langcode = $config_row['langcode'];

      foreach ($result_items as $id => $item) {
        $original = $item->getOriginalObject();

        if (!$original) {
          continue;
        }

        $entity = $original->getValue();
        if (!$entity instanceof NodeInterface) {
          continue;
        }

        if ((int) $entity->id() !== $nid) {
          continue;
        }

        $entity_langcode = $entity->language()->getId();
        if ($row_langcode !== '' && $entity_langcode !== $row_langcode) {
          continue;
        }

        // Mark this result as pinned and remove it from the pool so we
        // preserve its relative order among pinned items.
        $pinned[$id] = $item;
        unset($result_items[$id]);
      }
    }

    if (!$pinned) {
      return;
    }

    // Put pinned items first, keep original order for the rest.
    $result_set->setResultItems($pinned + $result_items);
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      SearchApiSolrEvents::POST_EXTRACT_RESULTS => 'onPostExtractResults',
    ];
  }

}
