<?php

namespace Drupal\search_api_remote_admin\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\search_api_remote_admin\Entity\SearchApiRemoteIndexInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\search_api_remote_admin\Ajax\ContinueProcessingCommand;

/**
 * Provides a form for performing actions on a remote Search API index.
 */
class RemoteActionForm extends FormBase {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * Constructs a new RemoteActionForm object.
   */
  public function __construct(ClientInterface $http_client) {
    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('http_client')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'search_api_remote_admin_action_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, SearchApiRemoteIndexInterface $search_api_remote_index = NULL) {
    $form['#title'] = $this->t('Actions for %label', ['%label' => $search_api_remote_index->label()]);
    $form['#attached']['library'][] = 'search_api_remote_admin/progress_updater';

    $form['status'] = [
      '#type' => 'details',
      '#title' => $this->t('Current Status'),
      '#open' => TRUE,
      // Add a wrapper for AJAX replacement.
      '#prefix' => '<div id="search-api-remote-status-wrapper">',
      '#suffix' => '</div>',
    ];

    $status_data = $this->fetchStatus($search_api_remote_index);

    if (isset($status_data['error'])) {
      $form['status']['error'] = [
        '#markup' => $this->t('<strong>Error fetching status:</strong><br><pre>@error</pre>', ['@error' => $status_data['error']]),
      ];
      return $form;
    }

    $percentage = $status_data['data']['percentage_complete'] ?? 0;
    $form['status']['progress'] = [
      '#theme' => 'progress_bar',
      '#percent' => $percentage,
      '#message' => $this->t('@indexed of @total items indexed.', [
        '@indexed' => $status_data['data']['indexed_items'] ?? 'N/A',
        '@total' => $status_data['data']['total_items'] ?? 'N/A',
      ]),
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['requeue'] = [
      '#type' => 'submit',
      '#value' => $this->t('Re-queue all items'),
      '#submit' => ['::submitRequeue'],
    ];
    $form['actions']['process'] = [
      '#type' => 'submit',
      '#value' => ($percentage == 100) ? $this->t('Processing Complete') : $this->t('Process all pending items'),
      '#name' => 'process-button',
      '#disabled' => ($percentage == 100),
      '#ajax' => [
        'callback' => '::ajaxProcessCallback',
        'wrapper' => 'search-api-remote-status-wrapper',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Processing...'),
        ],
      ],
    ];
    $form['actions']['delete'] = [
      '#type' => 'submit',
      '#value' => $this->t('Delete all items from index'),
      '#submit' => ['::submitDelete'],
    ];

    $form_state->set('search_api_remote_index', $search_api_remote_index);
    return $form;
  }

  /**
   * AJAX callback for the process button.
   */
  public function ajaxProcessCallback(array &$form, FormStateInterface $form_state) {
    $entity = $form_state->get('search_api_remote_index');
    $this->doRequest($entity, 'process', ['limit' => $entity->getBatchSize()]);

    // Rebuild the status part of the form to get the updated progress bar.
    $status_element = $form['status'];
    $status_data = $this->fetchStatus($entity);
    $percentage = $status_data['data']['percentage_complete'] ?? 0;

    $status_element['progress'] = [
      '#theme' => 'progress_bar',
      '#percent' => $percentage,
      '#message' => $this->t('@indexed of @total items indexed.', [
        '@indexed' => $status_data['data']['indexed_items'] ?? 'N/A',
        '@total' => $status_data['data']['total_items'] ?? 'N/A',
      ]),
    ];

    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand('#search-api-remote-status-wrapper', $status_element));

    if ($percentage < 100 && !isset($status_data['error'])) {
      // Re-enable the button before the next automatic click.
      $response->addCommand(new InvokeCommand('[name="process-button"]', 'prop', ['disabled', FALSE]));
      $response->addCommand(new ContinueProcessingCommand('[name="process-button"]'));
    }
    else {
      // When finished, disable the button and change the text.
      $response->addCommand(new InvokeCommand('[name="process-button"]', 'prop', ['disabled', TRUE]));
      $response->addCommand(new InvokeCommand('[name="process-button"]', 'val', [$this->t('Processing Complete')]));
      $this->messenger()->addStatus($this->t('Processing finished for %label.', ['%label' => $entity->label()]));
    }

    return $response;
  }

  /**
   * Submit handler for the Re-queue button.
   */
  public function submitRequeue(array &$form, FormStateInterface $form_state) {
    $entity = $form_state->get('search_api_remote_index');
    $result = $this->doRequest($entity, 'index');
    if (!isset($result['error'])) {
      $this->messenger()->addStatus($this->t('Successfully sent command to re-queue all items for %label.', ['%label' => $entity->label()]));
    }
  }

  /**
   * Submit handler for the Delete button.
   */
  public function submitDelete(array &$form, FormStateInterface $form_state) {
    $entity = $form_state->get('search_api_remote_index');
    $result = $this->doRequest($entity, 'delete');
    if (!isset($result['error'])) {
      $this->messenger()->addStatus($this->t('Successfully sent command to delete all items for %label.', ['%label' => $entity->label()]));
    }
  }

  /**
   * Helper to perform a request.
   */
  protected function doRequest(SearchApiRemoteIndexInterface $entity, $operation, array $query = []) {
    $url = $entity->getEndpointUrl() . '/' . $operation;
    $options = $this->buildAuthOptions($entity);
    $options['query'] = $query;

    try {
      $response = $this->httpClient->request('POST', $url, $options);
      return json_decode($response->getBody(), TRUE);
    }
    catch (RequestException $e) {
      $this->messenger()->addError($this->t('Error performing operation: @message', ['@message' => $e->getMessage()]));
      return ['error' => $e->getMessage()];
    }
  }

  /**
   * Fetches the status from the remote API.
   */
  protected function fetchStatus(SearchApiRemoteIndexInterface $entity) {
    $url = $entity->getEndpointUrl() . '/status';
    $options = $this->buildAuthOptions($entity);

    try {
      $response = $this->httpClient->request('GET', $url, $options);
      return json_decode($response->getBody(), TRUE);
    }
    catch (RequestException $e) {
      return ['error' => $e->getMessage()];
    }
  }

  /**
   * Builds Guzzle auth options.
   */
  protected function buildAuthOptions(SearchApiRemoteIndexInterface $entity) {
    $options = [];
    if ($entity->getAuthType() === 'basic_auth') {
      $auth_config = $entity->getAuthConfig();
      $options['auth'] = [
        $auth_config['username'] ?? '',
        $auth_config['password'] ?? '',
      ];
    }
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Default submit handler, not used for AJAX buttons.
  }

}