<?php

namespace Drupal\search_api_remote_admin\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;

/**
 * Form handler for the Search API Remote Index add/edit form.
 */
class SearchApiRemoteIndexForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    $entity = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $entity->label(),
      '#description' => $this->t('A label for this remote index.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity->id(),
      '#machine_name' => [
        'exists' => '\Drupal\search_api_remote_admin\Entity\SearchApiRemoteIndex::load',
      ],
      '#disabled' => !$entity->isNew(),
    ];

    $form['endpoint_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Endpoint URL'),
      '#description' => $this->t('The base URL of the remote Search API endpoint (e.g., https://example.com/api/search-api-remote/my_index).'),
      '#default_value' => $entity->getEndpointUrl(),
      '#required' => TRUE,
    ];

    $form['batch_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Batch Size'),
      '#description' => $this->t('The default number of items to process per request when using the "process" operation via the UI.'),
      '#default_value' => $entity->getBatchSize(),
      '#required' => TRUE,
      '#min' => 1,
    ];

    $form['authentication'] = [
      '#type' => 'details',
      '#title' => $this->t('Authentication'),
      '#open' => TRUE,
      '#tree' => TRUE, // Important for nested values.
    ];

    $form['authentication']['auth_type'] = [
      '#type' => 'radios',
      '#title' => $this->t('Authentication Type'),
      '#options' => [
        'none' => $this->t('None'),
        'basic_auth' => $this->t('Basic Auth'),
      ],
      '#default_value' => $entity->getAuthType(),
    ];

    $auth_config = $entity->getAuthConfig();
    $form['authentication']['auth_config']['username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username'),
      '#default_value' => $auth_config['username'] ?? '',
      '#states' => [
        'visible' => [
          ':input[name="authentication[auth_type]"]' => ['value' => 'basic_auth'],
        ],
      ],
    ];

    $form['authentication']['auth_config']['password'] = [
      '#type' => 'password',
      '#title' => $this->t('Password'),
      '#description' => $this->t('Leave blank to keep the existing password. The password is not stored securely.'),
      '#states' => [
        'visible' => [
          ':input[name="authentication[auth_type]"]' => ['value' => 'basic_auth'],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $entity = $this->entity;

    // Manually handle the auth_config mapping because of the nesting.
    $auth_values = $form_state->getValue('authentication');
    $entity->set('auth_type', $auth_values['auth_type']);

    if ($auth_values['auth_type'] === 'basic_auth') {
      $existing_auth_config = $entity->getAuthConfig();
      $new_auth_config = [];
      $new_auth_config['username'] = $auth_values['auth_config']['username'];

      // Only update the password if a new one was entered.
      if (!empty($auth_values['auth_config']['password'])) {
        $new_auth_config['password'] = $auth_values['auth_config']['password'];
      }
      else {
        // Keep the old password if it exists.
        $new_auth_config['password'] = $existing_auth_config['password'] ?? '';
      }
      $entity->set('auth_config', $new_auth_config);
    }
    else {
      $entity->set('auth_config', []);
    }

    $status = $entity->save();

    if ($status) {
      $this->messenger()->addStatus($this->t('Saved the %label Search API Remote Index.', [
        '%label' => $entity->label(),
      ]));
    }
    else {
      $this->messenger()->addError($this->t('The %label Search API Remote Index could not be saved.', [
        '%label' => $entity->label(),
      ]));
    }

    $form_state->setRedirectUrl($entity->toUrl('collection'));
  }

}