<?php

namespace Drupal\search_api_remote_admin;

use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Url;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a listing of Search API Remote Index entities.
 */
class SearchApiRemoteIndexListBuilder extends ConfigEntityListBuilder {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('http_client')
    );
  }

  /**
   * Constructs a new SearchApiRemoteIndexListBuilder object.
   */
  public function __construct(EntityTypeInterface $entity_type, EntityStorageInterface $storage, ClientInterface $http_client) {
    parent::__construct($entity_type, $storage);
    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $build = [];
    $entities = $this->load();
    $groups = [];

    // Group entities by domain.
    foreach ($entities as $entity) {
      $host = parse_url($entity->getEndpointUrl(), PHP_URL_HOST);
      if (empty($host)) {
        $host = $this->t('Undefined Domain');
      }
      $groups[$host][$entity->id()] = $entity;
    }

    // Build a separate table for each group.
    foreach ($groups as $host => $group_entities) {
      $build[$host] = [
        '#type' => 'details',
        '#title' => $host,
        '#open' => TRUE,
      ];

      $build[$host]['table'] = [
        '#type' => 'table',
        '#header' => $this->buildHeader(),
        '#rows' => [],
        '#empty' => $this->t('There are no remote indexes for this domain.'),
      ];

      foreach ($group_entities as $entity) {
        if ($row = $this->buildRow($entity)) {
          $build[$host]['table']['#rows'][] = $row;
        }
      }
    }

    if (empty($build)) {
      $build['empty'] = [
        '#markup' => $this->t('There are no remote indexes defined. <a href="@link">Add one</a>.', [
          '@link' => Url::fromRoute('entity.search_api_remote_index.add_form')->toString(),
        ]),
      ];
    }

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['label'] = $this->t('Label');
    $header['endpoint_url'] = $this->t('Endpoint URL');
    $header['status'] = $this->t('Status');
    $header['operations'] = $this->t('Operations');
    return $header;
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    $row['label'] = $entity->label();
    $row['endpoint_url'] = $entity->getEndpointUrl();

    $status_data = $this->fetchStatus($entity);
    if (isset($status_data['error'])) {
      $row['status'] = $this->t('Error fetching status');
    }
    else {
      $indexed = $status_data['data']['indexed_items'] ?? 'N/A';
      $total = $status_data['data']['total_items'] ?? 'N/A';
      $percentage = $status_data['data']['percentage_complete'] ?? 'N/A';
      $row['status'] = $this->t('@indexed / @total (@percentage%)', [
        '@indexed' => $indexed,
        '@total' => $total,
        '@percentage' => $percentage,
      ]);
    }

    $row['operations']['data'] = $this->buildOperations($entity);

    return $row;
  }

  /**
   * {@inheritdoc}
   */
  public function getOperations(EntityInterface $entity) {
    $operations = parent::getOperations($entity);

    // Manually build the URL as a workaround for stubborn config caching issues.
    $operations['actions'] = [
      'title' => $this->t('Actions'),
      'weight' => 10,
      'url' => Url::fromRoute('entity.search_api_remote_index.actions_form', ['search_api_remote_index' => $entity->id()]),
    ];

    return $operations;
  }

  /**
   * Fetches the status from the remote API.
   */
  protected function fetchStatus(EntityInterface $entity) {
    $url = $entity->getEndpointUrl() . '/status';
    $options = $this->buildAuthOptions($entity);

    try {
      $response = $this->httpClient->request('GET', $url, $options);
      return json_decode($response->getBody(), TRUE);
    }
    catch (RequestException $e) {
      return ['error' => $e->getMessage()];
    }
  }

  /**
   * Builds Guzzle auth options.
   */
  protected function buildAuthOptions(EntityInterface $entity) {
    $options = [];
    if ($entity->getAuthType() === 'basic_auth') {
      $auth_config = $entity->getAuthConfig();
      $options['auth'] = [
        $auth_config['username'] ?? '',
        $auth_config['password'] ?? '',
      ];
    }
    return $options;
  }

}