<?php

namespace Drupal\search_api_remote\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\search_api\IndexInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\search_api\Task\IndexTaskManager;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for handling remote Search API actions.
 */
class SearchApiRemoteController extends ControllerBase {

  /**
   * The Search API index task manager.
   *
   * @var \Drupal\search_api\Task\IndexTaskManager
   */
  protected $indexTaskManager;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a new SearchApiRemoteController object.
   *
   * @param \Drupal\search_api\Task\IndexTaskManager $index_task_manager
   *   The Search API index task manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(IndexTaskManager $index_task_manager, RequestStack $request_stack) {
    $this->indexTaskManager = $index_task_manager;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('search_api.index_task_manager'),
      $container->get('request_stack')
    );
  }

  /**
   * Handles the remote action on a search index.
   *
   * @param \Drupal\search_api\IndexInterface $search_api_index
   *   The search index entity.
   * @param string $operation
   *   The operation to perform (e.g., 'index', 'delete', 'status', 'process').
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function control(IndexInterface $search_api_index, string $operation): JsonResponse {
    if (!$search_api_index || !$search_api_index->status()) {
      throw new NotFoundHttpException('Search API index not found or disabled.');
    }

    try {
      switch ($operation) {
        case 'index':
          $search_api_index->reindex();
          $message = sprintf('All items in index "%s" have been queued for re-indexing.', $search_api_index->label());
          break;

        case 'delete':
          $search_api_index->clear();
          $message = sprintf('All items have been deleted from index "%s".', $search_api_index->label());
          break;

        case 'status':
          $tracker = $search_api_index->getTrackerInstance();
          $total = $tracker->getTotalItemsCount();
          $indexed = $tracker->getIndexedItemsCount();
          $percentage = $total > 0 ? round(($indexed / $total) * 100, 2) : 100;

          return new JsonResponse([
            'status' => 'success',
            'index' => $search_api_index->id(),
            'operation' => 'status',
            'data' => [
              'total_items' => $total,
              'indexed_items' => $indexed,
              'remaining_items' => $total - $indexed,
              'percentage_complete' => $percentage,
            ],
          ]);

        case 'process':
          $request = $this->requestStack->getCurrentRequest();
          $limit = $request->query->get('limit', -1);
          $limit = is_numeric($limit) ? (int) $limit : -1;

          $limit_for_message = ($limit == -1) ? 'one batch (as configured in settings)' : $limit . ' items';

          $search_api_index->indexItems($limit);

          $message = sprintf('Processed up to %s for index "%s".', $limit_for_message, $search_api_index->label());
          $remaining = $search_api_index->getTrackerInstance()->getRemainingItemsCount();

          return new JsonResponse([
            'status' => 'success',
            'message' => $message,
            'index' => $search_api_index->id(),
            'operation' => 'process',
            'data' => [
              'processed_with_limit' => $limit,
              'remaining_items' => $remaining,
            ],
          ]);

        default:
          return new JsonResponse([
            'error' => 'Invalid operation',
            'message' => sprintf('The operation "%s" is not supported. Use "index", "delete", "status", or "process".', $operation),
          ], 400);
      }

      return new JsonResponse([
        'status' => 'success',
        'message' => $message,
        'index' => $search_api_index->id(),
        'operation' => $operation,
      ]);

    }
    catch (\Exception $e) {
      return new JsonResponse([
        'status' => 'error',
        'message' => $e->getMessage(),
      ], 500);
    }
  }

}
