<?php

namespace Drupal\search_api_sitestudio_processor\Plugin\search_api\processor\Property;

use Drupal\cohesion_elements\CustomComponentDiscoveryInterface;
use Drupal\cohesion_elements\Entity\Component;
use Drupal\cohesion_elements\Entity\ComponentCategory;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\search_api\Item\FieldInterface;
use Drupal\search_api\Processor\ConfigurablePropertyBase;
/**
 * Defines a "sitestudio_components_value item" property.
 *
 * @see \Drupal\search_api_sitestudio_processor\Plugin\search_api\processor\SitestudioComponentsItem
 */
class SitestudioComponentsProperty extends ConfigurablePropertyBase {

  use StringTranslationTrait;

  /**
   * Custom component discovery service.
   *
   * @var \Drupal\cohesion_elements\CustomComponentDiscoveryInterface
   */
  protected $customComponentDiscovery;


  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
  */
  protected $entityTypeManager;

  /**
   * Sitestudio Component Property constructor.
   *
   * @param array $values
   * @param \Drupal\cohesion_elements\CustomComponentDiscoveryInterface $customComponentDiscovery
   *   Custom component discovery service.
   */
  public function __construct(array $values, CustomComponentDiscoveryInterface $customComponentDiscovery, EntityTypeManagerInterface $entityTypeManager) {
    $this->customComponentDiscovery = $customComponentDiscovery;
    $this->entityTypeManager = $entityTypeManager;
    $this->definition = $values;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      "sitestudio_components" => [
        "all_components" => TRUE,
      ],
      "components" => []
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(FieldInterface $field, array $form, FormStateInterface $form_state) {
    $form['#tree'] = TRUE;
    $configuration = $field->getConfiguration();

    $form['sitestudio_components_message'] = [
      '#type' => 'item',
      '#description' => $this->t('By default the processor will retrieve the values from all custom components but you can choose in this configuration which custom components you want to retrieve organized by component categories.'),
    ];

    $form['sitestudio_components'] = [
      '#type' => 'item',
      '#description' => NULL,
    ];

    $all_components_config = $configuration['sitestudio_components']['all_components'] ?? [];

    $form['sitestudio_components']['all_components'] = [
      '#type' => 'checkbox',
      '#title' => "Enable all sitestudio components",
      '#title_display' => 'before',
      '#description' => 'Toggle this button to add all the sitestudio components to the list of components enabled to being processed.',
      '#required' => FALSE,
      '#default_value' => $all_components_config == 1 ?? TRUE,
    ];

     // Retrieve sitestudio components categories
     /** @var array<\Drupal\cohesion_elements\Entity\ComponentCategory> $components_categories */
    $components_categories = $this->entityTypeManager->getStorage('cohesion_component_category')->loadMultiple();

    $sitestudio_components = $this->entityTypeManager->getStorage('cohesion_component')->loadMultiple();

    // Retrieve custom sitestudio components
    $sitestudio_custom_components = $this->customComponentDiscovery->getComponents();

    $form['components_categories'] = [
      '#type' => 'vertical_tabs',
      '#default_tab' => NULL,
    ];

    foreach ($components_categories as $componentCategory) {
      $custom_components_by_category = $this->getComponentsInCategory($sitestudio_custom_components, $componentCategory);
      $components_by_category = $this->getComponentsInCategory($sitestudio_components, $componentCategory);

      $components = array_merge($custom_components_by_category, $components_by_category);

      if (empty($components)) {
        continue;
      }

      $category_key = "{$componentCategory->id()}:{$componentCategory->getClass()}";

      $form['components'][$category_key] = [
        '#type' => 'details',
        '#title' => $componentCategory->label(),
        '#group' => 'components_categories',
      ];

      $form['components'][$category_key]['full_category'] = [
        '#type' => 'checkbox',
        '#title' => "Enable full {$componentCategory->label()} category",
        '#title_display' => 'after',
        '#suffix' => '<hr>',
        '#description' => 'Toggle this button to add all the components with this category to the list of components enabled to being processed.',
        '#required' => FALSE,
        '#default_value' => isset($configuration['components'][$category_key]['full_category']) ? $configuration['components'][$category_key]['full_category'] == 1 : FALSE,
      ];
      
      foreach ($components as $component) {
        $component_label = $component instanceof Component ? $component->label() : $component['name'];
        $component_id = $component instanceof Component ? $component->id() : $component['machine_name'];

        $form['components'][$category_key][$component_id] = [
          '#type' => 'checkbox',
          '#title' => "Enable {$component_label}",
          '#title_display' => 'after',
          '#description' => 'Toggle this button to add the component to the list of components enabled to being processed.',
          '#required' => FALSE,
          '#default_value' => isset($configuration['components'][$category_key][$component_id]) ? $configuration['components'][$category_key][$component_id] == 1 : FALSE,
        ];
      }
    }

    return $form;
  }


  /**
   *  Get custom components of a certain category.
   *
   * @param \Drupal\cohesion_elements\Entity\ComponentCategory $category
   *
   * @return array
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function getComponentsInCategory($components, ComponentCategory $category): array {
    $componentsInCategory = [];
    if ($components) {
      foreach ($components as $component) {
        $componentCategory = $component instanceof Component ? $component->getCategory() : $component['category'];

        if ($componentCategory == $category->id()) {
          $componentsInCategory[] = $component;
        }
      }
    }

    return $componentsInCategory;
  }
}
