<?php

namespace Drupal\search_api_sitestudio_processor\Plugin\search_api\processor;

use Drupal\cohesion\LayoutCanvas\LayoutCanvas;
use Drupal\cohesion_elements\CustomComponentDiscoveryInterface;
use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\RfcLogLevel;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Utility\Error;
use Drupal\search_api_sitestudio_processor\Plugin\search_api\processor\Property\SitestudioComponentsProperty;
use Drupal\search_api\Attribute\SearchApiProcessor;
use Drupal\search_api\Datasource\DatasourceInterface;
use Drupal\search_api\Item\ItemInterface;
use Drupal\search_api\LoggerTrait;
use Drupal\search_api\Processor\ProcessorPluginBase;
use Drupal\search_api\Utility\PostRequestIndexingInterface;
use HTMLPurifier;
use HTMLPurifier_Config;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Add an additional field containing the values from your custom sitestudio components.
 *
 */
#[SearchApiProcessor(
  id: 'sitestudio_component_item',
  label: new TranslatableMarkup('Sitestudio component item'),
  description: new TranslatableMarkup('Adds an additional field containing the values from your custom sitestudio components.'),
  stages: [
    'add_properties' => 0,
  ],
  locked: TRUE,
  hidden: TRUE,
)]
class SitestudioComponentsItem extends ProcessorPluginBase {

  use LoggerTrait;

  /**
   * The post request indexing service.
   */
  protected PostRequestIndexingInterface $postRequestIndexing;

  /**
   * Custom component discovery service.
   *
   * @var \Drupal\cohesion_elements\CustomComponentDiscoveryInterface
   */
  protected $customComponentDiscovery;


  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    /** @var static $plugin */
    $plugin = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $plugin->setPostRequestIndexing($container->get('search_api.post_request_indexing'));
    $plugin->setLogger($container->get('logger.channel.search_api'));
    $plugin->setCustomComponentDiscovery($container->get('cohesion_elements.custom_component.discovery'));
    $plugin->setEntityTypeManager($container->get('entity_type.manager'));

    return $plugin;
  }

  /**
   * Retrieves the entity type manager service.
   *
   * @return \Drupal\Core\Entity\EntityTypeManagerInterface
   *   The entity type manager service.
   */
  public function getEntityTypeManager(): EntityTypeManagerInterface {
    return $this->entityTypeManager ?? \Drupal::service('entity_type.manager');
  }

  /**
   * Retrieves the sitestudio custom components service.
   *
   * @return \Drupal\cohesion_elements\CustomComponentDiscoveryInterface
   *   The sitestudio custom components service.
   */
  public function getCustomComponentDiscovery(): CustomComponentDiscoveryInterface {
    return $this->customComponentDiscovery ?? \Drupal::service('cohesion_elements.custom_component.discovery');
  }


  /**
   * Retrieves the post request indexing service.
   *
   * @return \Drupal\search_api\Utility\PostRequestIndexingInterface
   *   The post request indexing service.
   */
  public function getPostRequestIndexing(): PostRequestIndexingInterface {
    return $this->postRequestIndexing ?? \Drupal::service('search_api.post_request_indexing');
  }

  /**
   * Sets the entity type manager service.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface
   *   The new entity type manager service.
   *
   * @return $this
   */
  public function setEntityTypeManager(EntityTypeManagerInterface $entityTypeManager,): static {
    $this->entityTypeManager = $entityTypeManager;
    return $this;
  }

  /**
   * Sets the post request indexing service.
   *
   * @param \Drupal\cohesion_elements\CustomComponentDiscoveryInterface $customComponentDiscovery
   *   The new custom component discovery service.
   *
   * @return $this
   */
  public function setCustomComponentDiscovery(CustomComponentDiscoveryInterface $customComponentDiscovery,): static {
    $this->customComponentDiscovery = $customComponentDiscovery;
    return $this;
  }

  /**
   * Sets the sitestudio custom component discovery service.
   *
   * @param \Drupal\search_api\Utility\PostRequestIndexingInterface $post_request_indexing
   *   The new post request indexing service.
   *
   * @return $this
   */
  public function setPostRequestIndexing(PostRequestIndexingInterface $post_request_indexing,): static {
    $this->postRequestIndexing = $post_request_indexing;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getPropertyDefinitions(?DatasourceInterface $datasource = NULL) {
    $properties = [];

    if (!$datasource) {
      $customComponentDiscoveryService = $this->getCustomComponentDiscovery();
      $entityTypeManager = $this->getEntityTypeManager();

      $definition = [
        'label' => $this->t('Sitestudio Components'),
        'description' => $this->t('All the values retrieved from the fields inside your sitestudio custom components'),
        'type' => 'search_api_html',
        'processor_id' => $this->getPluginId(),
      ];
      $properties['sitestudio_components_value'] = new SitestudioComponentsProperty($definition, $customComponentDiscoveryService, $entityTypeManager);
    }

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function addFieldValues(ItemInterface $item) {
    $fields = $this->getFieldsHelper()
      ->filterForPropertyPath($item->getFields(), NULL, 'sitestudio_components_value');
    foreach ($fields as $field) {
      try {
        /** @var \Drupal\node\NodeInterface $node_item */
        $node_item = $item->getOriginalObject()->getValue();
        $configuration = $field->getConfiguration();

        if ($node_item) {
          $node_fields = $node_item->getFields();

          // Retrieve all the layout canvas fields in the node.
          $layout_canvas_fields = array_filter($node_fields, function ($field_item) {
            return $field_item->getFieldDefinition()->getType() == "cohesion_entity_reference_revisions";
          });

          if (!empty($layout_canvas_fields)) {
            foreach ($layout_canvas_fields as $layout_canvas_field_item_key => $layout_canvas_field_item) {
              if ($node_item->{$layout_canvas_field_item_key} && $node_item->{$layout_canvas_field_item_key}->target_id) {
                $entityTypeManager = $this->getEntityTypeManager();

                $layout_canvas_target_id = $node_item->{$layout_canvas_field_item_key}->target_id;
                $content_layout = $entityTypeManager->getStorage('cohesion_layout')->load($layout_canvas_target_id);

                if ($content_layout) {
                  $layout_canvas = new LayoutCanvas($content_layout->json_values->value);

                  $decoded_canvas = $layout_canvas->getJsonValuesDecodedArray();

                  $components_uuids = $this->retrieveComponentsUuid($decoded_canvas, $configuration);

                  if (empty($components_uuids)) {
                    continue;
                  }

                  $components_info = $this->retrieveComponentsInfoFromUuid($decoded_canvas, $components_uuids);

                  if (!empty($components_info)) {
                    $components_info_string = implode(" ", $components_info);

                    $field->addValue($components_info_string);
                  }
                }
              }
            }
          }
        }
      } catch (\Throwable $e) {
        // This could throw all kinds of exceptions in specific scenarios, so we
        // just catch all of them here. Not having a field value for this field
        // probably makes sense in that case, so we just log an error and
        // continue.
        $variables = [
          '%item_id' => $item->getId(),
          '%index' => $this->index->label() ?? $this->index->id(),
        ];
        $variables += Error::decodeException($e);
        $level = RfcLogLevel::ERROR;

        // Special case: If this happened during post-request indexing (that is,
        // via the "Index items immediately" functionality) there is a chance
        // that this problem doesn't occur when indexing during cron. Therefore,
        // add a warning to the item so it will not be marked as "indexed" in
        // the tracker and will get reindexed during the next cron run.
        if ($this->getPostRequestIndexing()->isIndexingActive()) {
          $item->addWarning($this->t('%type while trying to retrieve custom components value from item %item_id for search index %index: @message in %function (line %line of %file).', $variables));
          // Only log a warning in this case instead of an error. If rendering
          // still fails during cron then an error will be logged at that point.
          $level = RfcLogLevel::WARNING;
        }

        $this->getLogger()->log(
          $level,
          '%type while trying to retrieve custom components value from item %item_id for search index %index: @message in %function (line %line of %file).',
          $variables,
        );
      }
    }
  }

  /**
   * Retrieve the sitestudio components info from uuid using the model property.
   */
  private function retrieveComponentsInfoFromUuid(array $decoded_canvas, array $components_uuids): array {
    $components_info = [];

    if (empty($components_uuids)) {
      return [];
    }

    if (isset($decoded_canvas['model'])) {
      foreach ($components_uuids as $component_uuid) {
        if (isset($decoded_canvas['model'][$component_uuid])) {
          $component_informations = $decoded_canvas['model'][$component_uuid];

          $component_info_string = [];

          foreach ($component_informations as $component_key => $component_value) {
            $can_add = !empty($component_value) && $component_key != "settings" && is_array($component_value) == FALSE && strpos($component_value, "media-reference") == FALSE && is_numeric($component_value) == FALSE && count(explode(" ", $component_value)) > 1;

            // Check if the value is a sentence string and if the value is not configured as a richtext.
            if ($can_add == FALSE && !isset($component_value['text'])) {
              continue;
            }

            // Logic to handle richtext values
            if (isset($component_value['text'])) {
              $strip_tags_with_spaces = str_replace('  ', ' ', strip_tags(str_replace('<', ' <', $component_value['text'])));

              $component_info_string[] = Html::decodeEntities($strip_tags_with_spaces);
            } else {
              $component_info_string[] = $component_value;
            }
          }

          $components_info[] = implode(',', $component_info_string);
        }
      }
    }

    return $components_info;
  }

  /**
   * Retrieve the components uuid from the layout_canvas field.
   */
  private function retrieveComponentsUuid(array $decoded_canvas, array $configuration): array {
    $components_uuids = [];
    $all_components = empty($configuration) || $configuration['sitestudio_components']['all_components'] == 1;

    if (empty($decoded_canvas) || (isset($decoded_canvas['canvas']) && empty($decoded_canvas['canvas']))) {
      return [];
    }

    foreach ($decoded_canvas['canvas'] as $canvas_item) {
      $canvas_item_to_retrieve = $canvas_item;

      if ($all_components == FALSE) {
        $component_available = FALSE;

        foreach ($configuration['components'] as $category_key => $component_list_item) {
          $component_category = $canvas_item['category'];
          $exploded_category_key = explode(":", $category_key);

          if ($component_list_item['full_category'] == 1 && $component_category == $exploded_category_key[1]) {
            // The component is available because the component category is enabled.
            $component_available = TRUE;
          } else if (isset($component_list_item[$canvas_item['uid']])) {
            // The component is available only if the single component is enabled.
            $component_available = $component_list_item[$canvas_item['uid']] == 1;

            // If the component is not available to process than check if contains other components and pass the childre item to the retrieve uuid function
            if ($component_available == FALSE && isset($canvas_item['children'])) {
              $component_available = TRUE;
              $canvas_item_to_retrieve = $canvas_item['children'];
            }
          }
        }

        if ($component_available == FALSE) {
          continue;
        }
      }

      $uuid_list_canvas_item = $this->retrieveCanvasItem($canvas_item_to_retrieve);

      if (!empty($uuid_list_canvas_item)) {
        $components_uuids = array_merge($components_uuids, $uuid_list_canvas_item);
      }
    }

    return $components_uuids;
  }

  /**
   * Retrieve the components uuids using also recursion to retrieve children values.
   */
  private function retrieveCanvasItem(array $canvas_item): array {
    $uuids = [];

    if (empty($canvas_item)) {
      return [];
    }

    $canvas_item_to_process = isset($canvas_item['uuid']) ? $canvas_item : $canvas_item[0];

    if (isset($canvas_item_to_process['uuid'])) {
      $uuids[] = $canvas_item_to_process['uuid'];
    }

    if (isset($canvas_item_to_process['children']) && count($canvas_item_to_process['children']) >= 1) {
      foreach ($canvas_item_to_process['children'] as $canvas_item_children) {
        $children_uuids = $this->retrieveCanvasItem($canvas_item_children);

        if (!empty($children_uuids)) {
          $uuids = array_merge($uuids, $children_uuids);
        }
      }
    }

    return $uuids;
  }
}
