<?php

declare(strict_types=1);

namespace Drupal\search_api_solr_dense_vector\EventSubscriber;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Embeddings\EmbeddingsInput;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\search_api\Plugin\search_api\data_type\value\TextValue;
use Drupal\search_api\SearchApiException;
use Drupal\search_api_solr\Event\PostConvertedQueryEvent;
use Drupal\search_api_solr\Event\PostExtractResultsEvent;
use Drupal\search_api_solr\Event\SearchApiSolrEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Alters the query to re-rank search with vectors.
 */
class SearchApiSolrDenseVectorQuerySubscriber implements EventSubscriberInterface {

  /**
   * Event subscriber constructor.
   */
  public function __construct(
    protected AiProviderPluginManager $aiProviderManager,
    protected LoggerChannelFactoryInterface $loggerFactory,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      SearchApiSolrEvents::POST_CONVERT_QUERY => 'postConvertQuery',
      SearchApiSolrEvents::POST_EXTRACT_RESULTS => 'postExtractResults',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function postConvertQuery(PostConvertedQueryEvent $event): void {
    $query = $event->getSearchApiQuery();
    $solarium_query = $event->getSolariumQuery();

    if ($query->getIndex()->isValidProcessor('solr_densevector')) {
      try {
        $processor = $query->getIndex()->getProcessor('solr_densevector');
        $settings = $processor->getConfiguration();

        if ($solarium_query->getType() === 'select' && !empty($settings['ai_provider'])) {
          $fields = $query->getIndex()->getFields();
          $vector_fields = [];

          foreach ($fields as $key => $field) {
            if ($field->getType() == 'solr_densevector') {
              $vector_fields[$key] = $field;
            }
          }

          if (count($vector_fields) && $keys = $query->getOriginalKeys()) {
            $provider = $this->aiProviderManager->createInstance($settings['ai_provider']);
            if (is_array($keys)) {
              $keys = implode(' ', $keys);
            }
            $embedding_input = new EmbeddingsInput($keys);
            $embedding_output = $provider->embeddings($embedding_input, $settings['ai_model_id']);
            $vectors[] = $embedding_output->getNormalized();
            $solr_fields = [];

            /** @var \Drupal\search_api\Index\Field $field */
            foreach ($vector_fields as $field) {
              $solr_fields[] = 'knns_' . $field->getFieldIdentifier();
            }

            // @todo should this re-rank? how can we do that correctly?
            $solarium_query->setQuery('{!knn f=' . implode(',', $solr_fields) . ' topK=' . $settings['top_k'] . '}[' . implode(', ', $vectors[0]) . ']');
          }
        }
      }
      catch (SearchApiException $e) {
        $this->loggerFactory->get('search_api_solr_dense_vector')->info('The solr_densevector processor is not enabled.');
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function postExtractResults(PostExtractResultsEvent $event): void {
    $query = $event->getSearchApiQuery();

    if ($query->getIndex()->isValidProcessor('solr_densevector')) {
      try {
        $processor = $query->getIndex()->getProcessor('solr_densevector');
        $settings = $processor->getConfiguration();

        if (!empty($settings['content_field'])) {
          $results = $event->getSearchApiQuery()->getResults();

          foreach ($results as $result) {
            $field = $result->getField($settings['content_field']);
            $field_values = [];

            $values = $field->getValues();

            foreach ($values as $value) {
              $text = ($value instanceof TextValue) ? $value->getText() : $value;
              $text = strip_tags($text);
              $field_values[] = $text;
            }

            $result->setExtraData('content', implode(' ', $field_values));
          }
        }
      }
      catch (SearchApiException $e) {
        $this->loggerFactory->get('search_api_solr_dense_vector')->info('The solr_densevector processor is not enabled.');
      }
    }
  }

}
