<?php

declare(strict_types=1);

namespace Drupal\search_api_solr_dense_vector\EventSubscriber;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Embeddings\EmbeddingsInput;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\search_api\Plugin\search_api\data_type\value\TextValue;
use Drupal\search_api\SearchApiException;
use Drupal\search_api_solr\Event\PostConvertedQueryEvent;
use Drupal\search_api_solr\Event\PostExtractResultsEvent;
use Drupal\search_api_solr\Event\SearchApiSolrEvents;
use Drupal\search_api_solr_dense_vector\DenseVectorRankerPluginManager;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Alters the query to re-rank search with vectors.
 */
class SearchApiSolrDenseVectorQuerySubscriber implements EventSubscriberInterface {

  /**
   * Event subscriber constructor.
   */
  public function __construct(
    protected AiProviderPluginManager $aiProviderManager,
    protected LoggerChannelFactoryInterface $loggerFactory,
    protected DenseVectorRankerPluginManager $denseVectorRankerPluginManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      SearchApiSolrEvents::POST_CONVERT_QUERY => 'postConvertQuery',
      SearchApiSolrEvents::POST_EXTRACT_RESULTS => 'postExtractResults',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function postConvertQuery(PostConvertedQueryEvent $event): void {
    $query = $event->getSearchApiQuery();
    $solarium_query = $event->getSolariumQuery();
    $index = $query->getIndex();

    if ($query->getIndex()->isValidProcessor('solr_densevector')) {
      try {
        $processor = $query->getIndex()->getProcessor('solr_densevector');
        $settings = $processor->getConfiguration();

        if ($solarium_query->getType() === 'select' && !empty($settings['ai_provider'])) {
          $fields = $query->getIndex()->getFields();
          $vector_fields = [];

          foreach ($fields as $key => $field) {
            if ($field->getType() == 'solr_densevector') {
              $vector_fields[$key] = $field;
              break;
            }
          }

          if (count($vector_fields) && $keys = $query->getOriginalKeys()) {
            $provider = $this->aiProviderManager->createInstance($settings['ai_provider']);

            if (is_array($keys)) {
              $keys = implode(' ', $keys);
            }

            $embedding_input = new EmbeddingsInput($keys);
            $embedding_output = $provider->embeddings($embedding_input, $settings['ai_model_id']);
            $vectors[] = $embedding_output->getNormalized();
            $vector_field = reset($vector_fields);
            $solr_field = 'knn_' . $vector_field->getFieldIdentifier();
            $index_settings = $index->getThirdPartySettings('search_api_solr');

            // The highlighter component is not compatible.
            $solarium_query->removeParam('hl');

            if (isset($index_settings['dense_vector']['enable_vector_rerank']) && (bool) $index_settings['dense_vector']['enable_vector_rerank']) {
              try {
                $plugin = $this->denseVectorRankerPluginManager->createInstance($index_settings['dense_vector']['dense_vector_rank_plugin']);
                $plugin->apply($solarium_query, $solr_field, (int) $index_settings['dense_vector']['top_k'], $vectors);
              }
              catch (\Exception $e) {
                $this->loggerFactory->get('search_api_solr_dense_vector')->error(
                  'There was an issue loading or executing the dense vector reranker @plugin: @message',
                  [
                    '@message' => $e->getMessage(),
                    '@plugin' => $index_settings['dense_vector']['dense_vector_rank_plugin'],
                  ]
                );
              }
            }
          }
        }
      }
      catch (SearchApiException $e) {
        $this->loggerFactory->get('search_api_solr_dense_vector')->info('The solr_densevector processor is not enabled.');
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function postExtractResults(PostExtractResultsEvent $event): void {
    $query = $event->getSearchApiQuery();

    if ($query->getIndex()->isValidProcessor('solr_densevector')) {
      try {
        $processor = $query->getIndex()->getProcessor('solr_densevector');
        $settings = $processor->getConfiguration();

        if (!empty($settings['content_field'])) {
          $results = $event->getSearchApiQuery()->getResults();

          foreach ($results as $result) {
            $field = $result->getField($settings['content_field']);
            $field_values = [];

            $values = $field->getValues();

            foreach ($values as $value) {
              $text = ($value instanceof TextValue) ? $value->getText() : $value;
              $text = strip_tags($text);
              $field_values[] = $text;
            }

            $result->setExtraData('content', implode(' ', $field_values));
          }
        }
      }
      catch (SearchApiException $e) {
        $this->loggerFactory->get('search_api_solr_dense_vector')->info('The solr_densevector processor is not enabled.');
      }
    }
  }

}
