<?php

declare(strict_types=1);

namespace Drupal\search_api_solr_dense_vector;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Solarium\QueryType\Select\Query\Query as SolariumQueryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for DenseVectorRanker plugins.
 */
abstract class DenseVectorRankerPluginBase extends PluginBase implements DenseVectorRankerInterface {
  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->configuration = NestedArray::mergeDeep($this->defaultConfiguration(), $this->configuration);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'top_k' => 10,
      'minimum_return' => 0.5,
      'minimum_traverse' => 0,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration(): array {
    return $this->configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration): void {
    $this->configuration = $configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['top_k'] = [
      '#type' => 'number',
      '#title' => $this->t('topK value'),
      '#step' => 1,
      '#min' => 1,
      '#description' => $this->t('How many k-nearest results to return.'),
      '#default_value' => $this->configuration['top_k'],
    ];

    $form['minimum_return'] = [
      '#title' => $this->t('Minimum return'),
      '#type' => 'number',
      '#step' => '0.001',
      '#min' => 0.001,
      '#max' => 1,
      '#description' => $this->t('A value between 0.1 and 1.0 that determines the minimum relevance threshold for a result to be returned. See the <a href=":uri">Solr documentation</a> for more information.', [':uri' => 'https://solr.apache.org/guide/solr/latest/query-guide/dense-vector-search.html#vectorsimilarity-query-parser']),
      '#default_value' => $this->configuration['minimum_return'],
    ];

    $form['minimum_traverse'] = [
      '#title' => $this->t('Minimum traverse'),
      '#type' => 'number',
      '#step' => 1,
      '#min' => 0,
      '#description' => $this->t('Minimum similarity of nodes in the graph to continue traversal of their neighbors. Enter 0 for unlimited. See the <a href=":uri">Solr documentation</a> for more information.', [':uri' => 'https://solr.apache.org/guide/solr/latest/query-guide/dense-vector-search.html#vectorsimilarity-query-parser']),
      '#default_value' => $this->configuration['minimum_traverse'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {}

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();
    $this->setConfiguration($values);
  }

  /**
   * {@inheritdoc}
   */
  abstract public function apply(SolariumQueryInterface $query, string $solr_field, array $vectors): SolariumQueryInterface;

}
