<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Autocomplete;

use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_autocomplete\SearchInterface;
use Drupal\search_api_autocomplete\Suggestion\SuggestionFactory;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Search\QueryBuilderInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for handling autocomplete suggestions.
 */
final class AutocompleteHandler implements AutocompleteHandlerInterface {

  /**
   * Constructs an AutocompleteHandler instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The connection manager.
   * @param \Drupal\search_api_sqlite\Database\SchemaManagerInterface $schemaManager
   *   The schema manager.
   * @param \Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface $fts5QueryRunner
   *   The FTS5 query runner.
   * @param \Drupal\search_api_sqlite\Search\QueryBuilderInterface $queryBuilder
   *   The query builder.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    private readonly ConnectionManagerInterface $connectionManager,
    private readonly SchemaManagerInterface $schemaManager,
    private readonly Fts5QueryRunnerInterface $fts5QueryRunner,
    private readonly QueryBuilderInterface $queryBuilder,
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getSuggestions(
    QueryInterface $query,
    SearchInterface $search,
    string $incomplete_key,
    string $user_input,
    array $fulltext_fields,
  ): array {
    $suggestions = [];
    $index = $query->getIndex();
    $index_id = (string) $index->id();

    // Check if tables exist.
    if (!$this->schemaManager->tablesExist($index_id)) {
      return [];
    }

    try {
      $fts_table = $this->schemaManager->getFtsTableName($index_id);

      if ($fulltext_fields === [] || $incomplete_key === '') {
        return [];
      }

      // Build a prefix match query for the incomplete key.
      $escaped_key = $this->queryBuilder->escapeTerm($incomplete_key);
      $prefix_query = '"' . $escaped_key . '"*';

      // Search for matching terms.
      $fts_results = $this->fts5QueryRunner->search(
        $index_id,
        $fts_table,
        $prefix_query,
        [
          'limit' => $search->getOption('limit') ?? 10,
          'offset' => 0,
          'order_by_rank' => TRUE,
        ]
      );

      if ($fts_results === []) {
        return [];
      }

      // Get the suggestion factory - search_api_autocomplete must be installed.
      if (!class_exists(SuggestionFactory::class)) {
        return [];
      }

      $factory = new SuggestionFactory($user_input);

      // Build the base suggestion text from user input.
      $base_text = trim(preg_replace('/\s+/', ' ', $user_input) ?? '');
      $words = explode(' ', $base_text);
      array_pop($words);
      $prefix = $words === [] ? '' : implode(' ', $words) . ' ';

      // Create suggestions from results.
      $seen = [];
      foreach ($fts_results as $row) {
        $item_id = $row['item_id'];

        // Skip duplicates.
        if (isset($seen[$item_id])) {
          continue;
        }

        $seen[$item_id] = TRUE;

        // Try to extract a completion from the indexed content.
        $completion = $this->extractCompletion(
          $index_id,
          $fts_table,
          $item_id,
          $incomplete_key,
          array_values($fulltext_fields)
        );

        if ($completion === NULL) {
          continue;
        }

        $suggested_text = $prefix . $completion;
        $suggestions[] = $factory->createFromSuggestedKeys($suggested_text);
      }
    }
    catch (\Exception $exception) {
      $this->logger->error('Autocomplete failed for index @index: @error', [
        '@index' => $index_id,
        '@error' => $exception->getMessage(),
      ]);
    }

    return $suggestions;
  }

  /**
   * Extracts a completion suggestion from indexed content.
   *
   * @param string $index_id
   *   The index ID.
   * @param string $fts_table
   *   The FTS table name.
   * @param string $item_id
   *   The item ID.
   * @param string $incomplete_key
   *   The incomplete search key.
   * @param array<int, string> $columns
   *   The FTS columns to search.
   *
   * @return string|null
   *   A completion suggestion or NULL if none found.
   */
  private function extractCompletion(
    string $index_id,
    string $fts_table,
    string $item_id,
    string $incomplete_key,
    array $columns,
  ): ?string {
    $pdo = $this->connectionManager->getPdo($index_id);

    // Get the first fulltext column's content.
    $column = $columns[0] ?? 'title';

    $sql = sprintf(
      'SELECT %s FROM %s WHERE item_id = :item_id LIMIT 1',
      $column,
      $fts_table
    );

    try {
      $stmt = $pdo->prepare($sql);
      $stmt->bindValue(':item_id', $item_id, \PDO::PARAM_STR);
      $stmt->execute();
      $content = $stmt->fetchColumn();

      if ($content === FALSE || $content === NULL) {
        return NULL;
      }

      // Find words starting with the incomplete key.
      $content = (string) $content;
      $pattern = '/\b(' . preg_quote($incomplete_key, '/') . '\w*)\b/iu';
      if (preg_match($pattern, $content, $matches)) {
        return $matches[1];
      }
    }
    catch (\Exception) {
      // Silently fail for individual items.
    }

    return $incomplete_key;
  }

}
