<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Database;

use Drupal\Core\Database\Connection;

/**
 * Defines the interface for SQLite connection management.
 */
interface ConnectionManagerInterface {

  /**
   * Gets or creates a database connection for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return \Drupal\Core\Database\Connection
   *   The database connection.
   *
   * @throws \RuntimeException
   *   If the connection cannot be established.
   */
  public function getConnection(string $index_id): Connection;

  /**
   * Gets the raw PDO connection for FTS5 operations.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return \PDO
   *   The PDO connection.
   */
  public function getPdo(string $index_id): \PDO;

  /**
   * Closes and removes a database connection.
   *
   * @param string $index_id
   *   The Search API index ID.
   */
  public function closeConnection(string $index_id): void;

  /**
   * Gets the database file path for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return string
   *   The absolute path to the database file.
   */
  public function getDatabasePath(string $index_id): string;

  /**
   * Checks if a database exists for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return bool
   *   TRUE if the database exists, FALSE otherwise.
   */
  public function databaseExists(string $index_id): bool;

  /**
   * Deletes the database file for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return bool
   *   TRUE if the database was deleted, FALSE otherwise.
   */
  public function deleteDatabase(string $index_id): bool;

}
