<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Enum;

/**
 * FTS5 matching modes.
 *
 * Defines how search terms are matched in full-text queries.
 */
enum MatchingMode: string {

  // Match all words - all terms must appear in results (AND).
  case Words = 'words';

  // Phrase matching - terms must appear consecutively in exact order.
  case Phrase = 'phrase';

  // Prefix matching - matches word beginnings, ideal for autocomplete.
  case Prefix = 'prefix';

  /**
   * Gets a human-readable label for the matching mode.
   *
   * @return string
   *   The matching mode label.
   */
  public function label(): string {
    return match ($this) {
      self::Words => 'Match all words (AND)',
      self::Phrase => 'Phrase matching',
      self::Prefix => 'Prefix matching (autocomplete)',
    };
  }

  /**
   * Gets a human-readable label for form options.
   *
   * @return string
   *   The form option label.
   */
  public function formLabel(): string {
    return match ($this) {
      self::Words => 'Match all words (AND) - most precise',
      self::Prefix => 'Prefix matching - best for autocomplete',
      self::Phrase => 'Phrase matching - exact word sequence',
    };
  }

  /**
   * Gets all matching modes as form options.
   *
   * @return array<string, string>
   *   Array of matching mode values to labels.
   */
  public static function formOptions(): array {
    return [
      self::Words->value => self::Words->formLabel(),
      self::Prefix->value => self::Prefix->formLabel(),
      self::Phrase->value => self::Phrase->formLabel(),
    ];
  }

  /**
   * Gets all matching modes as display labels.
   *
   * @return array<string, string>
   *   Array of matching mode values to labels.
   */
  public static function labels(): array {
    $labels = [];
    foreach (self::cases() as $case) {
      $labels[$case->value] = $case->label();
    }

    return $labels;
  }

}
