<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Enum;

/**
 * FTS5 tokenizer types.
 *
 * Defines the available tokenizer options for SQLite FTS5 full-text search.
 */
enum Tokenizer: string {

  // Unicode61 tokenizer - handles Unicode text with diacritics removal.
  case Unicode61 = 'unicode61';

  // Porter stemmer tokenizer - reduces words to their root form.
  case Porter = 'porter';

  // ASCII tokenizer - simple tokenizer for basic ASCII text.
  case Ascii = 'ascii';

  // Trigram tokenizer - enables substring matching.
  case Trigram = 'trigram';

  /**
   * Gets a human-readable label for the tokenizer.
   *
   * @return string
   *   The tokenizer label.
   */
  public function label(): string {
    return match ($this) {
      self::Unicode61 => 'Unicode61 (recommended)',
      self::Porter => 'Porter Stemmer',
      self::Ascii => 'ASCII',
      self::Trigram => 'Trigram (substring matching)',
    };
  }

  /**
   * Gets a human-readable label for form options.
   *
   * @return string
   *   The form option label.
   */
  public function formLabel(): string {
    return match ($this) {
      self::Unicode61 => 'Unicode61 (recommended for multilingual)',
      self::Porter => 'Porter Stemmer (English with word stemming)',
      self::Ascii => 'ASCII (basic English only)',
      self::Trigram => 'Trigram (substring matching for codes/SKUs)',
    };
  }

  /**
   * Gets all tokenizers as form options.
   *
   * @return array<string, string>
   *   Array of tokenizer values to labels.
   */
  public static function formOptions(): array {
    $options = [];
    foreach (self::cases() as $case) {
      $options[$case->value] = $case->formLabel();
    }

    return $options;
  }

  /**
   * Gets all tokenizers as display labels.
   *
   * @return array<string, string>
   *   Array of tokenizer values to labels.
   */
  public static function labels(): array {
    $labels = [];
    foreach (self::cases() as $case) {
      $labels[$case->value] = $case->label();
    }

    return $labels;
  }

  /**
   * Checks if this tokenizer supports case sensitivity option.
   *
   * @return bool
   *   TRUE if the tokenizer supports case sensitivity.
   */
  public function supportsCaseSensitivity(): bool {
    return $this === self::Trigram;
  }

}
