<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Database\Connection;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Query\ConditionGroupInterface;
use Drupal\search_api\Query\ConditionInterface;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Index\FieldTypeMapperInterface;
use Psr\Log\LoggerInterface;

/**
 * Handles search query conditions and filters.
 *
 * Processes Search API query conditions (including nested condition groups)
 * and filters item IDs accordingly. Supports AND/OR conjunctions and various
 * comparison operators.
 */
final class ConditionHandler implements ConditionHandlerInterface {

  /**
   * Constructs a ConditionHandler instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The database connection manager.
   * @param \Drupal\search_api_sqlite\Database\SchemaManagerInterface $schemaManager
   *   The schema manager.
   * @param \Drupal\search_api_sqlite\Index\FieldTypeMapperInterface $fieldTypeMapper
   *   The field type mapper.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    private readonly ConnectionManagerInterface $connectionManager,
    private readonly SchemaManagerInterface $schemaManager,
    private readonly FieldTypeMapperInterface $fieldTypeMapper,
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function applyConditions(QueryInterface $query, string $index_id, array $item_ids): array {
    $conditions = $query->getConditionGroup();

    if (empty($conditions->getConditions())) {
      return $item_ids;
    }

    $connection = $this->connectionManager->getConnection($index_id);
    $index = $query->getIndex();

    return $this->applyConditionGroup($conditions, $connection, $index, $index_id, $item_ids);
  }

  /**
   * {@inheritdoc}
   */
  public function getValueColumnForField(IndexInterface $index, string $field_name): string {
    $field = $index->getField($field_name);
    if (!$field) {
      // Default to string if field not found.
      return 'value_string';
    }

    $type = $field->getType();
    $storage_type = $this->fieldTypeMapper->getStorageType($type);

    return $this->fieldTypeMapper->getFieldDataColumn($storage_type);
  }

  /**
   * Recursively applies a condition group to filter item IDs.
   *
   * @param \Drupal\search_api\Query\ConditionGroupInterface $condition_group
   *   The condition group to apply.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param \Drupal\search_api\IndexInterface $index
   *   The search index.
   * @param string $index_id
   *   The index ID.
   * @param array<string> $item_ids
   *   Current item IDs.
   *
   * @return array<string>
   *   Filtered item IDs.
   */
  private function applyConditionGroup(
    ConditionGroupInterface $condition_group,
    Connection $connection,
    IndexInterface $index,
    string $index_id,
    array $item_ids,
  ): array {
    $conjunction = $condition_group->getConjunction();
    $field_data_table = $this->schemaManager->getFieldDataTableName($index_id);

    // For OR conjunction, we collect all matching IDs and union them.
    $or_results = [];

    foreach ($condition_group->getConditions() as $condition) {
      // If we have no items left and using AND, no need to continue.
      if ($item_ids === [] && $conjunction === 'AND') {
        return [];
      }

      // Handle nested condition groups recursively.
      if ($condition instanceof ConditionGroupInterface) {
        $nested_results = $this->applyConditionGroup(
          $condition,
          $connection,
          $index,
          $index_id,
          $item_ids
        );

        if ($conjunction === 'OR') {
          $or_results = array_merge($or_results, $nested_results);
        }
        else {
          $item_ids = $nested_results;
        }
        continue;
      }

      // At this point, condition must be a ConditionInterface.
      // @phpstan-ignore instanceof.alwaysTrue
      if (!$condition instanceof ConditionInterface) {
        continue;
      }

      $field = $condition->getField();
      $value = $condition->getValue();
      $operator = $condition->getOperator();

      // Determine the correct column based on field type.
      $value_column = $this->getValueColumnForField($index, $field);

      $this->logger->debug('Applying condition: field=@field, value=@value, operator=@op, column=@column', [
        '@field' => $field,
        '@value' => is_array($value) ? implode(',', $value) : (string) $value,
        '@op' => $operator,
        '@column' => $value_column,
      ]);

      // Build a query to filter items.
      $subquery = $connection->select($field_data_table, 'fd')
        ->fields('fd', ['item_id'])
        ->condition('fd.field_name', $field)
        ->condition('fd.item_id', $item_ids, 'IN');

      // Apply the condition based on operator.
      $this->applyOperatorCondition($subquery, $value_column, $value, $operator);

      $result = $subquery->execute();
      $matching_ids = $result !== NULL ? $result->fetchCol() : [];

      $this->logger->debug('Condition matched @count items out of @total', [
        '@count' => count($matching_ids),
        '@total' => count($item_ids),
      ]);

      if ($conjunction === 'OR') {
        $or_results = array_merge($or_results, $matching_ids);
      }
      else {
        // AND conjunction: intersect with current results.
        $item_ids = array_intersect($item_ids, $matching_ids);
      }
    }

    // For OR conjunction, return unique union of all matches.
    if ($conjunction === 'OR' && $or_results !== []) {
      return array_values(array_unique($or_results));
    }

    return array_values($item_ids);
  }

  /**
   * Applies an operator condition to a query.
   *
   * @param \Drupal\Core\Database\Query\SelectInterface $query
   *   The query to modify.
   * @param string $column
   *   The column name.
   * @param mixed $value
   *   The value to compare against.
   * @param string $operator
   *   The comparison operator.
   */
  private function applyOperatorCondition(
    SelectInterface $query,
    string $column,
    mixed $value,
    string $operator,
  ): void {
    switch ($operator) {
      case '=':
      case '<>':
      case '<':
      case '<=':
      case '>':
      case '>=':
        $query->condition('fd.' . $column, $value, $operator);
        break;

      case 'IN':
        $query->condition('fd.' . $column, (array) $value, 'IN');
        break;

      case 'NOT IN':
        $query->condition('fd.' . $column, (array) $value, 'NOT IN');
        break;

      case 'BETWEEN':
        if (is_array($value) && count($value) >= 2) {
          $query->condition('fd.' . $column, $value[0], '>=');
          $query->condition('fd.' . $column, $value[1], '<=');
        }
        break;
    }
  }

}
