<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_sqlite\Enum\MatchingMode;

/**
 * Builds FTS5 MATCH queries from Search API queries.
 */
final class QueryBuilder implements QueryBuilderInterface {

  /**
   * Characters that need escaping in FTS5.
   *
   * @var array<string>
   */
  private const array SPECIAL_CHARS = ['"', '*', '-', '+', 'AND', 'OR', 'NOT', 'NEAR', '(', ')', '.', ':', '^', '$'];

  /**
   * {@inheritdoc}
   */
  public function buildMatchQuery(QueryInterface $query, array $fulltext_fields, MatchingMode $matching_mode = MatchingMode::Words): ?string {
    $keys = $query->getKeys();

    if (empty($keys)) {
      return NULL;
    }

    // Get the fulltext fields to search in.
    $search_fields = $query->getFulltextFields();
    if (empty($search_fields)) {
      $search_fields = array_keys($fulltext_fields);
    }

    // Map to sanitized column names.
    $columns = [];
    foreach ($search_fields as $field_id) {
      if (isset($fulltext_fields[$field_id])) {
        $columns[] = $fulltext_fields[$field_id];
      }
    }

    if ($columns === []) {
      return NULL;
    }

    // Parse the search keys.
    $parsed = $this->parseKeys($keys);

    if (empty($parsed['terms']) && empty($parsed['phrases'])) {
      return NULL;
    }

    // Build the FTS5 query.
    return $this->buildFts5Query($parsed, $columns, $matching_mode);
  }

  /**
   * {@inheritdoc}
   */
  public function escapeTerm(string $term): string {
    // Remove or escape special characters.
    $term = trim($term);

    // Double quotes need to be escaped.
    $term = str_replace('"', '""', $term);

    // If term contains special chars, quote it.
    $needs_quoting = FALSE;
    foreach (self::SPECIAL_CHARS as $char) {
      if (stripos($term, $char) !== FALSE) {
        $needs_quoting = TRUE;
        break;
      }
    }

    if ($needs_quoting || preg_match('/\s/', $term)) {
      return '"' . $term . '"';
    }

    return $term;
  }

  /**
   * {@inheritdoc}
   */
  public function parseSearchInput(string $input): array {
    $result = [
      'terms' => [],
      'phrases' => [],
    ];

    $input = trim($input);
    if ($input === '' || $input === '0') {
      return $result;
    }

    // Extract quoted phrases first.
    if (preg_match_all('/"([^"]+)"/', $input, $matches)) {
      $result['phrases'] = $matches[1];
      // Remove matched phrases from input.
      $input = preg_replace('/"[^"]+"/', '', $input) ?? '';
    }

    // Split remaining text into terms.
    $input = trim($input);
    if ($input !== '') {
      $terms = preg_split('/\s+/', $input, -1, PREG_SPLIT_NO_EMPTY);
      if ($terms !== FALSE) {
        $result['terms'] = array_values(array_filter($terms, fn(string $term): bool => strlen($term) >= 2));
      }
    }

    return $result;
  }

  /**
   * Parses Search API keys into terms and phrases.
   *
   * @param mixed $keys
   *   The search keys from the query.
   *
   * @return array{terms: array<string>, phrases: array<string>, conjunction: string}
   *   Parsed search terms.
   */
  private function parseKeys(mixed $keys): array {
    $result = [
      'terms' => [],
      'phrases' => [],
      'conjunction' => 'AND',
    ];

    if (is_string($keys)) {
      return $this->parseSearchInput($keys) + ['conjunction' => 'AND'];
    }

    if (!is_array($keys)) {
      return $result;
    }

    // Handle Search API's structured keys format.
    $conjunction = $keys['#conjunction'] ?? 'AND';
    // Note: #negation is available but not yet implemented.
    $result['conjunction'] = strtoupper((string) $conjunction);

    foreach ($keys as $key => $value) {
      if (str_starts_with((string) $key, '#')) {
        continue;
      }

      if (is_array($value)) {
        // Nested group - recursively parse.
        $nested = $this->parseKeys($value);
        $result['terms'] = array_merge($result['terms'], $nested['terms']);
        $result['phrases'] = array_merge($result['phrases'], $nested['phrases']);
      }
      else {
        // Single term.
        $term = trim((string) $value);
        if (strlen($term) >= 2) {
          $result['terms'][] = $term;
        }
      }
    }

    return $result;
  }

  /**
   * Builds the final FTS5 query string.
   *
   * @param array{terms: array<string>, phrases: array<string>, conjunction: string} $parsed
   *   Parsed search terms.
   * @param array<string> $columns
   *   Column names to search in.
   * @param \Drupal\search_api_sqlite\Enum\MatchingMode $matching_mode
   *   The matching mode.
   *
   * @return string
   *   The FTS5 MATCH query.
   */
  private function buildFts5Query(array $parsed, array $columns, MatchingMode $matching_mode): string {
    $parts = [];
    $conjunction = $parsed['conjunction'] === 'OR' ? ' OR ' : ' AND ';

    // Build column filter if not searching all columns.
    $column_filter = $columns !== [] ? '{' . implode(' ', $columns) . '}: ' : '';

    // Process phrases (exact matches).
    foreach ($parsed['phrases'] as $phrase) {
      $escaped = $this->escapeTerm($phrase);
      $parts[] = $column_filter . $escaped;
    }

    // Process individual terms.
    foreach ($parsed['terms'] as $term) {
      $escaped = $this->escapeTerm($term);

      switch ($matching_mode) {
        case MatchingMode::Prefix:
          // Add prefix operator.
          if (!str_ends_with($escaped, '*')) {
            // Remove quotes for prefix matching.
            $escaped = trim($escaped, '"');
            $escaped .= '*';
          }

          break;

        case MatchingMode::Phrase:
          // Terms are already properly escaped.
          break;

        case MatchingMode::Words:
        default:
          // Standard word matching.
          break;
      }

      $parts[] = $column_filter . $escaped;
    }

    if ($parts === []) {
      return '';
    }

    // Combine with conjunction.
    $query = implode($conjunction, $parts);

    // Wrap in parentheses if multiple parts with OR.
    if (count($parts) > 1 && $conjunction === ' OR ') {
      return '(' . $query . ')';
    }

    return $query;
  }

}
