<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_sqlite\Enum\MatchingMode;

/**
 * Defines the interface for building FTS5 MATCH queries.
 */
interface QueryBuilderInterface {

  /**
   * Builds an FTS5 MATCH query from a Search API query.
   *
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The Search API query.
   * @param array<string, string> $fulltext_fields
   *   Map of field IDs to sanitized column names.
   * @param \Drupal\search_api_sqlite\Enum\MatchingMode $matching_mode
   *   The matching mode.
   *
   * @return string|null
   *   The FTS5 MATCH query string, or NULL if no fulltext search.
   */
  public function buildMatchQuery(QueryInterface $query, array $fulltext_fields, MatchingMode $matching_mode = MatchingMode::Words): ?string;

  /**
   * Escapes a search term for FTS5.
   *
   * @param string $term
   *   The search term.
   *
   * @return string
   *   The escaped term.
   */
  public function escapeTerm(string $term): string;

  /**
   * Parses user input into search terms.
   *
   * @param string $input
   *   The user's search input.
   *
   * @return array{terms: array<string>, phrases: array<string>}
   *   Array with 'terms' (individual words) and 'phrases' (quoted strings).
   */
  public function parseSearchInput(string $input): array;

}
