<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Status;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\search_api\ServerInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Enum\MatchingMode;
use Drupal\search_api_sqlite\Enum\Tokenizer;

/**
 * Service for reporting status information about the SQLite FTS5 backend.
 */
final class StatusReporter implements StatusReporterInterface {

  use StringTranslationTrait;

  /**
   * Constructs a StatusReporter instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The connection manager.
   * @param \Drupal\search_api_sqlite\Database\SchemaManagerInterface $schemaManager
   *   The schema manager.
   * @param \Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface $fts5QueryRunner
   *   The FTS5 query runner.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system service.
   */
  public function __construct(
    private readonly ConnectionManagerInterface $connectionManager,
    private readonly SchemaManagerInterface $schemaManager,
    private readonly Fts5QueryRunnerInterface $fts5QueryRunner,
    private readonly FileSystemInterface $fileSystem,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getViewSettings(array $configuration, ServerInterface $server): array {
    $info = [];

    // Database path.
    $info[] = [
      'label' => $this->t('Database path'),
      'info' => $configuration['database_path'] ?? $this->t('Default'),
    ];

    // Tokenizer.
    $tokenizers = Tokenizer::labels();
    $tokenizer_info = $tokenizers[$configuration['tokenizer']] ?? $configuration['tokenizer'];
    if ($configuration['tokenizer'] === Tokenizer::Trigram->value) {
      $case_sensitive = $configuration['trigram_case_sensitive'] ?? FALSE;
      $tokenizer_info .= ' - ' . ($case_sensitive ? $this->t('case-sensitive') : $this->t('case-insensitive'));
    }

    $info[] = [
      'label' => $this->t('Tokenizer'),
      'info' => $tokenizer_info,
    ];

    // Minimum word length.
    if ($configuration['min_chars'] > 1) {
      $info[] = [
        'label' => $this->t('Minimum word length'),
        'info' => $configuration['min_chars'],
      ];
    }

    // Matching mode.
    $matching_modes = MatchingMode::labels();
    $info[] = [
      'label' => $this->t('Matching mode'),
      'info' => $matching_modes[$configuration['matching']] ?? $configuration['matching'],
    ];

    // Highlighting settings.
    $highlighting_enabled = $configuration['highlighting']['enabled'] ?? FALSE;
    $info[] = [
      'label' => $this->t('Highlighting'),
      'info' => $highlighting_enabled ? $this->t('Enabled') : $this->t('Disabled'),
    ];

    if ($highlighting_enabled) {
      $info[] = [
        'label' => $this->t('Highlight tags'),
        'info' => $configuration['highlighting']['prefix'] . '...' . $configuration['highlighting']['suffix'],
      ];
      $info[] = [
        'label' => $this->t('Excerpt length'),
        'info' => $this->t('@length tokens', ['@length' => $configuration['highlighting']['excerpt_length']]),
      ];
    }

    // Optimization settings.
    $auto_optimize = $configuration['optimization']['auto_optimize'] ?? FALSE;
    $info[] = [
      'label' => $this->t('Auto-optimization'),
      'info' => $auto_optimize
        ? $this->t('Enabled (threshold: @count changes)', ['@count' => $configuration['optimization']['optimize_threshold']])
        : $this->t('Disabled'),
    ];

    // Verbose logging.
    $verbose_logging = $configuration['verbose_logging'] ?? FALSE;
    $info[] = [
      'label' => $this->t('Verbose logging'),
      'info' => $verbose_logging ? $this->t('Enabled') : $this->t('Disabled (errors/warnings only)'),
    ];

    // Add runtime status information.
    $info = array_merge($info, $this->getRuntimeStatusInfo($configuration, $server));

    return $info;
  }

  /**
   * {@inheritdoc}
   */
  public function getSqliteVersionInfo(): array {
    try {
      // Create a temporary in-memory connection to check SQLite version.
      $pdo = new \PDO('sqlite::memory:');
      $version = $pdo->query('SELECT sqlite_version()')->fetchColumn();

      // Check if FTS5 is available.
      $fts5_available = FALSE;
      try {
        $pdo->exec('CREATE VIRTUAL TABLE fts5_test USING fts5(content)');
        $pdo->exec('DROP TABLE fts5_test');
        $fts5_available = TRUE;
      }
      catch (\PDOException) {
        // FTS5 not available.
      }

      return [
        'version' => $version ?: 'Unknown',
        'fts5_available' => $fts5_available,
      ];
    }
    catch (\PDOException) {
      return [
        'version' => 'Unknown',
        'fts5_available' => FALSE,
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getIndexStatistics(string $index_id): array {
    $stats = [
      'database_exists' => FALSE,
      'tables_exist' => FALSE,
      'file_size' => 0,
      'document_count' => 0,
      'filename' => $index_id . '.sqlite',
    ];

    // Check if database exists.
    if (!$this->connectionManager->databaseExists($index_id)) {
      return $stats;
    }

    $stats['database_exists'] = TRUE;

    // Get file size.
    $db_path = $this->connectionManager->getDatabasePath($index_id);
    if (file_exists($db_path)) {
      $stats['file_size'] = (int) filesize($db_path);
      $stats['filename'] = basename($db_path);
    }

    // Check if tables exist.
    $stats['tables_exist'] = $this->schemaManager->tablesExist($index_id);

    // Get document count if tables exist.
    if ($stats['tables_exist']) {
      try {
        $fts_table = $this->schemaManager->getFtsTableName($index_id);
        $stats['document_count'] = $this->fts5QueryRunner->getDocumentCount($index_id, $fts_table);
      }
      catch (\Exception) {
        // Silently fail - count stays at 0.
      }
    }

    return $stats;
  }

  /**
   * Gets runtime status information for the backend.
   *
   * @param array<string, mixed> $configuration
   *   The backend configuration.
   * @param \Drupal\search_api\ServerInterface $server
   *   The search server.
   *
   * @return array<int, array<string, mixed>>
   *   An array of status information items.
   */
  private function getRuntimeStatusInfo(array $configuration, ServerInterface $server): array {
    $info = [];

    // SQLite version.
    $sqlite_info = $this->getSqliteVersionInfo();
    $info[] = [
      'label' => $this->t('SQLite version'),
      'info' => $sqlite_info['version'],
      'status' => $sqlite_info['fts5_available'] ? 'ok' : 'error',
    ];

    if (!$sqlite_info['fts5_available']) {
      $info[] = [
        'label' => $this->t('FTS5 extension'),
        'info' => $this->t('Not available - search will not work'),
        'status' => 'error',
      ];
    }

    // Database directory.
    $db_path = $configuration['database_path'] ?? 'private://search_api_sqlite';
    $resolved_path = $this->resolveDatabaseDirectory($db_path);
    $dir_status = $this->checkDirectoryStatus($resolved_path);
    $info[] = [
      'label' => $this->t('Storage directory'),
      'info' => $resolved_path ?: $db_path,
      'status' => $dir_status,
    ];

    // Per-index statistics.
    $indexes = $server->getIndexes();
    $total_size = 0;
    $index_count = 0;

    foreach ($indexes as $index) {
      $index_id = (string) $index->id();
      $stats = $this->getIndexStatistics($index_id);

      $index_count++;
      $total_size += $stats['file_size'];

      // Index header.
      $index_label = $this->t('Index: @name', ['@name' => $index->label()]);

      // Database file status.
      if ($stats['database_exists']) {
        $file_info = $this->t('@file (@size)', [
          '@file' => $stats['filename'],
          '@size' => $this->formatBytes($stats['file_size']),
        ]);
        $info[] = [
          'label' => $index_label,
          'info' => $file_info,
          'status' => 'ok',
        ];
      }
      else {
        $info[] = [
          'label' => $index_label,
          'info' => $this->t('Database not found'),
          'status' => 'warning',
        ];
        continue;
      }

      // Tables and document count.
      if ($stats['tables_exist']) {
        $info[] = [
          'label' => $this->t('- Documents indexed'),
          'info' => $this->t('@count items', ['@count' => number_format($stats['document_count'])]),
          'status' => $stats['document_count'] > 0 ? 'ok' : 'warning',
        ];
      }
      else {
        $info[] = [
          'label' => $this->t('- Tables'),
          'info' => $this->t('Missing - reindex required'),
          'status' => 'warning',
        ];
      }
    }

    // Total storage summary.
    if ($index_count > 0) {
      $info[] = [
        'label' => $this->t('Total storage'),
        'info' => $this->t('@size across @count indexes', [
          '@size' => $this->formatBytes($total_size),
          '@count' => $index_count,
        ]),
        'status' => 'info',
      ];
    }

    return $info;
  }

  /**
   * Resolves the database directory path.
   *
   * @param string $path
   *   The configured path (may contain stream wrappers).
   *
   * @return string|null
   *   The resolved absolute path or NULL if unresolvable.
   */
  private function resolveDatabaseDirectory(string $path): ?string {
    // Handle stream wrappers.
    if (str_contains($path, '://')) {
      $real_path = $this->fileSystem->realpath($path);
      return $real_path ?: NULL;
    }

    return $path;
  }

  /**
   * Checks the status of a directory.
   *
   * @param string|null $path
   *   The directory path.
   *
   * @return string
   *   The status: 'ok', 'warning', or 'error'.
   */
  private function checkDirectoryStatus(?string $path): string {
    if ($path === NULL) {
      return 'warning';
    }

    if (!file_exists($path)) {
      return 'warning';
    }

    if (!is_writable($path)) {
      return 'error';
    }

    return 'ok';
  }

  /**
   * Formats bytes into a human-readable string.
   *
   * @param int $bytes
   *   The number of bytes.
   *
   * @return string
   *   Human-readable size string.
   */
  private function formatBytes(int $bytes): string {
    if ($bytes === 0) {
      return '0 B';
    }

    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $factor = (int) floor(log($bytes, 1024));
    $factor = min($factor, count($units) - 1);

    return sprintf('%.1f %s', $bytes / (1024 ** $factor), $units[$factor]);
  }

}
