<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Database;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Schema;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Item\FieldInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\SchemaManager;
use Drupal\search_api_sqlite\Index\FieldTypeMapperInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests the SchemaManager service.
 */
#[CoversClass(SchemaManager::class)]
#[Group('search_api_sqlite')]
final class SchemaManagerTest extends UnitTestCase {

  /**
   * The schema manager under test.
   */
  private SchemaManager $schemaManager;

  /**
   * The mocked connection manager.
   */
  private ConnectionManagerInterface&MockObject $connectionManager;

  /**
   * The mocked field type mapper.
   */
  private FieldTypeMapperInterface&MockObject $fieldTypeMapper;

  /**
   * The mocked PDO connection.
   */
  private \PDO&MockObject $pdo;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->connectionManager = $this->createMock(ConnectionManagerInterface::class);
    $this->fieldTypeMapper = $this->createMock(FieldTypeMapperInterface::class);
    $this->pdo = $this->createMock(\PDO::class);

    $this->connectionManager->method('getPdo')
      ->willReturn($this->pdo);

    $this->schemaManager = new SchemaManager(
      $this->connectionManager,
      $this->fieldTypeMapper
    );
  }

  /**
   * Tests getFtsTableName() generates correct table name.
   */
  #[DataProvider('tableNameProvider')]
  public function testGetFtsTableName(string $index_id, string $expected): void {
    $this->assertSame($expected, $this->schemaManager->getFtsTableName($index_id));
  }

  /**
   * Tests getFieldDataTableName() generates correct table name.
   */
  #[DataProvider('fieldDataTableNameProvider')]
  public function testGetFieldDataTableName(string $index_id, string $expected): void {
    $this->assertSame($expected, $this->schemaManager->getFieldDataTableName($index_id));
  }

  /**
   * Tests getItemsTableName() generates correct table name.
   */
  #[DataProvider('itemsTableNameProvider')]
  public function testGetItemsTableName(string $index_id, string $expected): void {
    $this->assertSame($expected, $this->schemaManager->getItemsTableName($index_id));
  }

  /**
   * Data provider for FTS table name tests.
   *
   * @return array<array{string, string}>
   *   Test cases.
   */
  public static function tableNameProvider(): array {
    return [
      'simple index name' => ['my_index', 'my_index_fts'],
      'index with numbers' => ['index_123', 'index_123_fts'],
      'default index' => ['default', 'default_fts'],
      'long index name' => ['very_long_index_name', 'very_long_index_name_fts'],
    ];
  }

  /**
   * Data provider for field data table name tests.
   *
   * @return array<array{string, string}>
   *   Test cases.
   */
  public static function fieldDataTableNameProvider(): array {
    return [
      'simple index name' => ['my_index', 'my_index_field_data'],
      'index with numbers' => ['index_123', 'index_123_field_data'],
    ];
  }

  /**
   * Data provider for items table name tests.
   *
   * @return array<array{string, string}>
   *   Test cases.
   */
  public static function itemsTableNameProvider(): array {
    return [
      'simple index name' => ['my_index', 'my_index_items'],
      'index with numbers' => ['index_123', 'index_123_items'],
    ];
  }

  /**
   * Tests tablesExist() returns TRUE when table exists.
   */
  public function testTablesExistReturnsTrue(): void {
    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('fetch')
      ->willReturn(['name' => 'my_index_fts']);

    $this->pdo->method('prepare')
      ->willReturn($stmt);

    $stmt->method('execute')
      ->willReturn(TRUE);

    $this->assertTrue($this->schemaManager->tablesExist('my_index'));
  }

  /**
   * Tests tablesExist() returns FALSE when table does not exist.
   */
  public function testTablesExistReturnsFalse(): void {
    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('fetch')
      ->willReturn(FALSE);

    $this->pdo->method('prepare')
      ->willReturn($stmt);

    $stmt->method('execute')
      ->willReturn(TRUE);

    $this->assertFalse($this->schemaManager->tablesExist('missing_index'));
  }

  /**
   * Tests createIndexTables() creates all required tables.
   */
  public function testCreateIndexTablesCreatesAllTables(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $titleField = $this->createMock(FieldInterface::class);
    $titleField->method('getType')->willReturn('text');
    $titleField->method('getBoost')->willReturn(1.0);

    $index->method('getFields')->willReturn(['title' => $titleField]);

    $this->fieldTypeMapper->method('isFulltextType')
      ->with('text')
      ->willReturn(TRUE);

    // Set up mock connection and schema.
    $schema = $this->createMock(Schema::class);
    $schema->method('tableExists')->willReturn(FALSE);
    $schema->expects($this->exactly(2))
      ->method('createTable');

    $connection = $this->createMock(Connection::class);
    $connection->method('schema')->willReturn($schema);

    $this->connectionManager->method('getConnection')
      ->willReturn($connection);

    // Expect FTS5 table creation.
    $this->pdo->expects($this->once())
      ->method('exec')
      ->with($this->stringContains('CREATE VIRTUAL TABLE'));

    $this->schemaManager->createIndexTables($index, ['tokenizer' => 'unicode61']);
  }

  /**
   * Tests dropIndexTables() drops all tables.
   */
  public function testDropIndexTablesDropsAllTables(): void {
    $this->pdo->expects($this->exactly(3))
      ->method('exec')
      ->with($this->logicalOr(
        $this->stringContains('DROP TABLE IF EXISTS test_index_fts'),
        $this->stringContains('DROP TABLE IF EXISTS test_index_field_data'),
        $this->stringContains('DROP TABLE IF EXISTS test_index_items')
      ));

    $this->schemaManager->dropIndexTables('test_index');
  }

  /**
   * Tests that table names are generated consistently.
   */
  public function testTableNamingConsistency(): void {
    $index_id = 'products';

    $fts = $this->schemaManager->getFtsTableName($index_id);
    $field_data = $this->schemaManager->getFieldDataTableName($index_id);
    $items = $this->schemaManager->getItemsTableName($index_id);

    // All should start with the index ID.
    $this->assertStringStartsWith($index_id . '_', $fts);
    $this->assertStringStartsWith($index_id . '_', $field_data);
    $this->assertStringStartsWith($index_id . '_', $items);

    // Each should have distinct suffix.
    $this->assertNotSame($fts, $field_data);
    $this->assertNotSame($fts, $items);
    $this->assertNotSame($field_data, $items);
  }

  /**
   * Tests updateIndexSchema() always rebuilds and returns true.
   */
  public function testUpdateIndexSchemaAlwaysRebuildsAndReturnsTrue(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $titleField = $this->createMock(FieldInterface::class);
    $titleField->method('getType')->willReturn('text');
    $titleField->method('getBoost')->willReturn(1.0);

    $index->method('getFields')->willReturn([
      'title' => $titleField,
    ]);

    $this->fieldTypeMapper->method('isFulltextType')
      ->willReturn(TRUE);

    // Expect FTS5 table rebuild (create temp, drop old, rename).
    $this->pdo->expects($this->exactly(3))
      ->method('exec');

    $needsReindex = $this->schemaManager->updateIndexSchema($index, ['tokenizer' => 'unicode61']);

    $this->assertTrue($needsReindex);
  }

  /**
   * Tests createIndexTables() handles non-fulltext fields correctly.
   */
  public function testCreateIndexTablesWithMixedFieldTypes(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    // Text field (fulltext).
    $titleField = $this->createMock(FieldInterface::class);
    $titleField->method('getType')->willReturn('text');
    $titleField->method('getBoost')->willReturn(1.0);

    // Integer field (not fulltext).
    $priceField = $this->createMock(FieldInterface::class);
    $priceField->method('getType')->willReturn('integer');
    $priceField->method('getBoost')->willReturn(1.0);

    $index->method('getFields')->willReturn([
      'title' => $titleField,
      'price' => $priceField,
    ]);

    $this->fieldTypeMapper->method('isFulltextType')
      ->willReturnCallback(fn($type): bool => $type === 'text');

    $schema = $this->createMock(Schema::class);
    $schema->method('tableExists')->willReturn(FALSE);
    $schema->expects($this->exactly(2))
      ->method('createTable');

    $connection = $this->createMock(Connection::class);
    $connection->method('schema')->willReturn($schema);

    $this->connectionManager->method('getConnection')
      ->willReturn($connection);

    // FTS5 table should only include title (fulltext field).
    $this->pdo->expects($this->once())
      ->method('exec');

    $this->schemaManager->createIndexTables($index, ['tokenizer' => 'unicode61']);
  }

  /**
   * Tests createIndexTables() uses configured tokenizer.
   */
  public function testCreateIndexTablesUsesConfiguredTokenizer(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $titleField = $this->createMock(FieldInterface::class);
    $titleField->method('getType')->willReturn('text');
    $titleField->method('getBoost')->willReturn(1.0);

    $index->method('getFields')->willReturn(['title' => $titleField]);

    $this->fieldTypeMapper->method('isFulltextType')
      ->willReturn(TRUE);

    $schema = $this->createMock(Schema::class);
    $schema->method('tableExists')->willReturn(FALSE);
    $schema->method('createTable');

    $connection = $this->createMock(Connection::class);
    $connection->method('schema')->willReturn($schema);

    $this->connectionManager->method('getConnection')
      ->willReturn($connection);

    // Check that porter tokenizer is used.
    $this->pdo->expects($this->once())
      ->method('exec');

    $this->schemaManager->createIndexTables($index, ['tokenizer' => 'porter']);
  }

  /**
   * Tests createIndexTables() with trigram tokenizer (case-insensitive).
   */
  public function testCreateIndexTablesWithTrigramTokenizer(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $titleField = $this->createMock(FieldInterface::class);
    $titleField->method('getType')->willReturn('text');
    $titleField->method('getBoost')->willReturn(1.0);

    $index->method('getFields')->willReturn(['title' => $titleField]);

    $this->fieldTypeMapper->method('isFulltextType')
      ->willReturn(TRUE);

    $schema = $this->createMock(Schema::class);
    $schema->method('tableExists')->willReturn(FALSE);
    $schema->method('createTable');

    $connection = $this->createMock(Connection::class);
    $connection->method('schema')->willReturn($schema);

    $this->connectionManager->method('getConnection')
      ->willReturn($connection);

    // Check that trigram tokenizer is used.
    $this->pdo->expects($this->once())
      ->method('exec');

    $this->schemaManager->createIndexTables($index, [
      'tokenizer' => 'trigram',
      'trigram_case_sensitive' => FALSE,
    ]);
  }

  /**
   * Tests createIndexTables() with trigram tokenizer (case-sensitive).
   */
  public function testCreateIndexTablesWithTrigramCaseSensitive(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $titleField = $this->createMock(FieldInterface::class);
    $titleField->method('getType')->willReturn('text');
    $titleField->method('getBoost')->willReturn(1.0);

    $index->method('getFields')->willReturn(['title' => $titleField]);

    $this->fieldTypeMapper->method('isFulltextType')
      ->willReturn(TRUE);

    $schema = $this->createMock(Schema::class);
    $schema->method('tableExists')->willReturn(FALSE);
    $schema->method('createTable');

    $connection = $this->createMock(Connection::class);
    $connection->method('schema')->willReturn($schema);

    $this->connectionManager->method('getConnection')
      ->willReturn($connection);

    // Check that trigram tokenizer with case_sensitive is used.
    $this->pdo->expects($this->once())
      ->method('exec');

    $this->schemaManager->createIndexTables($index, [
      'tokenizer' => 'trigram',
      'trigram_case_sensitive' => TRUE,
    ]);
  }

  /**
   * Tests tablesExist() handles query exception gracefully.
   */
  public function testTablesExistHandlesException(): void {
    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')
      ->willThrowException(new \PDOException('Database error'));

    $this->pdo->method('prepare')
      ->willReturn($stmt);

    $this->expectException(\PDOException::class);
    $this->schemaManager->tablesExist('test_index');
  }

}
