<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Search;

use PHPUnit\Framework\MockObject\MockObject;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\Select;
use Drupal\Core\Database\StatementInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Search\FacetBuilder;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the FacetBuilder service.
 */
#[CoversClass(FacetBuilder::class)]
#[Group('search_api_sqlite')]
final class FacetBuilderTest extends UnitTestCase {

  /**
   * The facet builder under test.
   */
  private FacetBuilder $facetBuilder;

  /**
   * The mocked connection manager.
   */
  private MockObject $connectionManager;

  /**
   * The mocked schema manager.
   */
  private MockObject $schemaManager;

  /**
   * The mocked database connection.
   */
  private MockObject $connection;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->connectionManager = $this->createMock(ConnectionManagerInterface::class);
    $this->schemaManager = $this->createMock(SchemaManagerInterface::class);
    $this->connection = $this->createMock(Connection::class);

    $this->connectionManager->method('getConnection')
      ->willReturn($this->connection);

    // Configure schema manager to return expected table names.
    $this->schemaManager->method('getFieldDataTableName')
      ->willReturnCallback(fn(string $index_id): string => $index_id . '_field_data');

    $this->facetBuilder = new FacetBuilder($this->connectionManager, $this->schemaManager);
  }

  /**
   * Tests calculateFacets() returns empty array for empty item IDs.
   */
  public function testCalculateFacetsReturnsEmptyForEmptyItemIds(): void {
    $result = $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      [],
      []
    );

    $this->assertSame([], $result);
  }

  /**
   * Tests calculateFacets() returns facet values with counts.
   */
  public function testCalculateFacetsReturnsValuesWithCounts(): void {
    $selectQuery = $this->createMockSelectQuery([
      (object) ['value' => 'Electronics', 'count' => 15],
      (object) ['value' => 'Clothing', 'count' => 12],
      (object) ['value' => 'Books', 'count' => 8],
    ]);

    $this->connection->method('select')
      ->willReturn($selectQuery);

    $result = $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      ['item_1', 'item_2', 'item_3'],
      ['limit' => 10, 'min_count' => 1]
    );

    $this->assertCount(3, $result);
    $this->assertSame('Electronics', $result[0]['value']);
    $this->assertSame(15, $result[0]['count']);
    $this->assertSame('Clothing', $result[1]['value']);
    $this->assertSame(12, $result[1]['count']);
    $this->assertSame('Books', $result[2]['value']);
    $this->assertSame(8, $result[2]['count']);
  }

  /**
   * Tests calculateFacets() applies limit option correctly.
   */
  public function testCalculateFacetsRespectsLimit(): void {
    $selectQuery = $this->createMock(Select::class);
    $selectQuery->method('addExpression')->willReturnSelf();
    $selectQuery->method('condition')->willReturnSelf();
    $selectQuery->method('groupBy')->willReturnSelf();
    $selectQuery->method('having')->willReturnSelf();
    $selectQuery->method('orderBy')->willReturnSelf();

    // Verify range is called with correct limit.
    $selectQuery->expects($this->once())
      ->method('range')
      ->with(0, 5)
      ->willReturnSelf();

    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchAll')->willReturn([]);
    $selectQuery->method('execute')->willReturn($statement);

    $this->connection->method('select')
      ->willReturn($selectQuery);

    $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      ['item_1'],
      ['limit' => 5]
    );
  }

  /**
   * Tests calculateFacets() applies minimum count filter.
   */
  public function testCalculateFacetsRespectsMinCount(): void {
    $selectQuery = $this->createMock(Select::class);
    $selectQuery->method('addExpression')->willReturnSelf();
    $selectQuery->method('condition')->willReturnSelf();
    $selectQuery->method('groupBy')->willReturnSelf();
    $selectQuery->method('orderBy')->willReturnSelf();
    $selectQuery->method('range')->willReturnSelf();

    // Verify having clause is called with min_count.
    $selectQuery->expects($this->once())
      ->method('having')
      ->with('count >= :min', [':min' => 3])
      ->willReturnSelf();

    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchAll')->willReturn([]);
    $selectQuery->method('execute')->willReturn($statement);

    $this->connection->method('select')
      ->willReturn($selectQuery);

    $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      ['item_1'],
      ['min_count' => 3]
    );
  }

  /**
   * Tests calculateFacets() returns empty array when query returns NULL.
   */
  public function testCalculateFacetsReturnsEmptyOnNullResult(): void {
    $selectQuery = $this->createMock(Select::class);
    $selectQuery->method('addExpression')->willReturnSelf();
    $selectQuery->method('condition')->willReturnSelf();
    $selectQuery->method('groupBy')->willReturnSelf();
    $selectQuery->method('having')->willReturnSelf();
    $selectQuery->method('orderBy')->willReturnSelf();
    $selectQuery->method('range')->willReturnSelf();
    $selectQuery->method('execute')->willReturn(NULL);

    $this->connection->method('select')
      ->willReturn($selectQuery);

    $result = $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      ['item_1'],
      []
    );

    $this->assertSame([], $result);
  }

  /**
   * Tests calculateMultipleFacets() calculates facets for multiple fields.
   */
  public function testCalculateMultipleFacetsCalculatesAllFields(): void {
    $categoryResults = [
      (object) ['value' => 'Electronics', 'count' => 10],
    ];
    $brandResults = [
      (object) ['value' => 'Apple', 'count' => 5],
      (object) ['value' => 'Samsung', 'count' => 3],
    ];

    $callCount = 0;
    $this->connection->method('select')
      ->willReturnCallback(function () use (&$callCount, $categoryResults, $brandResults): Select {
        $results = $callCount === 0 ? $categoryResults : $brandResults;
        $callCount++;
        return $this->createMockSelectQuery($results);
      });

    $result = $this->facetBuilder->calculateMultipleFacets(
      'test_index',
      [
        'category' => ['limit' => 10],
        'brand' => ['limit' => 5],
      ],
      ['item_1', 'item_2']
    );

    $this->assertArrayHasKey('category', $result);
    $this->assertArrayHasKey('brand', $result);
    $this->assertCount(1, $result['category']);
    $this->assertCount(2, $result['brand']);
  }

  /**
   * Tests calculateMultipleFacets() returns empty for each field when no items.
   */
  public function testCalculateMultipleFacetsWithNoItems(): void {
    $result = $this->facetBuilder->calculateMultipleFacets(
      'test_index',
      [
        'category' => ['limit' => 10],
        'brand' => ['limit' => 5],
      ],
      []
    );

    $this->assertArrayHasKey('category', $result);
    $this->assertArrayHasKey('brand', $result);
    $this->assertSame([], $result['category']);
    $this->assertSame([], $result['brand']);
  }

  /**
   * Tests calculateFacets() uses default options.
   */
  public function testCalculateFacetsUsesDefaultOptions(): void {
    $selectQuery = $this->createMock(Select::class);
    $selectQuery->method('addExpression')->willReturnSelf();
    $selectQuery->method('condition')->willReturnSelf();
    $selectQuery->method('groupBy')->willReturnSelf();
    $selectQuery->method('orderBy')->willReturnSelf();

    // Default limit is 10.
    $selectQuery->expects($this->once())
      ->method('range')
      ->with(0, 10)
      ->willReturnSelf();

    // Default min_count is 1.
    $selectQuery->expects($this->once())
      ->method('having')
      ->with('count >= :min', [':min' => 1])
      ->willReturnSelf();

    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchAll')->willReturn([]);
    $selectQuery->method('execute')->willReturn($statement);

    $this->connection->method('select')
      ->willReturn($selectQuery);

    $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      ['item_1'],
      []
    );
  }

  /**
   * Tests calculateFacets() batches large item ID sets.
   */
  public function testCalculateFacetsBatchesLargeItemSets(): void {
    // Generate 600 item IDs (exceeds 500 batch size).
    $item_ids = [];
    for ($i = 1; $i <= 600; $i++) {
      $item_ids[] = 'item_' . $i;
    }

    // First batch returns some results.
    $batch1Results = [
      (object) ['value' => 'Electronics', 'count' => 100],
      (object) ['value' => 'Clothing', 'count' => 80],
    ];

    // Second batch returns overlapping and new results.
    $batch2Results = [
      (object) ['value' => 'Electronics', 'count' => 50],
      (object) ['value' => 'Books', 'count' => 30],
    ];

    $callCount = 0;
    $this->connection->method('select')
      ->willReturnCallback(function () use (&$callCount, $batch1Results, $batch2Results): Select {
        $results = $callCount === 0 ? $batch1Results : $batch2Results;
        $callCount++;
        return $this->createMockSelectQuery($results);
      });

    $result = $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      $item_ids,
      ['limit' => 10, 'min_count' => 1]
    );

    // Should have called select twice (2 batches: 500 + 100).
    $this->assertSame(2, $callCount);

    // Results should be aggregated.
    $this->assertCount(3, $result);

    // Electronics should be first with combined count (100 + 50 = 150).
    $this->assertSame('Electronics', $result[0]['value']);
    $this->assertSame(150, $result[0]['count']);

    // Clothing should be second (80).
    $this->assertSame('Clothing', $result[1]['value']);
    $this->assertSame(80, $result[1]['count']);

    // Books should be third (30).
    $this->assertSame('Books', $result[2]['value']);
    $this->assertSame(30, $result[2]['count']);
  }

  /**
   * Tests batched facets respects limit after aggregation.
   */
  public function testBatchedFacetsRespectsLimit(): void {
    // Generate 600 item IDs.
    $item_ids = [];
    for ($i = 1; $i <= 600; $i++) {
      $item_ids[] = 'item_' . $i;
    }

    $batch1Results = [
      (object) ['value' => 'A', 'count' => 100],
      (object) ['value' => 'B', 'count' => 80],
      (object) ['value' => 'C', 'count' => 60],
    ];

    $batch2Results = [
      (object) ['value' => 'A', 'count' => 50],
      (object) ['value' => 'D', 'count' => 40],
    ];

    $callCount = 0;
    $this->connection->method('select')
      ->willReturnCallback(function () use (&$callCount, $batch1Results, $batch2Results): Select {
        $results = $callCount === 0 ? $batch1Results : $batch2Results;
        $callCount++;
        return $this->createMockSelectQuery($results);
      });

    $result = $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      $item_ids,
      ['limit' => 2, 'min_count' => 1]
    );

    // Should only return 2 results (limit).
    $this->assertCount(2, $result);
    $this->assertSame('A', $result[0]['value']);
    $this->assertSame(150, $result[0]['count']);
    $this->assertSame('B', $result[1]['value']);
    $this->assertSame(80, $result[1]['count']);
  }

  /**
   * Tests batched facets respects min_count after aggregation.
   */
  public function testBatchedFacetsRespectsMinCount(): void {
    // Generate 600 item IDs.
    $item_ids = [];
    for ($i = 1; $i <= 600; $i++) {
      $item_ids[] = 'item_' . $i;
    }

    $batch1Results = [
      (object) ['value' => 'Popular', 'count' => 100],
      (object) ['value' => 'Rare', 'count' => 2],
    ];

    $batch2Results = [
      (object) ['value' => 'Popular', 'count' => 50],
      (object) ['value' => 'Rare', 'count' => 1],
    ];

    $callCount = 0;
    $this->connection->method('select')
      ->willReturnCallback(function () use (&$callCount, $batch1Results, $batch2Results): Select {
        $results = $callCount === 0 ? $batch1Results : $batch2Results;
        $callCount++;
        return $this->createMockSelectQuery($results);
      });

    $result = $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      $item_ids,
      ['limit' => 10, 'min_count' => 5]
    );

    // Only 'Popular' should pass min_count (150 >= 5).
    // 'Rare' has 3 total which is < 5.
    $this->assertCount(1, $result);
    $this->assertSame('Popular', $result[0]['value']);
    $this->assertSame(150, $result[0]['count']);
  }

  /**
   * Tests that small item sets do not trigger batching.
   */
  public function testSmallItemSetsDoNotBatch(): void {
    // 500 items exactly - should NOT trigger batching.
    $item_ids = [];
    for ($i = 1; $i <= 500; $i++) {
      $item_ids[] = 'item_' . $i;
    }

    $results = [
      (object) ['value' => 'Test', 'count' => 100],
    ];

    $callCount = 0;
    $this->connection->method('select')
      ->willReturnCallback(function () use (&$callCount, $results): Select {
        $callCount++;
        return $this->createMockSelectQuery($results);
      });

    $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      $item_ids,
      ['limit' => 10]
    );

    // Should only call select once (no batching).
    $this->assertSame(1, $callCount);
  }

  /**
   * Tests calculateMultipleFacets() uses temp table for 3+ facets.
   */
  public function testCalculateMultipleFacetsUsesTempTableForThreeOrMoreFacets(): void {
    $pdo = $this->createMockPdo([
      (object) ['field_name' => 'category', 'value' => 'Electronics', 'count' => 50],
      (object) ['field_name' => 'category', 'value' => 'Clothing', 'count' => 30],
      (object) ['field_name' => 'brand', 'value' => 'Apple', 'count' => 25],
      (object) ['field_name' => 'brand', 'value' => 'Samsung', 'count' => 20],
      (object) ['field_name' => 'color', 'value' => 'Red', 'count' => 15],
    ]);

    $this->connectionManager->method('getPdo')
      ->willReturn($pdo);

    $result = $this->facetBuilder->calculateMultipleFacets(
      'test_index',
      [
        'category' => ['limit' => 10],
        'brand' => ['limit' => 10],
        'color' => ['limit' => 10],
      ],
      ['item_1', 'item_2', 'item_3']
    );

    // Should have results for all three facets.
    $this->assertArrayHasKey('category', $result);
    $this->assertArrayHasKey('brand', $result);
    $this->assertArrayHasKey('color', $result);

    // Verify category facet.
    $this->assertCount(2, $result['category']);
    $this->assertSame('Electronics', $result['category'][0]['value']);
    $this->assertSame(50, $result['category'][0]['count']);

    // Verify brand facet.
    $this->assertCount(2, $result['brand']);
    $this->assertSame('Apple', $result['brand'][0]['value']);

    // Verify color facet.
    $this->assertCount(1, $result['color']);
    $this->assertSame('Red', $result['color'][0]['value']);
  }

  /**
   * Tests calculateMultipleFacets() applies per-facet limits.
   */
  public function testTempTableAppliesPerFacetLimits(): void {
    $pdo = $this->createMockPdo([
      (object) ['field_name' => 'category', 'value' => 'A', 'count' => 100],
      (object) ['field_name' => 'category', 'value' => 'B', 'count' => 80],
      (object) ['field_name' => 'category', 'value' => 'C', 'count' => 60],
      (object) ['field_name' => 'brand', 'value' => 'X', 'count' => 50],
      (object) ['field_name' => 'brand', 'value' => 'Y', 'count' => 40],
      (object) ['field_name' => 'brand', 'value' => 'Z', 'count' => 30],
      (object) ['field_name' => 'size', 'value' => 'L', 'count' => 20],
    ]);

    $this->connectionManager->method('getPdo')
      ->willReturn($pdo);

    $result = $this->facetBuilder->calculateMultipleFacets(
      'test_index',
      [
        'category' => ['limit' => 2],
        'brand' => ['limit' => 1],
        'size' => ['limit' => 10],
      ],
      ['item_1', 'item_2']
    );

    // Category should have 2 (limited).
    $this->assertCount(2, $result['category']);
    $this->assertSame('A', $result['category'][0]['value']);
    $this->assertSame('B', $result['category'][1]['value']);

    // Brand should have 1 (limited).
    $this->assertCount(1, $result['brand']);
    $this->assertSame('X', $result['brand'][0]['value']);

    // Size should have 1 (all available).
    $this->assertCount(1, $result['size']);
  }

  /**
   * Tests calculateMultipleFacets() applies per-facet min_count.
   */
  public function testTempTableAppliesPerFacetMinCount(): void {
    $pdo = $this->createMockPdo([
      (object) ['field_name' => 'category', 'value' => 'Popular', 'count' => 100],
      (object) ['field_name' => 'category', 'value' => 'Rare', 'count' => 2],
      (object) ['field_name' => 'brand', 'value' => 'Common', 'count' => 50],
      (object) ['field_name' => 'brand', 'value' => 'Uncommon', 'count' => 3],
      (object) ['field_name' => 'size', 'value' => 'Medium', 'count' => 10],
    ]);

    $this->connectionManager->method('getPdo')
      ->willReturn($pdo);

    $result = $this->facetBuilder->calculateMultipleFacets(
      'test_index',
      [
        'category' => ['limit' => 10, 'min_count' => 5],
        'brand' => ['limit' => 10, 'min_count' => 10],
        'size' => ['limit' => 10, 'min_count' => 1],
      ],
      ['item_1', 'item_2']
    );

    // Category: only 'Popular' (100) passes min_count of 5.
    $this->assertCount(1, $result['category']);
    $this->assertSame('Popular', $result['category'][0]['value']);

    // Brand: only 'Common' (50) passes min_count of 10.
    $this->assertCount(1, $result['brand']);
    $this->assertSame('Common', $result['brand'][0]['value']);

    // Size: 'Medium' (10) passes min_count of 1.
    $this->assertCount(1, $result['size']);
  }

  /**
   * Tests calculateMultipleFacets() uses temp table for large item sets.
   */
  public function testCalculateMultipleFacetsUsesTempTableForLargeItemSets(): void {
    // Generate 600 item IDs (exceeds 500 batch size).
    $item_ids = [];
    for ($i = 1; $i <= 600; $i++) {
      $item_ids[] = 'item_' . $i;
    }

    $pdo = $this->createMockPdo([
      (object) ['field_name' => 'category', 'value' => 'Test', 'count' => 100],
      (object) ['field_name' => 'brand', 'value' => 'TestBrand', 'count' => 50],
    ]);

    $this->connectionManager->method('getPdo')
      ->willReturn($pdo);

    // Even with only 2 facets, should use temp table due to large item set.
    $result = $this->facetBuilder->calculateMultipleFacets(
      'test_index',
      [
        'category' => ['limit' => 10],
        'brand' => ['limit' => 10],
      ],
      $item_ids
    );

    $this->assertArrayHasKey('category', $result);
    $this->assertArrayHasKey('brand', $result);
    $this->assertCount(1, $result['category']);
    $this->assertCount(1, $result['brand']);
  }

  /**
   * Tests calculateMultipleFacets() returns empty arrays for empty items.
   */
  public function testCalculateMultipleFacetsReturnsEmptyForEmptyItems(): void {
    $result = $this->facetBuilder->calculateMultipleFacets(
      'test_index',
      [
        'category' => ['limit' => 10],
        'brand' => ['limit' => 10],
        'color' => ['limit' => 10],
      ],
      []
    );

    $this->assertArrayHasKey('category', $result);
    $this->assertArrayHasKey('brand', $result);
    $this->assertArrayHasKey('color', $result);
    $this->assertSame([], $result['category']);
    $this->assertSame([], $result['brand']);
    $this->assertSame([], $result['color']);
  }

  /**
   * Tests calculateMultipleFacets() returns empty array for missing field.
   */
  public function testTempTableReturnsEmptyForMissingFields(): void {
    // Only category has results.
    $pdo = $this->createMockPdo([
      (object) ['field_name' => 'category', 'value' => 'Test', 'count' => 100],
    ]);

    $this->connectionManager->method('getPdo')
      ->willReturn($pdo);

    $result = $this->facetBuilder->calculateMultipleFacets(
      'test_index',
      [
        'category' => ['limit' => 10],
        'brand' => ['limit' => 10],
        'color' => ['limit' => 10],
      ],
      ['item_1', 'item_2']
    );

    // Category has results.
    $this->assertCount(1, $result['category']);

    // Brand and color should be empty arrays.
    $this->assertSame([], $result['brand']);
    $this->assertSame([], $result['color']);
  }

  /**
   * Creates a mock select query that returns given results.
   *
   * @param array<object> $results
   *   The results to return.
   *
   * @return \Drupal\Core\Database\Query\Select
   *   The mock select query.
   */
  private function createMockSelectQuery(array $results): Select {
    $selectQuery = $this->createMock(Select::class);
    $selectQuery->method('addExpression')->willReturnSelf();
    $selectQuery->method('condition')->willReturnSelf();
    $selectQuery->method('groupBy')->willReturnSelf();
    $selectQuery->method('having')->willReturnSelf();
    $selectQuery->method('orderBy')->willReturnSelf();
    $selectQuery->method('range')->willReturnSelf();

    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchAll')->willReturn($results);
    $selectQuery->method('execute')->willReturn($statement);

    return $selectQuery;
  }

  /**
   * Creates a mock PDO connection for temp table tests.
   *
   * @param array<object> $facetResults
   *   The facet results to return from the query.
   *
   * @return \PDO&\PHPUnit\Framework\MockObject\MockObject
   *   The mock PDO connection.
   */
  private function createMockPdo(array $facetResults): MockObject {
    $pdo = $this->createMock(\PDO::class);

    // Mock the PDO statement for the facet query.
    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willReturn(TRUE);
    $stmt->method('fetchAll')->willReturn($facetResults);

    $pdo->method('prepare')->willReturn($stmt);
    $pdo->method('exec')->willReturn(0);

    return $pdo;
  }

}
