<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Database;

/**
 * Executes FTS5-specific queries using raw PDO.
 */
final readonly class Fts5QueryRunner implements Fts5QueryRunnerInterface {

  /**
   * Constructs a Fts5QueryRunner instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The connection manager.
   */
  public function __construct(
    private ConnectionManagerInterface $connectionManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function search(string $index_id, string $table_name, string $match_query, array $options = []): array {
    $busy_timeout = $options['busy_timeout'] ?? 10000;
    $pdo = $this->connectionManager->getReadOnlyPdo($index_id, $busy_timeout);

    $limit = $options['limit'] ?? 10;
    $offset = $options['offset'] ?? 0;
    $order_by_rank = $options['order_by_rank'] ?? TRUE;

    // Build the query with BM25 scoring.
    $sql = sprintf(
      'SELECT item_id, bm25(%s) AS bm25_score FROM %s WHERE %s MATCH :query',
      $table_name,
      $table_name,
      $table_name
    );

    if ($order_by_rank) {
      $sql .= ' ORDER BY bm25_score';
    }

    $sql .= ' LIMIT :limit OFFSET :offset';

    try {
      $stmt = $pdo->prepare($sql);
      $stmt->bindValue(':query', $match_query, \PDO::PARAM_STR);
      $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
      $stmt->bindValue(':offset', $offset, \PDO::PARAM_INT);
      $stmt->execute();

      return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }
    catch (\PDOException $pdoException) {
      // Re-throw with more context about the query.
      throw new \PDOException(
        $pdoException->getMessage() . ' [Query: ' . $match_query . ']',
        (int) $pdoException->getCode(),
        $pdoException
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function highlight(
    string $index_id,
    string $table_name,
    string $match_query,
    string $column,
    array $item_ids,
    string $prefix = '<mark>',
    string $suffix = '</mark>',
    int $busy_timeout = 10000,
  ): array {
    if ($item_ids === []) {
      return [];
    }

    $pdo = $this->connectionManager->getReadOnlyPdo($index_id, $busy_timeout);

    // Build named placeholders for item IDs.
    // Avoids mixing named and positional parameters in PDO.
    $placeholders = [];
    foreach (array_keys($item_ids) as $i) {
      $placeholders[] = ':id_' . $i;
    }

    $placeholder_str = implode(',', $placeholders);

    // Get column index for highlight function.
    $column_index = $this->getColumnIndex($pdo, $table_name, $column);

    $sql = sprintf(
      'SELECT item_id, highlight(%s, %d, :prefix, :suffix) AS highlighted
       FROM %s
       WHERE %s MATCH :query AND item_id IN (%s)',
      $table_name,
      $column_index,
      $table_name,
      $table_name,
      $placeholder_str
    );

    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':query', $match_query, \PDO::PARAM_STR);
    $stmt->bindValue(':prefix', $prefix, \PDO::PARAM_STR);
    $stmt->bindValue(':suffix', $suffix, \PDO::PARAM_STR);

    // Bind item IDs with named parameters.
    foreach (array_values($item_ids) as $i => $item_id) {
      $stmt->bindValue(':id_' . $i, $item_id, \PDO::PARAM_STR);
    }

    $stmt->execute();

    $results = [];
    while ($row = $stmt->fetch(\PDO::FETCH_ASSOC)) {
      $results[$row['item_id']] = $row['highlighted'];
    }

    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function snippet(
    string $index_id,
    string $table_name,
    string $match_query,
    string $column,
    array $item_ids,
    string $prefix = '<mark>',
    string $suffix = '</mark>',
    int $length = 64,
    int $busy_timeout = 10000,
  ): array {
    if ($item_ids === []) {
      return [];
    }

    $pdo = $this->connectionManager->getReadOnlyPdo($index_id, $busy_timeout);

    // Build named placeholders for item IDs.
    // Avoids mixing named and positional parameters in PDO.
    $placeholders = [];
    foreach (array_keys($item_ids) as $i) {
      $placeholders[] = ':id_' . $i;
    }

    $placeholder_str = implode(',', $placeholders);

    // Get column index for snippet function.
    $column_index = $this->getColumnIndex($pdo, $table_name, $column);

    $sql = sprintf(
      'SELECT item_id, snippet(%s, %d, :prefix, :suffix, \'...\', :length) AS snippet
       FROM %s
       WHERE %s MATCH :query AND item_id IN (%s)',
      $table_name,
      $column_index,
      $table_name,
      $table_name,
      $placeholder_str
    );

    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':query', $match_query, \PDO::PARAM_STR);
    $stmt->bindValue(':prefix', $prefix, \PDO::PARAM_STR);
    $stmt->bindValue(':suffix', $suffix, \PDO::PARAM_STR);
    $stmt->bindValue(':length', $length, \PDO::PARAM_INT);

    // Bind item IDs with named parameters.
    foreach (array_values($item_ids) as $i => $item_id) {
      $stmt->bindValue(':id_' . $i, $item_id, \PDO::PARAM_STR);
    }

    $stmt->execute();

    $results = [];
    while ($row = $stmt->fetch(\PDO::FETCH_ASSOC)) {
      $results[$row['item_id']] = $row['snippet'];
    }

    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function optimize(string $index_id, string $table_name): void {
    $pdo = $this->connectionManager->getPdo($index_id);

    // Run FTS5 optimize command.
    $sql = sprintf("INSERT INTO %s(%s) VALUES('optimize')", $table_name, $table_name);
    $pdo->exec($sql);
  }

  /**
   * {@inheritdoc}
   */
  public function rebuild(string $index_id, string $table_name): void {
    $pdo = $this->connectionManager->getPdo($index_id);

    // Run FTS5 rebuild command.
    $sql = sprintf("INSERT INTO %s(%s) VALUES('rebuild')", $table_name, $table_name);
    $pdo->exec($sql);
  }

  /**
   * {@inheritdoc}
   */
  public function getDocumentCount(string $index_id, string $table_name): int {
    $pdo = $this->connectionManager->getReadOnlyPdo($index_id);

    $sql = sprintf('SELECT COUNT(*) FROM %s', $table_name);
    $result = $pdo->query($sql);

    if ($result === FALSE) {
      return 0;
    }

    return (int) $result->fetchColumn();
  }

  /**
   * {@inheritdoc}
   */
  public function getMatchCount(string $index_id, string $table_name, string $match_query): int {
    $pdo = $this->connectionManager->getReadOnlyPdo($index_id);

    $sql = sprintf(
      'SELECT COUNT(*) FROM %s WHERE %s MATCH :query',
      $table_name,
      $table_name
    );

    try {
      $stmt = $pdo->prepare($sql);
      $stmt->bindValue(':query', $match_query, \PDO::PARAM_STR);
      $stmt->execute();

      return (int) $stmt->fetchColumn();
    }
    catch (\PDOException $pdoException) {
      // Re-throw with more context about the query.
      throw new \PDOException(
        $pdoException->getMessage() . ' [Match query: ' . $match_query . ']',
        (int) $pdoException->getCode(),
        $pdoException
      );
    }
  }

  /**
   * Gets the column index for a column name.
   *
   * FTS5 highlight() and snippet() functions require column index (0-based).
   *
   * @param \PDO $pdo
   *   The PDO connection.
   * @param string $table_name
   *   The FTS5 table name.
   * @param string $column_name
   *   The column name.
   *
   * @return int
   *   The column index.
   */
  private function getColumnIndex(\PDO $pdo, string $table_name, string $column_name): int {
    $result = $pdo->query(sprintf('PRAGMA table_info(%s)', $table_name));
    $columns = [];

    if ($result !== FALSE) {
      while ($row = $result->fetch(\PDO::FETCH_ASSOC)) {
        $columns[] = $row['name'];
      }
    }

    $index = array_search($column_name, $columns, TRUE);

    return $index !== FALSE ? $index : 0;
  }

}
