<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Database;

/**
 * Defines the interface for running FTS5 queries.
 */
interface Fts5QueryRunnerInterface {

  /**
   * Executes an FTS5 search query.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $table_name
   *   The FTS5 table name.
   * @param string $match_query
   *   The FTS5 MATCH query string.
   * @param array<string, mixed> $options
   *   Additional options:
   *   - limit: Maximum number of results.
   *   - offset: Number of results to skip.
   *   - order_by_rank: Whether to order by BM25 rank.
   *   - busy_timeout: SQLite busy timeout in milliseconds.
   *
   * @return array<array<string, mixed>>
   *   Array of result rows with item_id and bm25_score.
   */
  public function search(string $index_id, string $table_name, string $match_query, array $options = []): array;

  /**
   * Gets highlighted snippets for search results.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $table_name
   *   The FTS5 table name.
   * @param string $match_query
   *   The FTS5 MATCH query string.
   * @param string $column
   *   The column to highlight.
   * @param array<string> $item_ids
   *   Item IDs to get highlights for.
   * @param string $prefix
   *   The highlight start tag.
   * @param string $suffix
   *   The highlight end tag.
   * @param int $busy_timeout
   *   SQLite busy timeout in milliseconds.
   *
   * @return array<string, string>
   *   Array of highlighted text keyed by item_id.
   */
  public function highlight(
    string $index_id,
    string $table_name,
    string $match_query,
    string $column,
    array $item_ids,
    string $prefix = '<mark>',
    string $suffix = '</mark>',
    int $busy_timeout = 10000,
  ): array;

  /**
   * Gets snippets (excerpts) for search results.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $table_name
   *   The FTS5 table name.
   * @param string $match_query
   *   The FTS5 MATCH query string.
   * @param string $column
   *   The column to get snippets from.
   * @param array<string> $item_ids
   *   Item IDs to get snippets for.
   * @param string $prefix
   *   The highlight start tag.
   * @param string $suffix
   *   The highlight end tag.
   * @param int $length
   *   Maximum snippet length in tokens.
   * @param int $busy_timeout
   *   SQLite busy timeout in milliseconds.
   *
   * @return array<string, string>
   *   Array of snippets keyed by item_id.
   */
  public function snippet(
    string $index_id,
    string $table_name,
    string $match_query,
    string $column,
    array $item_ids,
    string $prefix = '<mark>',
    string $suffix = '</mark>',
    int $length = 64,
    int $busy_timeout = 10000,
  ): array;

  /**
   * Optimizes the FTS5 index.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $table_name
   *   The FTS5 table name.
   */
  public function optimize(string $index_id, string $table_name): void;

  /**
   * Rebuilds the FTS5 index.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $table_name
   *   The FTS5 table name.
   */
  public function rebuild(string $index_id, string $table_name): void;

  /**
   * Gets the total count of documents in the FTS5 index.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $table_name
   *   The FTS5 table name.
   *
   * @return int
   *   The document count.
   */
  public function getDocumentCount(string $index_id, string $table_name): int;

  /**
   * Gets the count of documents matching an FTS5 query.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $table_name
   *   The FTS5 table name.
   * @param string $match_query
   *   The FTS5 MATCH query string.
   *
   * @return int
   *   The matching document count.
   */
  public function getMatchCount(string $index_id, string $table_name, string $match_query): int;

}
