<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Plugin\search_api\backend;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\search_api\Backend\BackendPluginBase;
use Drupal\search_api\Contrib\AutocompleteBackendInterface;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api\Query\ResultSetInterface;
use Drupal\search_api\SearchApiException;
use Drupal\search_api_autocomplete\SearchInterface;
use Drupal\search_api_sqlite\Autocomplete\AutocompleteHandlerInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Enum\MatchingMode;
use Drupal\search_api_sqlite\Enum\Tokenizer;
use Drupal\search_api_sqlite\Index\FieldTypeMapperInterface;
use Drupal\search_api_sqlite\Index\IndexerInterface;
use Drupal\search_api_sqlite\Search\ConditionHandlerInterface;
use Drupal\search_api_sqlite\Search\FacetBuilderInterface;
use Drupal\search_api_sqlite\Search\HighlighterInterface;
use Drupal\search_api_sqlite\Search\QueryBuilderInterface;
use Drupal\search_api_sqlite\Status\StatusReporterInterface;
use Drupal\search_api_sqlite\Utility\ColumnNameHelper;
use Drupal\search_api_sqlite\Utility\VerboseLoggerTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * SQLite FTS5 backend for Search API.
 *
 * @SearchApiBackend(
 *   id = "search_api_sqlite",
 *   label = @Translation("SQLite FTS5"),
 *   description = @Translation("High-performance search backend using SQLite FTS5 for full-text search.")
 * )
 */
final class SqliteFts5 extends BackendPluginBase implements ContainerFactoryPluginInterface, PluginFormInterface, AutocompleteBackendInterface {

  use VerboseLoggerTrait;

  /**
   * The connection manager.
   */
  private ConnectionManagerInterface $connectionManager;

  /**
   * The schema manager.
   */
  private SchemaManagerInterface $schemaManager;

  /**
   * The FTS5 query runner.
   */
  private Fts5QueryRunnerInterface $fts5QueryRunner;

  /**
   * The indexer.
   */
  private IndexerInterface $indexer;

  /**
   * The field type mapper.
   */
  private FieldTypeMapperInterface $fieldTypeMapper;

  /**
   * The query builder.
   */
  private QueryBuilderInterface $queryBuilder;

  /**
   * The highlighter.
   */
  private HighlighterInterface $highlighter;

  /**
   * The facet builder.
   */
  private FacetBuilderInterface $facetBuilder;

  /**
   * The condition handler.
   */
  private ConditionHandlerInterface $conditionHandler;

  /**
   * The status reporter.
   */
  private StatusReporterInterface $statusReporter;

  /**
   * The autocomplete handler.
   */
  private AutocompleteHandlerInterface $autocompleteHandler;

  /**
   * {@inheritdoc}
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   * @param array<string, mixed> $configuration
   *   Plugin configuration.
   * @param string $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definition.
   */
  #[\Override]
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = new self($configuration, $plugin_id, $plugin_definition);
    $instance->connectionManager = $container->get('search_api_sqlite.connection_manager');
    $instance->schemaManager = $container->get('search_api_sqlite.schema_manager');
    $instance->fts5QueryRunner = $container->get('search_api_sqlite.fts5_query_runner');
    $instance->indexer = $container->get('search_api_sqlite.indexer');
    $instance->fieldTypeMapper = $container->get('search_api_sqlite.field_type_mapper');
    $instance->queryBuilder = $container->get('search_api_sqlite.query_builder');
    $instance->highlighter = $container->get('search_api_sqlite.highlighter');
    $instance->facetBuilder = $container->get('search_api_sqlite.facet_builder');
    $instance->conditionHandler = $container->get('search_api_sqlite.condition_handler');
    $instance->statusReporter = $container->get('search_api_sqlite.status_reporter');
    $instance->autocompleteHandler = $container->get('search_api_sqlite.autocomplete_handler');
    $instance->setLogger($container->get('logger.channel.search_api_sqlite'));
    return $instance;
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string, mixed>
   *   The default configuration.
   */
  #[\Override]
  public function defaultConfiguration(): array {
    return [
      'database_path' => 'private://search_api_sqlite',
      'min_chars' => 3,
      'tokenizer' => Tokenizer::Unicode61->value,
      'trigram_case_sensitive' => FALSE,
      'matching' => MatchingMode::Words->value,
      'verbose_logging' => FALSE,
      'highlighting' => [
        'enabled' => TRUE,
        'prefix' => '<strong>',
        'suffix' => '</strong>',
        'excerpt_length' => 256,
      ],
      'optimization' => [
        'auto_optimize' => TRUE,
        'optimize_threshold' => 1000,
      ],
      'concurrency' => [
        'busy_timeout' => 10000,
        'max_retries' => 5,
        'retry_delay' => 200,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array<string, mixed>
   *   The built form.
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['tokenizer'] = [
      '#type' => 'select',
      '#title' => $this->t('Tokenizer'),
      '#description' => $this->t('<p>Determines how text is split into searchable tokens:</p><ul><li><strong>Unicode61</strong>: Best for most languages, handles accents and international characters.</li><li><strong>Porter Stemmer</strong>: Reduces words to their root form (e.g., "running" → "run"), improves recall for English content.</li><li><strong>ASCII</strong>: Simple and fast, only for basic English text without special characters.</li><li><strong>Trigram</strong>: Enables substring matching (e.g., "abc" matches "xabcdef"). Ideal for product codes, SKUs, and part numbers. <em>Warning: Creates larger indexes.</em></li></ul>'),
      '#options' => Tokenizer::formOptions(),
      '#default_value' => $this->configuration['tokenizer'],
    ];

    $form['trigram_case_sensitive'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Case-sensitive matching'),
      '#description' => $this->t('Enable case-sensitive matching for trigram tokenizer. When disabled, "ABC" matches "abc".'),
      '#default_value' => $this->configuration['trigram_case_sensitive'],
      '#states' => [
        'visible' => [
          ':input[name="backend_config[tokenizer]"]' => ['value' => Tokenizer::Trigram->value],
        ],
      ],
    ];

    $form['min_chars'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum word length'),
      '#description' => $this->t('Minimum number of characters for a word to be indexed. Not applicable for trigram tokenizer.'),
      '#default_value' => $this->configuration['min_chars'],
      '#min' => 1,
      '#max' => 10,
      '#states' => [
        'invisible' => [
          ':input[name="backend_config[tokenizer]"]' => ['value' => Tokenizer::Trigram->value],
        ],
      ],
    ];

    $form['matching'] = [
      '#type' => 'select',
      '#title' => $this->t('Default matching mode'),
      '#description' => $this->t('<p>Determines how multiple search terms are matched:</p><ul><li><strong>Match all words</strong>: All terms must appear in results (e.g., "red car" finds only items with both words).</li><li><strong>Prefix matching</strong>: Matches word beginnings, ideal for autocomplete (e.g., "auto" matches "automatic", "automobile").</li><li><strong>Phrase matching</strong>: Terms must appear consecutively in exact order (e.g., "red car" won\'t match "car is red").</li></ul>'),
      '#options' => MatchingMode::formOptions(),
      '#default_value' => $this->configuration['matching'],
    ];

    $form['highlighting'] = [
      '#type' => 'details',
      '#title' => $this->t('Highlighting'),
      '#description' => $this->t('Native SQLite FTS5 highlighting generates excerpts directly from the search index at query time. This is faster than the Search API Highlighting processor but works only with indexed content. If you need to highlight original HTML content or fields not in the FTS index, disable this and use the Highlighting processor on your index instead.'),
      '#open' => TRUE,
    ];

    $form['highlighting']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable native FTS5 highlighting'),
      '#description' => $this->t('Use SQLite FTS5 built-in highlighting. Disable if using the Search API Highlighting processor.'),
      '#default_value' => $this->configuration['highlighting']['enabled'],
    ];

    $form['highlighting']['prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Highlight prefix'),
      '#default_value' => $this->configuration['highlighting']['prefix'],
      '#states' => [
        'visible' => [
          ':input[name="backend_config[highlighting][enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['highlighting']['suffix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Highlight suffix'),
      '#default_value' => $this->configuration['highlighting']['suffix'],
      '#states' => [
        'visible' => [
          ':input[name="backend_config[highlighting][enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['highlighting']['excerpt_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Excerpt length'),
      '#description' => $this->t('Maximum number of tokens in excerpts.'),
      '#default_value' => $this->configuration['highlighting']['excerpt_length'],
      '#min' => 32,
      '#max' => 512,
      '#states' => [
        'visible' => [
          ':input[name="backend_config[highlighting][enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['verbose_logging'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable verbose logging'),
      '#description' => $this->t('Log informational messages (e.g., indexing counts, optimization events). When disabled, only warnings and errors are logged. Recommended to disable in production for cleaner logs.'),
      '#default_value' => $this->configuration['verbose_logging'],
    ];

    $form['optimization'] = [
      '#type' => 'details',
      '#title' => $this->t('Optimization'),
      '#open' => FALSE,
    ];

    $form['optimization']['auto_optimize'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable automatic optimization'),
      '#description' => $this->t('Automatically optimize the FTS5 index after a threshold of changes.'),
      '#default_value' => $this->configuration['optimization']['auto_optimize'],
    ];

    $form['optimization']['optimize_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Optimization threshold'),
      '#description' => $this->t('Number of changes before automatic optimization.'),
      '#default_value' => $this->configuration['optimization']['optimize_threshold'],
      '#min' => 100,
      '#max' => 10000,
      '#states' => [
        'visible' => [
          ':input[name="backend_config[optimization][auto_optimize]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['concurrency'] = [
      '#type' => 'details',
      '#title' => $this->t('Concurrency'),
      '#description' => $this->t('Settings to handle concurrent access. SQLite supports multiple readers but only one writer at a time. These settings help manage write contention on busy sites.'),
      '#open' => FALSE,
    ];

    $form['concurrency']['busy_timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Busy timeout (ms)'),
      '#description' => $this->t('How long SQLite waits for a locked database before returning an error. Higher values reduce failures under load but may increase response times.'),
      '#default_value' => $this->configuration['concurrency']['busy_timeout'],
      '#min' => 1000,
      '#max' => 60000,
      '#field_suffix' => 'ms',
    ];

    $form['concurrency']['max_retries'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum retries'),
      '#description' => $this->t('Number of times to retry a write operation if the database is locked.'),
      '#default_value' => $this->configuration['concurrency']['max_retries'],
      '#min' => 1,
      '#max' => 10,
    ];

    $form['concurrency']['retry_delay'] = [
      '#type' => 'number',
      '#title' => $this->t('Retry delay (ms)'),
      '#description' => $this->t('Base delay between retries. Uses exponential backoff (delay × attempt number).'),
      '#default_value' => $this->configuration['concurrency']['retry_delay'],
      '#min' => 50,
      '#max' => 2000,
      '#field_suffix' => 'ms',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // No additional validation needed.
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();

    // Check if tokenizer config changed (requires reindex).
    $old_tokenizer = $this->configuration['tokenizer'] ?? Tokenizer::Unicode61->value;
    $old_trigram_case = $this->configuration['trigram_case_sensitive'] ?? FALSE;
    $new_tokenizer = $values['tokenizer'];
    $new_trigram_case = (bool) ($values['trigram_case_sensitive'] ?? FALSE);

    $tokenizer_changed = ($old_tokenizer !== $new_tokenizer)
      || ($new_tokenizer === Tokenizer::Trigram->value && $old_trigram_case !== $new_trigram_case);

    $this->configuration['tokenizer'] = $values['tokenizer'];
    $this->configuration['trigram_case_sensitive'] = (bool) ($values['trigram_case_sensitive'] ?? FALSE);
    $this->configuration['min_chars'] = (int) $values['min_chars'];
    $this->configuration['matching'] = $values['matching'];
    $this->configuration['verbose_logging'] = (bool) ($values['verbose_logging'] ?? FALSE);
    $this->configuration['highlighting'] = [
      'enabled' => (bool) $values['highlighting']['enabled'],
      'prefix' => $values['highlighting']['prefix'],
      'suffix' => $values['highlighting']['suffix'],
      'excerpt_length' => (int) $values['highlighting']['excerpt_length'],
    ];
    $this->configuration['optimization'] = [
      'auto_optimize' => (bool) $values['optimization']['auto_optimize'],
      'optimize_threshold' => (int) $values['optimization']['optimize_threshold'],
    ];
    $this->configuration['concurrency'] = [
      'busy_timeout' => (int) ($values['concurrency']['busy_timeout'] ?? $this->configuration['concurrency']['busy_timeout'] ?? 10000),
      'max_retries' => (int) ($values['concurrency']['max_retries'] ?? $this->configuration['concurrency']['max_retries'] ?? 5),
      'retry_delay' => (int) ($values['concurrency']['retry_delay'] ?? $this->configuration['concurrency']['retry_delay'] ?? 200),
    ];

    // If tokenizer changed, rebuild indexes and trigger reindex.
    if ($tokenizer_changed) {
      $this->rebuildIndexesForTokenizerChange();
    }
  }

  /**
   * Rebuilds all indexes when tokenizer configuration changes.
   */
  private function rebuildIndexesForTokenizerChange(): void {
    $server = $this->getServer();

    foreach ($server->getIndexes() as $index) {
      if (!$index->isReadOnly() && $index->status()) {
        // Rebuild the FTS5 table with new tokenizer.
        $this->schemaManager->updateIndexSchema($index, $this->configuration);
        // Mark all items for reindexing.
        $index->reindex();
        $this->getMessenger()->addWarning($this->t(
          'Tokenizer changed for index %index. All items have been queued for reindexing.',
          ['%index' => $index->label()]
        ));
      }
    }
  }

  /**
   * {@inheritdoc}
   *
   * @return array<int, array<string, mixed>>
   *   An array of settings info.
   */
  #[\Override]
  public function viewSettings(): array {
    return $this->statusReporter->getViewSettings($this->configuration, $this->getServer());
  }

  /**
   * {@inheritdoc}
   */
  public function addIndex(IndexInterface $index): void {
    $this->schemaManager->createIndexTables($index, $this->configuration);

    $this->logVerbose('Created SQLite FTS5 index tables for @index', [
      '@index' => $index->id(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function updateIndex(IndexInterface $index): void {
    $needs_reindex = $this->schemaManager->updateIndexSchema($index, $this->configuration);

    if ($needs_reindex) {
      $index->reindex();
      $this->logVerbose('Index @index schema changed, reindexing required.', [
        '@index' => $index->id(),
      ]);
    }
    else {
      $this->logVerbose('Updated SQLite FTS5 index schema for @index', [
        '@index' => $index->id(),
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function removeIndex($index): void {
    $index_id = (string) (is_object($index) ? $index->id() : $index);

    $this->schemaManager->dropIndexTables($index_id);
    $this->connectionManager->deleteDatabase($index_id);

    $this->logVerbose('Removed SQLite FTS5 index @index', [
      '@index' => $index_id,
    ]);
  }

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The index to add items to.
   * @param \Drupal\search_api\Item\ItemInterface[] $items
   *   The items to index.
   *
   * @return array<int, string>
   *   The IDs of all items that were successfully indexed.
   */
  public function indexItems(IndexInterface $index, array $items): array {
    return $this->indexer->indexItems($index, $items, $this->configuration);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteItems(IndexInterface $index, array $item_ids): void {
    $this->indexer->deleteItems($index, $item_ids);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAllIndexItems(IndexInterface $index, $datasource_id = NULL): void {
    $this->indexer->deleteAllItems($index, $datasource_id);
  }

  /**
   * {@inheritdoc}
   *
   * @return \Drupal\search_api\Query\ResultSetInterface
   *   The search results.
   */
  public function search(QueryInterface $query): ResultSetInterface {
    $results = $query->getResults();
    $index = $query->getIndex();
    $index_id = (string) $index->id();

    // Check if tables exist.
    if (!$this->schemaManager->tablesExist($index_id)) {
      $this->getLogger()->warning('Index @index has no tables, returning empty results.', [
        '@index' => $index_id,
      ]);
      return $results;
    }

    try {
      // Build the fulltext field mapping.
      $fulltext_fields = $this->buildFulltextFieldMapping($index);

      // Build FTS5 MATCH query.
      $matching_mode = MatchingMode::tryFrom($this->configuration['matching'] ?? '') ?? MatchingMode::Words;
      $match_query = $this->queryBuilder->buildMatchQuery($query, $fulltext_fields, $matching_mode);

      // Get query range options.
      $limit = $query->getOption('limit', 10);
      $offset = $query->getOption('offset', 0);

      // Execute search - get all matching item IDs first, then filter.
      $item_ids = [];
      $scores = [];

      if ($match_query) {
        // Fulltext search - get ALL matching items first (before filtering).
        $fts_table = $this->schemaManager->getFtsTableName($index_id);
        $busy_timeout = $this->configuration['concurrency']['busy_timeout'] ?? 10000;

        // Get all FTS results (no pagination yet - we need to filter first).
        $fts_results = $this->fts5QueryRunner->search(
          $index_id,
          $fts_table,
          $match_query,
          [
            'limit' => 10000,
            'offset' => 0,
            'order_by_rank' => TRUE,
            'busy_timeout' => $busy_timeout,
          ]
        );

        foreach ($fts_results as $row) {
          $item_ids[] = $row['item_id'];
          $scores[$row['item_id']] = abs((float) $row['bm25_score']);
        }
      }
      else {
        // No fulltext search - get all item IDs.
        $connection = $this->connectionManager->getConnection($index_id);
        $items_table = $this->schemaManager->getItemsTableName($index_id);

        $query_builder = $connection->select($items_table, 'i')
          ->fields('i', ['item_id']);

        $result = $query_builder->execute();
        $item_ids = $result !== NULL ? $result->fetchCol() : [];
      }

      // Apply condition filters to get the filtered set of items.
      $filtered_ids = $this->conditionHandler->applyConditions($query, $index_id, $item_ids, $this->configuration);

      // Set result count based on filtered results.
      $total_count = count($filtered_ids);
      $results->setResultCount($total_count);

      // Apply pagination to the filtered results.
      $paginated_ids = array_slice($filtered_ids, $offset, $limit > 0 ? $limit : NULL);

      // Create result items.
      $fields_helper = $this->getFieldsHelper();
      foreach ($paginated_ids as $item_id) {
        $result_item = $fields_helper->createItem($index, $item_id);
        if (isset($scores[$item_id])) {
          $result_item->setScore($scores[$item_id]);
        }

        $results->addResultItem($result_item);
      }

      // Add highlighting if enabled and we have a match query.
      if ($paginated_ids !== [] && $match_query && ($this->configuration['highlighting']['enabled'] ?? FALSE)) {
        $this->addHighlighting($results, $index_id, $match_query, $paginated_ids, array_values($fulltext_fields));
      }

      // Process facets if requested - use ALL filtered IDs for accurate counts.
      $facet_options = $query->getOption('search_api_facets');
      if (!empty($facet_options) && $filtered_ids !== []) {
        $this->addFacets($results, $index_id, $filtered_ids, $facet_options);
      }

    }
    catch (\Exception $exception) {
      $this->getLogger()->error('Search failed for index @index: @error', [
        '@index' => $index_id,
        '@error' => $exception->getMessage(),
      ]);
      throw new SearchApiException('Search failed: ' . $exception->getMessage(), 0, $exception);
    }

    return $results;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getSupportedFeatures(): array {
    return [
      'search_api_facets',
      'search_api_facets_operator_or',
      'search_api_highlighting',
      'search_api_autocomplete',
    ];
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function supportsDataType($type): bool {
    return in_array($type, [
      'text',
      'string',
      'integer',
      'decimal',
      'date',
      'boolean',
      'uri',
    ], TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function getIndexedItemsCount(IndexInterface $index): int {
    return $this->indexer->getIndexedItemsCount($index);
  }

  /**
   * Builds the fulltext field mapping for an index.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The index.
   *
   * @return array<string, string>
   *   Map of field IDs to sanitized column names.
   */
  private function buildFulltextFieldMapping(IndexInterface $index): array {
    $mapping = [];
    $fields = $index->getFields();

    foreach ($fields as $field_id => $field) {
      if ($this->fieldTypeMapper->isFulltextType($field->getType())) {
        $mapping[$field_id] = ColumnNameHelper::sanitize($field_id);
      }
    }

    return $mapping;
  }

  /**
   * Adds highlighting to search results.
   *
   * @param \Drupal\search_api\Query\ResultSetInterface $results
   *   The result set.
   * @param string $index_id
   *   The index ID.
   * @param string $match_query
   *   The FTS5 match query.
   * @param array<int, string> $item_ids
   *   The item IDs.
   * @param array<int, string> $columns
   *   The columns to highlight.
   */
  private function addHighlighting(
    ResultSetInterface $results,
    string $index_id,
    string $match_query,
    array $item_ids,
    array $columns,
  ): void {
    $options = [
      'prefix' => $this->configuration['highlighting']['prefix'],
      'suffix' => $this->configuration['highlighting']['suffix'],
      'excerpt_length' => $this->configuration['highlighting']['excerpt_length'],
    ];

    $excerpts = $this->highlighter->getExcerpts(
      $index_id,
      $match_query,
      $item_ids,
      $columns,
      $options
    );

    if ($excerpts === []) {
      return;
    }

    // Set excerpts on result items for Views integration.
    foreach ($results->getResultItems() as $item) {
      $item_id = $item->getId();
      if (isset($excerpts[$item_id])) {
        $item->setExcerpt(implode(' ... ', $excerpts[$item_id]));
      }
    }
  }

  /**
   * Adds facet data to search results.
   *
   * @param \Drupal\search_api\Query\ResultSetInterface $results
   *   The result set.
   * @param string $index_id
   *   The index ID.
   * @param array<int, string> $item_ids
   *   The item IDs from search results.
   * @param array<string, array<string, mixed>> $facet_options
   *   The facet options from the query.
   */
  private function addFacets(
    ResultSetInterface $results,
    string $index_id,
    array $item_ids,
    array $facet_options,
  ): void {
    $facet_results = [];

    foreach ($facet_options as $facet_id => $facet_info) {
      $field_name = $facet_info['field'];
      $options = [
        'limit' => $facet_info['limit'] ?? 10,
        'min_count' => $facet_info['min_count'] ?? 1,
        'missing' => $facet_info['missing'] ?? FALSE,
      ];

      $facet_values = $this->facetBuilder->calculateFacets(
        $index_id,
        $field_name,
        $item_ids,
        $options
      );

      // Convert to Search API facet format.
      $facet_results[$facet_id] = [];
      foreach ($facet_values as $facet_value) {
        $facet_results[$facet_id][] = [
          'count' => $facet_value['count'],
          'filter' => $facet_value['value'] !== NULL ? '"' . $facet_value['value'] . '"' : '!',
        ];
      }
    }

    $results->setExtraData('search_api_facets', $facet_results);
  }

  /**
   * {@inheritdoc}
   */
  public function getAutocompleteSuggestions(
    QueryInterface $query,
    SearchInterface $search,
    string $incomplete_key,
    string $user_input,
  ): array {
    $index = $query->getIndex();
    $fulltext_fields = $this->buildFulltextFieldMapping($index);

    return $this->autocompleteHandler->getSuggestions(
      $query,
      $search,
      $incomplete_key,
      $user_input,
      $fulltext_fields
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getVerboseLogger(): LoggerInterface {
    return $this->getLogger();
  }

  /**
   * {@inheritdoc}
   */
  protected function getVerboseLoggerConfig(): array {
    return $this->configuration;
  }

}
