<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;

/**
 * Builds facet queries and calculates facet counts.
 */
final readonly class FacetBuilder implements FacetBuilderInterface {

  /**
   * Maximum number of item IDs per query batch.
   *
   * SQLite has a default SQLITE_MAX_VARIABLE_NUMBER of 999.
   * We use 500 to stay well under the limit and improve query planning.
   */
  private const int BATCH_SIZE = 500;

  /**
   * Minimum number of facets to trigger temp table optimization.
   *
   * When calculating multiple facets, using a temp table with a single query
   * is more efficient than running separate queries for each facet.
   */
  private const int OPTIMIZE_MULTI_FACET_THRESHOLD = 3;

  /**
   * Constructs a FacetBuilder instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The connection manager.
   * @param \Drupal\search_api_sqlite\Database\SchemaManagerInterface $schemaManager
   *   The schema manager.
   */
  public function __construct(
    private ConnectionManagerInterface $connectionManager,
    private SchemaManagerInterface $schemaManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function calculateFacets(
    string $index_id,
    string $field_name,
    array $item_ids,
    array $options = [],
  ): array {
    if ($item_ids === []) {
      return [];
    }

    // Batch large item ID lists to avoid SQLite limits and improve performance.
    if (count($item_ids) > self::BATCH_SIZE) {
      return $this->calculateFacetsBatched($index_id, $field_name, $item_ids, $options);
    }

    return $this->executeFacetQuery($index_id, $field_name, $item_ids, $options);
  }

  /**
   * Calculates facets for large result sets by batching queries.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $field_name
   *   The field name to calculate facets for.
   * @param array<int, string> $item_ids
   *   The item IDs to calculate facets for.
   * @param array<string, mixed> $options
   *   Options for the facet calculation.
   *
   * @return array<int, array{value: mixed, count: int}>
   *   Array of facet values with counts.
   */
  private function calculateFacetsBatched(
    string $index_id,
    string $field_name,
    array $item_ids,
    array $options,
  ): array {
    $batches = array_chunk($item_ids, self::BATCH_SIZE);
    $aggregated = [];

    // Execute queries for each batch and aggregate results.
    foreach ($batches as $batch) {
      $batch_results = $this->executeFacetQuery(
        $index_id,
        $field_name,
        $batch,
        [
          'limit' => 0,
          'min_count' => 1,
        ]
      );

      foreach ($batch_results as $facet) {
        $value = $facet['value'];
        $aggregated[$value] = ($aggregated[$value] ?? 0) + $facet['count'];
      }
    }

    // Apply limit and min_count after aggregation.
    $limit = $options['limit'] ?? 10;
    $min_count = $options['min_count'] ?? 1;

    // Sort by count descending.
    arsort($aggregated);

    $results = [];
    foreach ($aggregated as $value => $count) {
      if ($count < $min_count) {
        continue;
      }

      $results[] = ['value' => $value, 'count' => $count];
      if ($limit > 0 && count($results) >= $limit) {
        break;
      }
    }

    return $results;
  }

  /**
   * Executes a single facet query for a set of item IDs.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $field_name
   *   The field name to calculate facets for.
   * @param array<int, string> $item_ids
   *   The item IDs to calculate facets for.
   * @param array<string, mixed> $options
   *   Options for the facet calculation.
   *
   * @return array<int, array{value: mixed, count: int}>
   *   Array of facet values with counts.
   */
  private function executeFacetQuery(
    string $index_id,
    string $field_name,
    array $item_ids,
    array $options,
  ): array {
    $limit = $options['limit'] ?? 10;
    $min_count = $options['min_count'] ?? 1;

    $connection = $this->connectionManager->getConnection($index_id);
    $field_data_table = $this->schemaManager->getFieldDataTableName($index_id);

    // Build query to get facet counts.
    $query = $connection->select($field_data_table, 'fd');
    $query->addExpression('COALESCE(fd.value_string, CAST(fd.value_integer AS CHAR), CAST(fd.value_decimal AS CHAR))', 'value');
    $query->addExpression('COUNT(DISTINCT fd.item_id)', 'count');
    $query->condition('fd.field_name', $field_name);
    $query->condition('fd.item_id', $item_ids, 'IN');
    $query->groupBy('value');
    $query->having('count >= :min', [':min' => $min_count]);
    $query->orderBy('count', 'DESC');

    if ($limit > 0) {
      $query->range(0, $limit);
    }

    $result = $query->execute();
    if ($result === NULL) {
      return [];
    }

    $rows = $result->fetchAll();

    $facets = [];
    foreach ($rows as $row) {
      $facets[] = [
        'value' => $row->value,
        'count' => (int) $row->count,
      ];
    }

    return $facets;
  }

  /**
   * {@inheritdoc}
   */
  public function calculateMultipleFacets(
    string $index_id,
    array $field_options,
    array $item_ids,
  ): array {
    if ($item_ids === []) {
      // Return empty arrays for each requested facet.
      return array_fill_keys(array_keys($field_options), []);
    }

    // Use optimized temp table approach when:
    // - Multiple facets are requested (reduces query count significantly)
    // - Or item count exceeds batch size (avoids batched queries per facet)
    $should_optimize = count($field_options) >= self::OPTIMIZE_MULTI_FACET_THRESHOLD
      || count($item_ids) > self::BATCH_SIZE;

    if ($should_optimize) {
      return $this->calculateFacetsWithTempTable($index_id, $field_options, $item_ids);
    }

    // Fall back to individual queries for small numbers of facets.
    $results = [];
    foreach ($field_options as $field_name => $options) {
      $results[$field_name] = $this->calculateFacets(
        $index_id,
        $field_name,
        $item_ids,
        $options
      );
    }

    return $results;
  }

  /**
   * Calculates multiple facets using a temp table for efficiency.
   *
   * This method creates a temporary table with the search result item IDs,
   * then executes a single query to calculate all facets at once using a JOIN.
   * This is significantly faster than running separate queries for each facet.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param array<string, array<string, mixed>> $field_options
   *   Array of field options keyed by field name.
   * @param array<int, string> $item_ids
   *   The item IDs to calculate facets for.
   *
   * @return array<string, array<int, array{value: mixed, count: int}>>
   *   Array of facet results keyed by field name.
   */
  private function calculateFacetsWithTempTable(
    string $index_id,
    array $field_options,
    array $item_ids,
  ): array {
    $pdo = $this->connectionManager->getPdo($index_id);
    $field_data_table = $this->schemaManager->getFieldDataTableName($index_id);

    // Create and populate temp table with search result item IDs.
    $temp_table = $this->createTempResultsTable($pdo, $index_id, $item_ids);

    // Build single query to get all facet counts.
    $field_names = array_keys($field_options);
    $placeholders = implode(', ', array_fill(0, count($field_names), '?'));

    $sql = sprintf(
      'SELECT
        fd.field_name,
        COALESCE(fd.value_string, CAST(fd.value_integer AS TEXT), CAST(fd.value_decimal AS TEXT)) AS value,
        COUNT(DISTINCT fd.item_id) AS count
      FROM %s fd
      INNER JOIN %s sr ON fd.item_id = sr.item_id
      WHERE fd.field_name IN (%s)
      GROUP BY fd.field_name, value
      ORDER BY fd.field_name, count DESC',
      $field_data_table,
      $temp_table,
      $placeholders
    );

    $stmt = $pdo->prepare($sql);
    $stmt->execute($field_names);

    $rows = $stmt->fetchAll(\PDO::FETCH_OBJ);

    // Clean up temp table.
    $pdo->exec(sprintf('DROP TABLE IF EXISTS %s', $temp_table));

    // Group results by field and apply per-facet limits.
    return $this->groupAndFilterResults($rows, $field_options);
  }

  /**
   * Creates a temporary table and populates it with item IDs.
   *
   * @param \PDO $pdo
   *   The PDO connection.
   * @param string $index_id
   *   The Search API index ID.
   * @param array<int, string> $item_ids
   *   The item IDs to insert.
   *
   * @return string
   *   The name of the created temp table.
   */
  private function createTempResultsTable(\PDO $pdo, string $index_id, array $item_ids): string {
    // Use index_id in table name to avoid conflicts.
    $temp_table = 'temp_facet_results_' . preg_replace('/[^a-z0-9_]/', '_', $index_id);

    // Create temp table (will be auto-dropped when connection closes).
    $pdo->exec(sprintf(
      'CREATE TEMP TABLE IF NOT EXISTS %s (item_id TEXT PRIMARY KEY)',
      $temp_table
    ));

    // Clear any existing data (in case of reuse within same connection).
    $pdo->exec(sprintf('DELETE FROM %s', $temp_table));

    // Batch insert item IDs for efficiency.
    $batch_size = 100;
    $batches = array_chunk($item_ids, $batch_size);

    foreach ($batches as $batch) {
      $placeholders = implode(', ', array_fill(0, count($batch), '(?)'));
      $sql = sprintf('INSERT OR IGNORE INTO %s (item_id) VALUES %s', $temp_table, $placeholders);
      $stmt = $pdo->prepare($sql);
      $stmt->execute($batch);
    }

    return $temp_table;
  }

  /**
   * Groups raw facet results by field and applies per-facet limits.
   *
   * @param array<int, object> $rows
   *   Raw result rows with field_name, value, and count properties.
   * @param array<string, array<string, mixed>> $field_options
   *   Array of field options keyed by field name.
   *
   * @return array<string, array<int, array{value: mixed, count: int}>>
   *   Array of facet results keyed by field name.
   */
  private function groupAndFilterResults(array $rows, array $field_options): array {
    // Initialize results with empty arrays for all requested fields.
    $results = array_fill_keys(array_keys($field_options), []);

    // Group rows by field name.
    $grouped = [];
    foreach ($rows as $row) {
      $field_name = $row->field_name;
      if (!isset($grouped[$field_name])) {
        $grouped[$field_name] = [];
      }

      $grouped[$field_name][] = [
        'value' => $row->value,
        'count' => (int) $row->count,
      ];
    }

    // Apply per-facet limit and min_count.
    foreach ($field_options as $field_name => $options) {
      if (!isset($grouped[$field_name])) {
        continue;
      }

      $limit = $options['limit'] ?? 10;
      $min_count = $options['min_count'] ?? 1;

      $filtered = [];
      foreach ($grouped[$field_name] as $facet) {
        if ($facet['count'] < $min_count) {
          continue;
        }

        $filtered[] = $facet;
        if ($limit > 0 && count($filtered) >= $limit) {
          break;
        }
      }

      $results[$field_name] = $filtered;
    }

    return $results;
  }

}
