<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Database;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\File\FileSystemInterface;
use Drupal\search_api_sqlite\Database\ConnectionManager;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Log\LoggerInterface;

/**
 * Tests the ConnectionManager service.
 */
#[CoversClass(ConnectionManager::class)]
#[Group('search_api_sqlite')]
final class ConnectionManagerTest extends UnitTestCase {

  /**
   * The mocked file system service.
   */
  private FileSystemInterface&MockObject $fileSystem;

  /**
   * The mocked config factory.
   */
  private ConfigFactoryInterface&MockObject $configFactory;

  /**
   * The mocked logger.
   */
  private LoggerInterface&MockObject $logger;

  /**
   * The mocked config object.
   */
  private ImmutableConfig&MockObject $config;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->fileSystem = $this->createMock(FileSystemInterface::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->logger = $this->createMock(LoggerInterface::class);
    $this->config = $this->createMock(ImmutableConfig::class);

    $this->configFactory->method('get')
      ->with('search_api_sqlite.settings')
      ->willReturn($this->config);
  }

  /**
   * Creates a ConnectionManager instance for testing.
   *
   * @return \Drupal\search_api_sqlite\Database\ConnectionManager
   *   The connection manager.
   */
  private function createConnectionManager(): ConnectionManager {
    return new ConnectionManager(
      $this->fileSystem,
      $this->configFactory,
      $this->logger
    );
  }

  /**
   * Tests getDatabasePath() returns correct path with default config.
   */
  public function testGetDatabasePathWithDefaultConfig(): void {
    $this->config->method('get')
      ->with('database_path')
      ->willReturn(NULL);

    $this->fileSystem->method('realpath')
      ->willReturnMap([
        ['private://search_api_sqlite', FALSE],
        ['private://', '/var/private'],
      ]);

    $connectionManager = $this->createConnectionManager();
    $path = $connectionManager->getDatabasePath('test_index');

    $this->assertSame('/var/private/search_api_sqlite/test_index.sqlite', $path);
  }

  /**
   * Tests getDatabasePath() returns correct path with custom config.
   */
  public function testGetDatabasePathWithCustomConfig(): void {
    $this->config->method('get')
      ->with('database_path')
      ->willReturn('/custom/path');

    $this->fileSystem->method('realpath')
      ->with('/custom/path')
      ->willReturn('/custom/path');

    $connectionManager = $this->createConnectionManager();
    $path = $connectionManager->getDatabasePath('my_index');

    $this->assertSame('/custom/path/my_index.sqlite', $path);
  }

  /**
   * Tests getDatabasePath() throws exception when private path not configured.
   */
  public function testGetDatabasePathThrowsWhenPrivateNotConfigured(): void {
    $this->config->method('get')
      ->with('database_path')
      ->willReturn(NULL);

    $this->fileSystem->method('realpath')
      ->willReturn(FALSE);

    $connectionManager = $this->createConnectionManager();

    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage('Private file system is not configured');

    $connectionManager->getDatabasePath('test_index');
  }

  /**
   * Tests databaseExists() returns true when file exists.
   */
  public function testDatabaseExistsReturnsTrueWhenFileExists(): void {
    $this->config->method('get')
      ->with('database_path')
      ->willReturn(NULL);

    // Use a real temporary directory for this test.
    $tempDir = sys_get_temp_dir() . '/search_api_sqlite_test_' . uniqid();
    mkdir($tempDir, 0755, TRUE);
    $dbPath = $tempDir . '/test.sqlite';
    touch($dbPath);

    $this->fileSystem->method('realpath')
      ->willReturnMap([
        ['private://search_api_sqlite', $tempDir],
        ['private://', dirname($tempDir)],
      ]);

    $connectionManager = $this->createConnectionManager();
    $exists = $connectionManager->databaseExists('test');

    $this->assertTrue($exists);

    // Cleanup.
    unlink($dbPath);
    rmdir($tempDir);
  }

  /**
   * Tests databaseExists() returns false when file does not exist.
   */
  public function testDatabaseExistsReturnsFalseWhenFileNotExists(): void {
    $this->config->method('get')
      ->with('database_path')
      ->willReturn('/nonexistent/path');

    $this->fileSystem->method('realpath')
      ->with('/nonexistent/path')
      ->willReturn('/nonexistent/path');

    $connectionManager = $this->createConnectionManager();
    $exists = $connectionManager->databaseExists('test_index');

    $this->assertFalse($exists);
  }

  /**
   * Tests deleteDatabase() returns true and cleans up files.
   */
  public function testDeleteDatabaseRemovesAllFiles(): void {
    // Create temporary test files.
    $tempDir = sys_get_temp_dir() . '/search_api_sqlite_test_' . uniqid();
    mkdir($tempDir, 0755, TRUE);
    $dbPath = $tempDir . '/myindex.sqlite';
    $walPath = $dbPath . '-wal';
    $shmPath = $dbPath . '-shm';
    touch($dbPath);
    touch($walPath);
    touch($shmPath);

    $this->config->method('get')
      ->with('database_path')
      ->willReturn($tempDir);

    $this->fileSystem->method('realpath')
      ->with($tempDir)
      ->willReturn($tempDir);

    $this->logger->expects($this->once())
      ->method('info')
      ->with(
        'Deleted SQLite database for index @index',
        ['@index' => 'myindex']
      );

    $connectionManager = $this->createConnectionManager();
    $result = $connectionManager->deleteDatabase('myindex');

    $this->assertTrue($result);
    $this->assertFileDoesNotExist($dbPath);
    $this->assertFileDoesNotExist($walPath);
    $this->assertFileDoesNotExist($shmPath);

    // Cleanup.
    rmdir($tempDir);
  }

  /**
   * Tests deleteDatabase() returns true when no files exist.
   */
  public function testDeleteDatabaseReturnsTrueWhenNoFiles(): void {
    $this->config->method('get')
      ->with('database_path')
      ->willReturn('/tmp/nonexistent');

    $this->fileSystem->method('realpath')
      ->with('/tmp/nonexistent')
      ->willReturn('/tmp/nonexistent');

    // Logger should still log the deletion.
    $this->logger->expects($this->once())
      ->method('info');

    $connectionManager = $this->createConnectionManager();
    $result = $connectionManager->deleteDatabase('nonexistent_index');

    $this->assertTrue($result);
  }

  /**
   * Tests closeConnection() clears PDO cache.
   */
  public function testCloseConnectionClearsPdoCache(): void {
    // Create a temp directory and database.
    $tempDir = sys_get_temp_dir() . '/search_api_sqlite_test_' . uniqid();
    mkdir($tempDir, 0755, TRUE);

    $this->config->method('get')
      ->with('database_path')
      ->willReturn($tempDir);

    $this->fileSystem->method('realpath')
      ->willReturnMap([
        [$tempDir, $tempDir],
        ['private://', dirname($tempDir)],
      ]);

    $connectionManager = $this->createConnectionManager();

    // Get PDO to cache it.
    $pdo1 = $connectionManager->getPdo('cache_test');

    // Close should clear the cache.
    $connectionManager->closeConnection('cache_test');

    // Getting PDO again should create a new connection.
    $pdo2 = $connectionManager->getPdo('cache_test');

    // They should be different objects (new connection created).
    $this->assertNotSame($pdo1, $pdo2);

    // Cleanup.
    $connectionManager->deleteDatabase('cache_test');
    rmdir($tempDir);
  }

  /**
   * Tests getPdo() returns same cached connection on subsequent calls.
   */
  public function testGetPdoReturnsCachedConnection(): void {
    $tempDir = sys_get_temp_dir() . '/search_api_sqlite_test_' . uniqid();
    mkdir($tempDir, 0755, TRUE);

    $this->config->method('get')
      ->with('database_path')
      ->willReturn($tempDir);

    $this->fileSystem->method('realpath')
      ->willReturnMap([
        [$tempDir, $tempDir],
        ['private://', dirname($tempDir)],
      ]);

    $connectionManager = $this->createConnectionManager();

    $pdo1 = $connectionManager->getPdo('cached_test');
    $pdo2 = $connectionManager->getPdo('cached_test');

    $this->assertSame($pdo1, $pdo2);

    // Cleanup.
    $connectionManager->deleteDatabase('cached_test');
    rmdir($tempDir);
  }

  /**
   * Tests getPdo() creates different connections for different indexes.
   */
  public function testGetPdoCreatesSeparateConnectionsPerIndex(): void {
    $tempDir = sys_get_temp_dir() . '/search_api_sqlite_test_' . uniqid();
    mkdir($tempDir, 0755, TRUE);

    $this->config->method('get')
      ->with('database_path')
      ->willReturn($tempDir);

    $this->fileSystem->method('realpath')
      ->willReturnMap([
        [$tempDir, $tempDir],
        ['private://', dirname($tempDir)],
      ]);

    $connectionManager = $this->createConnectionManager();

    $pdo1 = $connectionManager->getPdo('index_a');
    $pdo2 = $connectionManager->getPdo('index_b');

    $this->assertNotSame($pdo1, $pdo2);

    // Cleanup.
    $connectionManager->deleteDatabase('index_a');
    $connectionManager->deleteDatabase('index_b');
    rmdir($tempDir);
  }

}
