<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Database;

use PHPUnit\Framework\MockObject\MockObject;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\Fts5QueryRunner;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the Fts5QueryRunner service.
 */
#[CoversClass(Fts5QueryRunner::class)]
#[Group('search_api_sqlite')]
final class Fts5QueryRunnerTest extends UnitTestCase {

  /**
   * The FTS5 query runner under test.
   */
  private Fts5QueryRunner $queryRunner;

  /**
   * The mocked connection manager.
   */
  private MockObject $connectionManager;

  /**
   * The mocked PDO connection.
   */
  private MockObject $pdo;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->connectionManager = $this->createMock(ConnectionManagerInterface::class);
    $this->pdo = $this->createMock(\PDO::class);

    // Mock both getPdo (for write operations) and getReadOnlyPdo (for reads).
    $this->connectionManager->method('getPdo')
      ->willReturn($this->pdo);
    $this->connectionManager->method('getReadOnlyPdo')
      ->willReturn($this->pdo);

    $this->queryRunner = new Fts5QueryRunner($this->connectionManager);
  }

  /**
   * Tests search() returns expected results.
   */
  public function testSearchReturnsResults(): void {
    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->expects($this->exactly(3))
      ->method('bindValue');
    $stmt->expects($this->once())
      ->method('execute');
    $stmt->method('fetchAll')
      ->willReturn([
        ['item_id' => 'item_1', 'bm25_score' => -2.5],
        ['item_id' => 'item_2', 'bm25_score' => -1.8],
      ]);

    $this->pdo->method('prepare')
      ->with($this->callback(fn(string $sql): bool => str_contains($sql, 'SELECT item_id, bm25')
        && str_contains($sql, 'MATCH :query')
        && str_contains($sql, 'ORDER BY bm25_score')
        && str_contains($sql, 'LIMIT')))
      ->willReturn($stmt);

    $results = $this->queryRunner->search(
      'test_index',
      'test_index_fts',
      'search term',
      ['limit' => 10, 'offset' => 0]
    );

    $this->assertCount(2, $results);
    $this->assertSame('item_1', $results[0]['item_id']);
    $this->assertSame('item_2', $results[1]['item_id']);
  }

  /**
   * Tests search() respects limit and offset.
   */
  public function testSearchRespectsLimitAndOffset(): void {
    $stmt = $this->createMock(\PDOStatement::class);

    $boundParams = [];
    $stmt->method('bindValue')
      ->willReturnCallback(function ($param, $value) use (&$boundParams): true {
        $boundParams[$param] = $value;
        return TRUE;
      });

    $stmt->method('execute')->willReturn(TRUE);
    $stmt->method('fetchAll')->willReturn([]);

    $this->pdo->method('prepare')->willReturn($stmt);

    $this->queryRunner->search(
      'test_index',
      'test_fts',
      'query',
      ['limit' => 25, 'offset' => 50]
    );

    $this->assertSame(25, $boundParams[':limit']);
    $this->assertSame(50, $boundParams[':offset']);
  }

  /**
   * Tests highlight() returns empty array for empty item IDs.
   */
  public function testHighlightReturnsEmptyForEmptyItemIds(): void {
    $results = $this->queryRunner->highlight(
      'test_index',
      'test_fts',
      'query',
      'title',
      []
    );

    $this->assertSame([], $results);
  }

  /**
   * Tests highlight() returns highlighted content.
   */
  public function testHighlightReturnsHighlightedContent(): void {
    // Mock PRAGMA table_info query.
    $pragmaResult = $this->createMock(\PDOStatement::class);
    $pragmaResult->method('fetch')
      ->willReturnOnConsecutiveCalls(
        ['name' => 'item_id'],
        ['name' => 'title'],
        ['name' => 'body'],
        FALSE
      );

    $this->pdo->method('query')
      ->willReturn($pragmaResult);

    // Mock highlight query.
    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('bindValue')->willReturn(TRUE);
    $stmt->method('execute')->willReturn(TRUE);
    $stmt->method('fetch')
      ->willReturnOnConsecutiveCalls(
        ['item_id' => 'item_1', 'highlighted' => '<mark>test</mark> content'],
        FALSE
      );

    $this->pdo->method('prepare')
      ->with($this->stringContains('highlight'))
      ->willReturn($stmt);

    $results = $this->queryRunner->highlight(
      'test_index',
      'test_fts',
      'test',
      'title',
      ['item_1'],
      '<mark>',
      '</mark>'
    );

    $this->assertArrayHasKey('item_1', $results);
    $this->assertSame('<mark>test</mark> content', $results['item_1']);
  }

  /**
   * Tests snippet() returns empty array for empty item IDs.
   */
  public function testSnippetReturnsEmptyForEmptyItemIds(): void {
    $results = $this->queryRunner->snippet(
      'test_index',
      'test_fts',
      'query',
      'title',
      []
    );

    $this->assertSame([], $results);
  }

  /**
   * Tests snippet() returns snippets with ellipsis.
   */
  public function testSnippetReturnsSnippets(): void {
    // Mock PRAGMA table_info query.
    $pragmaResult = $this->createMock(\PDOStatement::class);
    $pragmaResult->method('fetch')
      ->willReturnOnConsecutiveCalls(
        ['name' => 'item_id'],
        ['name' => 'body'],
        FALSE
      );

    $this->pdo->method('query')
      ->willReturn($pragmaResult);

    // Mock snippet query.
    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('bindValue')->willReturn(TRUE);
    $stmt->method('execute')->willReturn(TRUE);
    $stmt->method('fetch')
      ->willReturnOnConsecutiveCalls(
        ['item_id' => 'item_1', 'snippet' => '...some <mark>text</mark> here...'],
        FALSE
      );

    $this->pdo->method('prepare')
      ->with($this->stringContains('snippet'))
      ->willReturn($stmt);

    $results = $this->queryRunner->snippet(
      'test_index',
      'test_fts',
      'text',
      'body',
      ['item_1'],
      '<mark>',
      '</mark>',
      64
    );

    $this->assertArrayHasKey('item_1', $results);
    $this->assertStringContainsString('<mark>text</mark>', $results['item_1']);
  }

  /**
   * Tests getDocumentCount() returns correct count.
   */
  public function testGetDocumentCountReturnsCorrectCount(): void {
    $result = $this->createMock(\PDOStatement::class);
    $result->method('fetchColumn')
      ->willReturn('42');

    $this->pdo->method('query')
      ->with($this->stringContains('COUNT(*)'))
      ->willReturn($result);

    $count = $this->queryRunner->getDocumentCount('test_index', 'test_fts');

    $this->assertSame(42, $count);
  }

  /**
   * Tests getDocumentCount() returns zero on query failure.
   */
  public function testGetDocumentCountReturnsZeroOnFailure(): void {
    $this->pdo->method('query')
      ->willReturn(FALSE);

    $count = $this->queryRunner->getDocumentCount('test_index', 'test_fts');

    $this->assertSame(0, $count);
  }

  /**
   * Tests getMatchCount() returns correct count for matching query.
   */
  public function testGetMatchCountReturnsCorrectCount(): void {
    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('bindValue')->willReturn(TRUE);
    $stmt->method('execute')->willReturn(TRUE);
    $stmt->method('fetchColumn')->willReturn('15');

    $this->pdo->method('prepare')
      ->with($this->stringContains('COUNT(*)'))
      ->willReturn($stmt);

    $count = $this->queryRunner->getMatchCount('test_index', 'test_fts', 'search term');

    $this->assertSame(15, $count);
  }

  /**
   * Tests optimize() executes FTS5 optimize command.
   */
  public function testOptimizeExecutesFts5Command(): void {
    $this->pdo->expects($this->once())
      ->method('exec')
      ->with($this->callback(fn(string $sql): bool => str_contains($sql, "INSERT INTO test_fts(test_fts) VALUES('optimize')")));

    $this->queryRunner->optimize('test_index', 'test_fts');
  }

  /**
   * Tests rebuild() executes FTS5 rebuild command.
   */
  public function testRebuildExecutesFts5Command(): void {
    $this->pdo->expects($this->once())
      ->method('exec')
      ->with($this->callback(fn(string $sql): bool => str_contains($sql, "INSERT INTO test_fts(test_fts) VALUES('rebuild')")));

    $this->queryRunner->rebuild('test_index', 'test_fts');
  }

}
