<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Search;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Database\StatementInterface;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Item\FieldInterface;
use Drupal\search_api\Query\ConditionGroupInterface;
use Drupal\search_api\Query\ConditionInterface;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Index\FieldTypeMapperInterface;
use Drupal\search_api_sqlite\Search\ConditionHandler;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use Psr\Log\LoggerInterface;

/**
 * Tests the ConditionHandler service.
 */
#[CoversClass(ConditionHandler::class)]
#[Group('search_api_sqlite')]
final class ConditionHandlerTest extends UnitTestCase {

  /**
   * The condition handler under test.
   */
  private ConditionHandler $handler;

  /**
   * The mocked connection manager.
   */
  private ConnectionManagerInterface $connectionManager;

  /**
   * The mocked schema manager.
   */
  private SchemaManagerInterface $schemaManager;

  /**
   * The mocked field type mapper.
   */
  private FieldTypeMapperInterface $fieldTypeMapper;

  /**
   * The mocked logger.
   */
  private LoggerInterface $logger;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->connectionManager = $this->createMock(ConnectionManagerInterface::class);
    $this->schemaManager = $this->createMock(SchemaManagerInterface::class);
    $this->fieldTypeMapper = $this->createMock(FieldTypeMapperInterface::class);
    $this->logger = $this->createMock(LoggerInterface::class);

    $this->handler = new ConditionHandler(
      $this->connectionManager,
      $this->schemaManager,
      $this->fieldTypeMapper,
      $this->logger,
    );
  }

  /**
   * Tests getValueColumnForField with a string field.
   */
  public function testGetValueColumnForFieldString(): void {
    $field = $this->createMock(FieldInterface::class);
    $field->method('getType')->willReturn('string');

    $index = $this->createMock(IndexInterface::class);
    $index->method('getField')->with('category')->willReturn($field);

    $this->fieldTypeMapper->method('getStorageType')
      ->with('string')
      ->willReturn('string');
    $this->fieldTypeMapper->method('getFieldDataColumn')
      ->with('string')
      ->willReturn('value_string');

    $column = $this->handler->getValueColumnForField($index, 'category');

    $this->assertSame('value_string', $column);
  }

  /**
   * Tests getValueColumnForField with an integer field.
   */
  public function testGetValueColumnForFieldInteger(): void {
    $field = $this->createMock(FieldInterface::class);
    $field->method('getType')->willReturn('integer');

    $index = $this->createMock(IndexInterface::class);
    $index->method('getField')->with('nid')->willReturn($field);

    $this->fieldTypeMapper->method('getStorageType')
      ->with('integer')
      ->willReturn('integer');
    $this->fieldTypeMapper->method('getFieldDataColumn')
      ->with('integer')
      ->willReturn('value_integer');

    $column = $this->handler->getValueColumnForField($index, 'nid');

    $this->assertSame('value_integer', $column);
  }

  /**
   * Tests getValueColumnForField with a boolean field.
   */
  public function testGetValueColumnForFieldBoolean(): void {
    $field = $this->createMock(FieldInterface::class);
    $field->method('getType')->willReturn('boolean');

    $index = $this->createMock(IndexInterface::class);
    $index->method('getField')->with('status')->willReturn($field);

    $this->fieldTypeMapper->method('getStorageType')
      ->with('boolean')
      ->willReturn('integer');
    $this->fieldTypeMapper->method('getFieldDataColumn')
      ->with('integer')
      ->willReturn('value_integer');

    $column = $this->handler->getValueColumnForField($index, 'status');

    $this->assertSame('value_integer', $column);
  }

  /**
   * Tests getValueColumnForField with an unknown field defaults to string.
   */
  public function testGetValueColumnForFieldUnknown(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('getField')->with('unknown_field')->willReturn(NULL);

    $this->fieldTypeMapper->expects($this->never())->method('getStorageType');
    $this->fieldTypeMapper->expects($this->never())->method('getFieldDataColumn');

    $column = $this->handler->getValueColumnForField($index, 'unknown_field');

    $this->assertSame('value_string', $column);
  }

  /**
   * Tests applyConditions returns original IDs when no conditions.
   */
  public function testApplyConditionsReturnsOriginalIdsWhenNoConditions(): void {
    $conditionGroup = $this->createMock(ConditionGroupInterface::class);
    $conditionGroup->method('getConditions')->willReturn([]);

    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $query = $this->createMock(QueryInterface::class);
    $query->method('getConditionGroup')->willReturn($conditionGroup);
    $query->method('getIndex')->willReturn($index);

    $item_ids = ['item:1', 'item:2', 'item:3'];

    $result = $this->handler->applyConditions($query, 'test_index', $item_ids);

    $this->assertSame($item_ids, $result);
  }

  /**
   * Tests applyConditions with a simple equals condition.
   */
  public function testApplyConditionsWithEqualsCondition(): void {
    $condition = $this->createMock(ConditionInterface::class);
    $condition->method('getField')->willReturn('category');
    $condition->method('getValue')->willReturn('news');
    $condition->method('getOperator')->willReturn('=');

    $conditionGroup = $this->createMock(ConditionGroupInterface::class);
    $conditionGroup->method('getConditions')->willReturn([$condition]);
    $conditionGroup->method('getConjunction')->willReturn('AND');

    $field = $this->createMock(FieldInterface::class);
    $field->method('getType')->willReturn('string');

    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');
    $index->method('getField')->with('category')->willReturn($field);

    $query = $this->createMock(QueryInterface::class);
    $query->method('getConditionGroup')->willReturn($conditionGroup);
    $query->method('getIndex')->willReturn($index);

    // Set up field type mapper.
    $this->fieldTypeMapper->method('getStorageType')->willReturn('string');
    $this->fieldTypeMapper->method('getFieldDataColumn')->willReturn('value_string');

    // Set up schema manager.
    $this->schemaManager->method('getFieldDataTableName')
      ->with('test_index')
      ->willReturn('field_data_test_index');

    // Set up database mocks.
    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchCol')->willReturn(['item:1', 'item:3']);

    $selectQuery = $this->createMock(SelectInterface::class);
    $selectQuery->method('fields')->willReturnSelf();
    $selectQuery->method('condition')->willReturnSelf();
    $selectQuery->method('execute')->willReturn($statement);

    $connection = $this->createMock(Connection::class);
    $connection->method('select')
      ->with('field_data_test_index', 'fd')
      ->willReturn($selectQuery);

    $this->connectionManager->method('getConnection')
      ->with('test_index')
      ->willReturn($connection);

    $item_ids = ['item:1', 'item:2', 'item:3'];

    $result = $this->handler->applyConditions($query, 'test_index', $item_ids);

    $this->assertSame(['item:1', 'item:3'], $result);
  }

  /**
   * Tests applyConditions with AND conjunction filters progressively.
   */
  public function testApplyConditionsWithAndConjunction(): void {
    $condition1 = $this->createMock(ConditionInterface::class);
    $condition1->method('getField')->willReturn('category');
    $condition1->method('getValue')->willReturn('news');
    $condition1->method('getOperator')->willReturn('=');

    $condition2 = $this->createMock(ConditionInterface::class);
    $condition2->method('getField')->willReturn('status');
    $condition2->method('getValue')->willReturn(1);
    $condition2->method('getOperator')->willReturn('=');

    $conditionGroup = $this->createMock(ConditionGroupInterface::class);
    $conditionGroup->method('getConditions')->willReturn([$condition1, $condition2]);
    $conditionGroup->method('getConjunction')->willReturn('AND');

    $stringField = $this->createMock(FieldInterface::class);
    $stringField->method('getType')->willReturn('string');

    $intField = $this->createMock(FieldInterface::class);
    $intField->method('getType')->willReturn('integer');

    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');
    $index->method('getField')->willReturnCallback(fn($name) => match ($name) {
      'category' => $stringField,
      'status' => $intField,
      default => NULL,
    });

    $query = $this->createMock(QueryInterface::class);
    $query->method('getConditionGroup')->willReturn($conditionGroup);
    $query->method('getIndex')->willReturn($index);

    // Set up field type mapper to return appropriate columns.
    $this->fieldTypeMapper->method('getStorageType')->willReturnCallback(
      fn($type): string => $type === 'string' ? 'string' : 'integer'
    );
    $this->fieldTypeMapper->method('getFieldDataColumn')->willReturnCallback(
      fn($type): string => $type === 'string' ? 'value_string' : 'value_integer'
    );

    $this->schemaManager->method('getFieldDataTableName')->willReturn('field_data_test_index');

    // First condition returns items 1, 2, 3.
    $statement1 = $this->createMock(StatementInterface::class);
    $statement1->method('fetchCol')->willReturn(['item:1', 'item:2', 'item:3']);

    // Second condition returns items 1, 3.
    $statement2 = $this->createMock(StatementInterface::class);
    $statement2->method('fetchCol')->willReturn(['item:1', 'item:3']);

    $selectQuery = $this->createMock(SelectInterface::class);
    $selectQuery->method('fields')->willReturnSelf();
    $selectQuery->method('condition')->willReturnSelf();
    $selectQuery->method('execute')->willReturnOnConsecutiveCalls($statement1, $statement2);

    $connection = $this->createMock(Connection::class);
    $connection->method('select')->willReturn($selectQuery);

    $this->connectionManager->method('getConnection')->willReturn($connection);

    $item_ids = ['item:1', 'item:2', 'item:3', 'item:4'];

    $result = $this->handler->applyConditions($query, 'test_index', $item_ids);

    // AND conjunction: intersection of [1,2,3] and [1,3] = [1,3].
    $this->assertSame(['item:1', 'item:3'], $result);
  }

  /**
   * Tests applyConditions with OR conjunction unions results.
   */
  public function testApplyConditionsWithOrConjunction(): void {
    $condition1 = $this->createMock(ConditionInterface::class);
    $condition1->method('getField')->willReturn('category');
    $condition1->method('getValue')->willReturn('news');
    $condition1->method('getOperator')->willReturn('=');

    $condition2 = $this->createMock(ConditionInterface::class);
    $condition2->method('getField')->willReturn('category');
    $condition2->method('getValue')->willReturn('blog');
    $condition2->method('getOperator')->willReturn('=');

    $conditionGroup = $this->createMock(ConditionGroupInterface::class);
    $conditionGroup->method('getConditions')->willReturn([$condition1, $condition2]);
    $conditionGroup->method('getConjunction')->willReturn('OR');

    $field = $this->createMock(FieldInterface::class);
    $field->method('getType')->willReturn('string');

    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');
    $index->method('getField')->willReturn($field);

    $query = $this->createMock(QueryInterface::class);
    $query->method('getConditionGroup')->willReturn($conditionGroup);
    $query->method('getIndex')->willReturn($index);

    $this->fieldTypeMapper->method('getStorageType')->willReturn('string');
    $this->fieldTypeMapper->method('getFieldDataColumn')->willReturn('value_string');
    $this->schemaManager->method('getFieldDataTableName')->willReturn('field_data_test_index');

    // First condition returns item 1.
    $statement1 = $this->createMock(StatementInterface::class);
    $statement1->method('fetchCol')->willReturn(['item:1']);

    // Second condition returns items 2, 3.
    $statement2 = $this->createMock(StatementInterface::class);
    $statement2->method('fetchCol')->willReturn(['item:2', 'item:3']);

    $selectQuery = $this->createMock(SelectInterface::class);
    $selectQuery->method('fields')->willReturnSelf();
    $selectQuery->method('condition')->willReturnSelf();
    $selectQuery->method('execute')->willReturnOnConsecutiveCalls($statement1, $statement2);

    $connection = $this->createMock(Connection::class);
    $connection->method('select')->willReturn($selectQuery);

    $this->connectionManager->method('getConnection')->willReturn($connection);

    $item_ids = ['item:1', 'item:2', 'item:3', 'item:4'];

    $result = $this->handler->applyConditions($query, 'test_index', $item_ids);

    // OR conjunction: union of [1] and [2,3] = [1,2,3].
    $this->assertCount(3, $result);
    $this->assertContains('item:1', $result);
    $this->assertContains('item:2', $result);
    $this->assertContains('item:3', $result);
  }

  /**
   * Tests applyConditions returns empty array when AND has no matches.
   */
  public function testApplyConditionsReturnsEmptyOnAndWithNoMatches(): void {
    $condition = $this->createMock(ConditionInterface::class);
    $condition->method('getField')->willReturn('category');
    $condition->method('getValue')->willReturn('nonexistent');
    $condition->method('getOperator')->willReturn('=');

    $conditionGroup = $this->createMock(ConditionGroupInterface::class);
    $conditionGroup->method('getConditions')->willReturn([$condition]);
    $conditionGroup->method('getConjunction')->willReturn('AND');

    $field = $this->createMock(FieldInterface::class);
    $field->method('getType')->willReturn('string');

    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');
    $index->method('getField')->willReturn($field);

    $query = $this->createMock(QueryInterface::class);
    $query->method('getConditionGroup')->willReturn($conditionGroup);
    $query->method('getIndex')->willReturn($index);

    $this->fieldTypeMapper->method('getStorageType')->willReturn('string');
    $this->fieldTypeMapper->method('getFieldDataColumn')->willReturn('value_string');
    $this->schemaManager->method('getFieldDataTableName')->willReturn('field_data_test_index');

    // Condition returns no matches.
    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchCol')->willReturn([]);

    $selectQuery = $this->createMock(SelectInterface::class);
    $selectQuery->method('fields')->willReturnSelf();
    $selectQuery->method('condition')->willReturnSelf();
    $selectQuery->method('execute')->willReturn($statement);

    $connection = $this->createMock(Connection::class);
    $connection->method('select')->willReturn($selectQuery);

    $this->connectionManager->method('getConnection')->willReturn($connection);

    $item_ids = ['item:1', 'item:2', 'item:3'];

    $result = $this->handler->applyConditions($query, 'test_index', $item_ids);

    $this->assertSame([], $result);
  }

  /**
   * Tests applyConditions with IN operator.
   */
  public function testApplyConditionsWithInOperator(): void {
    $condition = $this->createMock(ConditionInterface::class);
    $condition->method('getField')->willReturn('category');
    $condition->method('getValue')->willReturn(['news', 'blog']);
    $condition->method('getOperator')->willReturn('IN');

    $conditionGroup = $this->createMock(ConditionGroupInterface::class);
    $conditionGroup->method('getConditions')->willReturn([$condition]);
    $conditionGroup->method('getConjunction')->willReturn('AND');

    $field = $this->createMock(FieldInterface::class);
    $field->method('getType')->willReturn('string');

    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');
    $index->method('getField')->willReturn($field);

    $query = $this->createMock(QueryInterface::class);
    $query->method('getConditionGroup')->willReturn($conditionGroup);
    $query->method('getIndex')->willReturn($index);

    $this->fieldTypeMapper->method('getStorageType')->willReturn('string');
    $this->fieldTypeMapper->method('getFieldDataColumn')->willReturn('value_string');
    $this->schemaManager->method('getFieldDataTableName')->willReturn('field_data_test_index');

    $statement = $this->createMock(StatementInterface::class);
    $statement->method('fetchCol')->willReturn(['item:1', 'item:2']);

    $selectQuery = $this->createMock(SelectInterface::class);
    $selectQuery->method('fields')->willReturnSelf();
    $selectQuery->method('condition')->willReturnSelf();
    $selectQuery->method('execute')->willReturn($statement);

    $connection = $this->createMock(Connection::class);
    $connection->method('select')->willReturn($selectQuery);

    $this->connectionManager->method('getConnection')->willReturn($connection);

    $item_ids = ['item:1', 'item:2', 'item:3'];

    $result = $this->handler->applyConditions($query, 'test_index', $item_ids);

    $this->assertSame(['item:1', 'item:2'], $result);
  }

  /**
   * Tests applyConditions with empty item_ids returns early for AND.
   */
  public function testApplyConditionsWithEmptyItemIdsReturnsEarly(): void {
    $condition = $this->createMock(ConditionInterface::class);
    $condition->method('getField')->willReturn('category');
    $condition->method('getValue')->willReturn('news');
    $condition->method('getOperator')->willReturn('=');

    $conditionGroup = $this->createMock(ConditionGroupInterface::class);
    $conditionGroup->method('getConditions')->willReturn([$condition]);
    $conditionGroup->method('getConjunction')->willReturn('AND');

    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $query = $this->createMock(QueryInterface::class);
    $query->method('getConditionGroup')->willReturn($conditionGroup);
    $query->method('getIndex')->willReturn($index);

    // Connection should not be called when item_ids is empty.
    $this->connectionManager->expects($this->once())
      ->method('getConnection')
      ->willReturn($this->createMock(Connection::class));

    $this->schemaManager->method('getFieldDataTableName')->willReturn('field_data_test_index');

    $result = $this->handler->applyConditions($query, 'test_index', []);

    $this->assertSame([], $result);
  }

}
