<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Search;

use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_sqlite\Enum\MatchingMode;
use Drupal\search_api_sqlite\Search\QueryBuilder;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the QueryBuilder service.
 */
#[CoversClass(QueryBuilder::class)]
#[Group('search_api_sqlite')]
final class QueryBuilderTest extends UnitTestCase {

  /**
   * The query builder under test.
   */
  private QueryBuilder $queryBuilder;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();
    $this->queryBuilder = new QueryBuilder();
  }

  /**
   * Tests parseSearchInput() with various inputs.
   *
   * @param string $input
   *   The input string.
   * @param array<int, string> $expected_terms
   *   Expected terms.
   * @param array<int, string> $expected_phrases
   *   Expected phrases.
   */
  #[DataProvider('searchInputProvider')]
  public function testParseSearchInput(string $input, array $expected_terms, array $expected_phrases): void {
    $result = $this->queryBuilder->parseSearchInput($input);

    $this->assertSame($expected_terms, $result['terms']);
    $this->assertSame($expected_phrases, $result['phrases']);
  }

  /**
   * Data provider for search input parsing tests.
   *
   * @return array<array{string, array<string>, array<string>}>
   *   Test cases.
   */
  public static function searchInputProvider(): array {
    return [
      'simple terms' => [
        'hello world',
        ['hello', 'world'],
        [],
      ],
      'quoted phrase' => [
        '"hello world"',
        [],
        ['hello world'],
      ],
      'mixed terms and phrases' => [
        'foo "hello world" bar',
        ['foo', 'bar'],
        ['hello world'],
      ],
      'multiple phrases' => [
        '"phrase one" "phrase two"',
        [],
        ['phrase one', 'phrase two'],
      ],
      'empty input' => [
        '',
        [],
        [],
      ],
      'only whitespace' => [
        '   ',
        [],
        [],
      ],
      'short words filtered' => [
        'a b cd efg',
        ['cd', 'efg'],
        [],
      ],
    ];
  }

  /**
   * Tests escapeTerm() with various inputs.
   */
  #[DataProvider('escapeTermProvider')]
  public function testEscapeTerm(string $input, string $expected): void {
    $this->assertSame($expected, $this->queryBuilder->escapeTerm($input));
  }

  /**
   * Data provider for escape term tests.
   *
   * @return array<array{string, string}>
   *   Test cases.
   */
  public static function escapeTermProvider(): array {
    return [
      'simple term' => ['hello', 'hello'],
      'term with space' => ['hello world', '"hello world"'],
      'term with quotes' => ['hello"world', '"hello""world"'],
      'term with AND' => ['AND', '"AND"'],
      'term with OR' => ['OR', '"OR"'],
      'term with NOT' => ['NOT', '"NOT"'],
      'term with asterisk' => ['test*', '"test*"'],
      'term with dash' => ['test-case', '"test-case"'],
    ];
  }

  /**
   * Tests buildMatchQuery() returns NULL for empty keys.
   */
  public function testBuildMatchQueryReturnsNullForEmptyKeys(): void {
    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn([]);
    $query->method('getFulltextFields')->willReturn([]);

    $result = $this->queryBuilder->buildMatchQuery($query, ['title' => 'title']);

    $this->assertNull($result);
  }

  /**
   * Tests buildMatchQuery() with simple string keys.
   */
  public function testBuildMatchQueryWithSimpleKeys(): void {
    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn('hello world');
    $query->method('getFulltextFields')->willReturn(['title']);

    $fulltext_fields = ['title' => 'title', 'body' => 'body'];

    $result = $this->queryBuilder->buildMatchQuery($query, $fulltext_fields);

    $this->assertNotNull($result);
    $this->assertStringContainsString('hello', $result);
    $this->assertStringContainsString('world', $result);
    $this->assertStringContainsString('title', $result);
  }

  /**
   * Tests buildMatchQuery() with structured keys.
   */
  public function testBuildMatchQueryWithStructuredKeys(): void {
    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn([
      '#conjunction' => 'AND',
      0 => 'term1',
      1 => 'term2',
    ]);
    $query->method('getFulltextFields')->willReturn([]);

    $fulltext_fields = ['title' => 'title'];

    $result = $this->queryBuilder->buildMatchQuery($query, $fulltext_fields);

    $this->assertNotNull($result);
    $this->assertStringContainsString('term1', $result);
    $this->assertStringContainsString('term2', $result);
    $this->assertStringContainsString('AND', $result);
  }

  /**
   * Tests buildMatchQuery() with OR conjunction.
   */
  public function testBuildMatchQueryWithOrConjunction(): void {
    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn([
      '#conjunction' => 'OR',
      0 => 'term1',
      1 => 'term2',
    ]);
    $query->method('getFulltextFields')->willReturn([]);

    $fulltext_fields = ['title' => 'title'];

    $result = $this->queryBuilder->buildMatchQuery($query, $fulltext_fields);

    $this->assertNotNull($result);
    $this->assertStringContainsString('OR', $result);
  }

  /**
   * Tests buildMatchQuery() with prefix matching mode.
   */
  public function testBuildMatchQueryWithPrefixMatching(): void {
    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn('test');
    $query->method('getFulltextFields')->willReturn([]);

    $fulltext_fields = ['title' => 'title'];

    $result = $this->queryBuilder->buildMatchQuery(
      $query,
      $fulltext_fields,
      MatchingMode::Prefix
    );

    $this->assertNotNull($result);
    $this->assertStringContainsString('test*', $result);
  }

  /**
   * Tests buildMatchQuery() with phrase matching mode.
   */
  public function testBuildMatchQueryWithPhraseMatching(): void {
    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn('hello world');
    $query->method('getFulltextFields')->willReturn([]);

    $fulltext_fields = ['title' => 'title'];

    $result = $this->queryBuilder->buildMatchQuery(
      $query,
      $fulltext_fields,
      MatchingMode::Phrase
    );

    $this->assertNotNull($result);
    $this->assertStringContainsString('hello', $result);
    $this->assertStringContainsString('world', $result);
  }

  /**
   * Tests buildMatchQuery() returns NULL when no columns match.
   */
  public function testBuildMatchQueryReturnsNullWhenNoColumnsMatch(): void {
    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn('test');
    $query->method('getFulltextFields')->willReturn(['nonexistent_field']);

    $fulltext_fields = ['title' => 'title', 'body' => 'body'];

    $result = $this->queryBuilder->buildMatchQuery($query, $fulltext_fields);

    $this->assertNull($result);
  }

  /**
   * Tests buildMatchQuery() with quoted phrase in input.
   */
  public function testBuildMatchQueryWithQuotedPhrase(): void {
    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn('"exact phrase" additional');
    $query->method('getFulltextFields')->willReturn([]);

    $fulltext_fields = ['title' => 'title'];

    $result = $this->queryBuilder->buildMatchQuery($query, $fulltext_fields);

    $this->assertNotNull($result);
    $this->assertStringContainsString('exact phrase', $result);
    $this->assertStringContainsString('additional', $result);
  }

  /**
   * Tests buildMatchQuery() with nested structured keys.
   */
  public function testBuildMatchQueryWithNestedStructuredKeys(): void {
    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn([
      '#conjunction' => 'AND',
      0 => 'outer',
      1 => [
        '#conjunction' => 'OR',
        0 => 'inner1',
        1 => 'inner2',
      ],
    ]);
    $query->method('getFulltextFields')->willReturn([]);

    $fulltext_fields = ['title' => 'title'];

    $result = $this->queryBuilder->buildMatchQuery($query, $fulltext_fields);

    $this->assertNotNull($result);
    $this->assertStringContainsString('outer', $result);
    $this->assertStringContainsString('inner1', $result);
    $this->assertStringContainsString('inner2', $result);
  }

  /**
   * Tests buildMatchQuery() with non-array non-string keys returns null.
   */
  public function testBuildMatchQueryWithInvalidKeysType(): void {
    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn(12345);
    $query->method('getFulltextFields')->willReturn([]);

    $fulltext_fields = ['title' => 'title'];

    $result = $this->queryBuilder->buildMatchQuery($query, $fulltext_fields);

    $this->assertNull($result);
  }

  /**
   * Tests buildMatchQuery() filters short terms.
   */
  public function testBuildMatchQueryFiltersShortTerms(): void {
    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn([
      '#conjunction' => 'AND',
      0 => 'a',
      1 => 'valid',
    ]);
    $query->method('getFulltextFields')->willReturn([]);

    $fulltext_fields = ['title' => 'title'];

    $result = $this->queryBuilder->buildMatchQuery($query, $fulltext_fields);

    $this->assertNotNull($result);
    $this->assertStringContainsString('valid', $result);
    // Short term 'a' should not appear.
    $this->assertStringNotContainsString('{title}: a ', $result);
  }

  /**
   * Tests buildMatchQuery() with multiple columns.
   */
  public function testBuildMatchQueryWithMultipleColumns(): void {
    $query = $this->createMock(QueryInterface::class);
    $query->method('getKeys')->willReturn('search');
    $query->method('getFulltextFields')->willReturn(['title', 'body']);

    $fulltext_fields = ['title' => 'title', 'body' => 'body'];

    $result = $this->queryBuilder->buildMatchQuery($query, $fulltext_fields);

    $this->assertNotNull($result);
    $this->assertStringContainsString('{title body}', $result);
  }

  /**
   * Tests parseSearchInput() with unclosed quote.
   */
  public function testParseSearchInputWithUnclosedQuote(): void {
    $result = $this->queryBuilder->parseSearchInput('"unclosed phrase');

    // Unclosed quotes should be treated as regular text.
    $this->assertEmpty($result['phrases']);
    $this->assertContains('phrase', $result['terms']);
  }

  /**
   * Tests escapeTerm() with NEAR operator.
   */
  public function testEscapeTermWithNearOperator(): void {
    $result = $this->queryBuilder->escapeTerm('NEAR');
    $this->assertSame('"NEAR"', $result);
  }

  /**
   * Tests escapeTerm() with parentheses.
   */
  public function testEscapeTermWithParentheses(): void {
    $result = $this->queryBuilder->escapeTerm('test(value)');
    $this->assertSame('"test(value)"', $result);
  }

}
