<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Database;

use Psr\Log\LoggerInterface;

/**
 * Logs database queries for debugging and performance analysis.
 */
final class QueryLogger implements QueryLoggerInterface {

  /**
   * Whether query logging is enabled.
   */
  private bool $enabled = FALSE;

  /**
   * Constructs a QueryLogger instance.
   *
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function log(string $sql, array $params, float $duration_ms, string $source): void {
    if (!$this->enabled) {
      return;
    }

    $this->logger->debug('[@source] @sql | Params: @params | Duration: @duration ms', [
      '@source' => $source,
      '@sql' => $this->formatSql($sql),
      '@params' => $this->formatParams($params),
      '@duration' => number_format($duration_ms, 2),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function startTimer(): float {
    return hrtime(TRUE) / 1e6;
  }

  /**
   * {@inheritdoc}
   */
  public function endTimer(float $start): float {
    $end = hrtime(TRUE) / 1e6;
    return $end - $start;
  }

  /**
   * {@inheritdoc}
   */
  public function setEnabled(bool $enabled): void {
    $this->enabled = $enabled;
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(): bool {
    return $this->enabled;
  }

  /**
   * Formats SQL for logging by normalizing whitespace.
   *
   * @param string $sql
   *   The raw SQL string.
   *
   * @return string
   *   The formatted SQL.
   */
  private function formatSql(string $sql): string {
    // Normalize whitespace for readability.
    return preg_replace('/\s+/', ' ', trim($sql)) ?? $sql;
  }

  /**
   * Formats query parameters for logging.
   *
   * @param array<string, mixed> $params
   *   The query parameters.
   *
   * @return string
   *   JSON-encoded parameters or empty indicator.
   */
  private function formatParams(array $params): string {
    if ($params === []) {
      return '(none)';
    }

    // Truncate long values for readability.
    $formatted = [];
    foreach ($params as $key => $value) {
      if (is_string($value) && strlen($value) > 100) {
        $value = substr($value, 0, 100) . '...';
      }

      $formatted[$key] = $value;
    }

    $json = json_encode($formatted, JSON_UNESCAPED_SLASHES);
    return $json !== FALSE ? $json : '(error encoding params)';
  }

}
