<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Hook;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\search_api\IndexInterface;

/**
 * Hook implementations for entity operations.
 */
final class EntityHooks {

  use StringTranslationTrait;

  /**
   * The SQLite backend plugin ID.
   */
  private const string BACKEND_PLUGIN_ID = 'search_api_sqlite';

  /**
   * Constructs an EntityHooks instance.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   */
  public function __construct(
    private readonly AccountProxyInterface $currentUser,
  ) {}

  /**
   * Implements hook_entity_operation().
   *
   * Adds "SQLite Operations" link to the operations dropdown for indexes
   * that use the SQLite backend.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return array<string, array<string, mixed>>
   *   An array of operation links.
   */
  #[Hook('entity_operation')]
  public function entityOperation(EntityInterface $entity): array {
    $operations = [];

    // Only add for Search API indexes.
    if (!$entity instanceof IndexInterface) {
      return $operations;
    }

    // Check user permission.
    if (!$this->currentUser->hasPermission('administer search_api')) {
      return $operations;
    }

    // Check if index uses SQLite backend.
    if (!$this->indexUsesSqliteBackend($entity)) {
      return $operations;
    }

    $operations['sqlite_operations'] = [
      'title' => $this->t('SQLite Operations'),
      'url' => Url::fromRoute('search_api_sqlite.index_operations', [
        'search_api_index' => $entity->id(),
      ]),
      'weight' => 50,
    ];

    return $operations;
  }

  /**
   * Checks if an index uses the SQLite backend.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The index to check.
   *
   * @return bool
   *   TRUE if the index uses SQLite backend, FALSE otherwise.
   */
  private function indexUsesSqliteBackend(IndexInterface $index): bool {
    if (!$index->hasValidServer()) {
      return FALSE;
    }

    try {
      $server = $index->getServerInstance();
      return $server->getBackendId() === self::BACKEND_PLUGIN_ID;
    }
    catch (\Exception) {
      return FALSE;
    }
  }

}
