<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Index;

use Drupal\search_api\Item\FieldInterface;
use Drupal\search_api\Utility\DataTypeHelperInterface;

/**
 * Maps Search API field types to SQLite storage types.
 */
final readonly class FieldTypeMapper implements FieldTypeMapperInterface {

  /**
   * Type mapping to storage types.
   *
   * @var array<string, string>
   */
  private const array TYPE_MAP = [
    // Fulltext types.
    'text' => self::STORAGE_FTS5,
    // String types.
    'string' => self::STORAGE_STRING,
    'uri' => self::STORAGE_STRING,
    // Integer types.
    'integer' => self::STORAGE_INTEGER,
    'boolean' => self::STORAGE_INTEGER,
    'date' => self::STORAGE_INTEGER,
    // Decimal types.
    'decimal' => self::STORAGE_DECIMAL,
  ];

  /**
   * Column names for storage types.
   *
   * @var array<string, string>
   */
  private const array COLUMN_MAP = [
    self::STORAGE_STRING => 'value_string',
    self::STORAGE_INTEGER => 'value_integer',
    self::STORAGE_DECIMAL => 'value_decimal',
  ];

  /**
   * Constructs a FieldTypeMapper instance.
   *
   * @param \Drupal\search_api\Utility\DataTypeHelperInterface $dataTypeHelper
   *   The Search API data type helper.
   */
  public function __construct(
    private DataTypeHelperInterface $dataTypeHelper,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function isFulltextType(string $type): bool {
    return $this->dataTypeHelper->isTextType($type);
  }

  /**
   * {@inheritdoc}
   */
  public function getStorageType(string $type): string {
    // Check for text type (including derived text types).
    if ($this->dataTypeHelper->isTextType($type)) {
      return self::STORAGE_FTS5;
    }

    // Default to string for unknown types.
    return self::TYPE_MAP[$type] ?? self::STORAGE_STRING;
  }

  /**
   * {@inheritdoc}
   */
  public function convertValue(mixed $value, string $type): mixed {
    if ($value === NULL || $value === '') {
      return NULL;
    }

    $storage_type = $this->getStorageType($type);

    return match ($storage_type) {
      self::STORAGE_FTS5 => $this->convertToText($value),
      self::STORAGE_INTEGER => $this->convertToInteger($value, $type),
      self::STORAGE_DECIMAL => $this->convertToDecimal($value),
      default => $this->convertToString($value),
    };
  }

  /**
   * {@inheritdoc}
   */
  public function getFieldDataColumn(string $storage_type): string {
    return self::COLUMN_MAP[$storage_type] ?? 'value_string';
  }

  /**
   * {@inheritdoc}
   *
   * @return array<int, mixed>
   *   Array of converted values.
   */
  public function extractFieldValues(FieldInterface $field): array {
    $values = $field->getValues();

    if ($values === []) {
      return [];
    }

    $type = $field->getType();
    $converted = [];

    foreach ($values as $value) {
      $converted_value = $this->convertValue($value, $type);
      if ($converted_value !== NULL) {
        $converted[] = $converted_value;
      }
    }

    return $converted;
  }

  /**
   * Converts a value to text for FTS5 storage.
   *
   * @param mixed $value
   *   The value to convert.
   *
   * @return string|null
   *   The text value or NULL.
   */
  private function convertToText(mixed $value): ?string {
    if ($value === NULL || $value === '') {
      return NULL;
    }

    // Handle TextValue objects from Search API.
    if (is_object($value) && method_exists($value, 'getText')) {
      $value = $value->getText();
    }

    if (!is_scalar($value)) {
      return NULL;
    }

    $text = (string) $value;

    // Strip HTML tags but preserve content.
    $text = strip_tags($text);

    // Normalize whitespace.
    $text = preg_replace('/\s+/', ' ', $text) ?? $text;

    return trim($text);
  }

  /**
   * Converts a value to integer for storage.
   *
   * @param mixed $value
   *   The value to convert.
   * @param string $type
   *   The original field type.
   *
   * @return int|null
   *   The integer value or NULL.
   */
  private function convertToInteger(mixed $value, string $type): ?int {
    if ($value === NULL || $value === '') {
      return NULL;
    }

    // Handle boolean type.
    if ($type === 'boolean') {
      return (bool) $value ? 1 : 0;
    }

    // Handle date type (expect timestamp or date string).
    if ($type === 'date') {
      if (is_numeric($value)) {
        return (int) $value;
      }

      // Try to parse date string.
      $timestamp = strtotime((string) $value);
      return $timestamp !== FALSE ? $timestamp : NULL;
    }

    // Regular integer.
    if (!is_numeric($value)) {
      return NULL;
    }

    return (int) $value;
  }

  /**
   * Converts a value to decimal for storage.
   *
   * @param mixed $value
   *   The value to convert.
   *
   * @return float|null
   *   The decimal value or NULL.
   */
  private function convertToDecimal(mixed $value): ?float {
    if ($value === NULL || $value === '') {
      return NULL;
    }

    if (!is_numeric($value)) {
      return NULL;
    }

    return (float) $value;
  }

  /**
   * Converts a value to string for storage.
   *
   * @param mixed $value
   *   The value to convert.
   *
   * @return string|null
   *   The string value or NULL.
   */
  private function convertToString(mixed $value): ?string {
    if ($value === NULL || $value === '') {
      return NULL;
    }

    if (!is_scalar($value)) {
      return NULL;
    }

    $string = (string) $value;

    // Truncate to max column length.
    if (strlen($string) > 1024) {
      return substr($string, 0, 1024);
    }

    return $string;
  }

}
