<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Database\Connection;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Query\ConditionGroupInterface;
use Drupal\search_api\Query\ConditionInterface;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Index\FieldTypeMapperInterface;
use Drupal\search_api_sqlite\Utility\VerboseLoggerTrait;
use Psr\Log\LoggerInterface;

/**
 * Handles search query conditions and filters.
 *
 * Processes Search API query conditions (including nested condition groups)
 * and filters item IDs accordingly. Supports AND/OR conjunctions and various
 * comparison operators.
 */
final class ConditionHandler implements ConditionHandlerInterface {

  use VerboseLoggerTrait;

  /**
   * The current backend configuration.
   *
   * @var array<string, mixed>
   */
  private array $backendConfig = [];

  /**
   * Temporary table name for large item sets, if created.
   */
  private ?string $currentTempTable = NULL;

  /**
   * The current index ID for temp table operations.
   */
  private ?string $currentIndexId = NULL;

  /**
   * Constructs a ConditionHandler instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The database connection manager.
   * @param \Drupal\search_api_sqlite\Database\SchemaManagerInterface $schemaManager
   *   The schema manager.
   * @param \Drupal\search_api_sqlite\Index\FieldTypeMapperInterface $fieldTypeMapper
   *   The field type mapper.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    private readonly ConnectionManagerInterface $connectionManager,
    private readonly SchemaManagerInterface $schemaManager,
    private readonly FieldTypeMapperInterface $fieldTypeMapper,
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function applyConditions(QueryInterface $query, string $index_id, array $item_ids, array $backend_config = []): array {
    return $this->applyConditionsExcludingTags($query, $index_id, $item_ids, [], $backend_config);
  }

  /**
   * {@inheritdoc}
   */
  public function applyConditionsExcludingTags(
    QueryInterface $query,
    string $index_id,
    array $item_ids,
    array $exclude_tags,
    array $backend_config = [],
  ): array {
    $this->backendConfig = $backend_config;
    $conditions = $query->getConditionGroup();

    if (empty($conditions->getConditions())) {
      return $item_ids;
    }

    $connection = $this->connectionManager->getConnection($index_id);
    $index = $query->getIndex();

    // For large item sets, use temp table approach for better performance.
    // This avoids SQLite's parameter limit and query planner inefficiencies.
    $use_temp_table = $this->connectionManager->shouldUseTempTable(count($item_ids));

    if ($use_temp_table) {
      $this->currentTempTable = $this->connectionManager->createTempItemsTable(
        $index_id,
        $item_ids,
        'condition_filter'
      );
      $this->currentIndexId = $index_id;
    }

    try {
      return $this->applyConditionGroup(
        $conditions,
        $connection,
        $index,
        $index_id,
        $item_ids,
        $exclude_tags,
        $use_temp_table
      );
    }
    finally {
      // Always clean up temp table.
      if ($this->currentTempTable !== NULL && $this->currentIndexId !== NULL) {
        $this->connectionManager->dropTempTable($this->currentIndexId, $this->currentTempTable);
        $this->currentTempTable = NULL;
        $this->currentIndexId = NULL;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getValueColumnForField(IndexInterface $index, string $field_name): string {
    $field = $index->getField($field_name);
    if (!$field) {
      // Default to string if field not found.
      return 'value_string';
    }

    $type = $field->getType();
    $storage_type = $this->fieldTypeMapper->getStorageType($type);

    return $this->fieldTypeMapper->getFieldDataColumn($storage_type);
  }

  /**
   * Recursively applies a condition group to filter item IDs.
   *
   * @param \Drupal\search_api\Query\ConditionGroupInterface $condition_group
   *   The condition group to apply.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param \Drupal\search_api\IndexInterface $index
   *   The search index.
   * @param string $index_id
   *   The index ID.
   * @param array<string> $item_ids
   *   Current item IDs.
   * @param array<string> $exclude_tags
   *   Tags of condition groups to skip.
   * @param bool $use_temp_table
   *   Whether to use temp table for filtering instead of IN clause.
   *
   * @return array<string>
   *   Filtered item IDs.
   */
  private function applyConditionGroup(
    ConditionGroupInterface $condition_group,
    Connection $connection,
    IndexInterface $index,
    string $index_id,
    array $item_ids,
    array $exclude_tags = [],
    bool $use_temp_table = FALSE,
  ): array {
    // Check if this condition group should be excluded based on its tags.
    if ($exclude_tags !== []) {
      $group_tags = $condition_group->getTags();
      foreach ($exclude_tags as $exclude_tag) {
        if (in_array($exclude_tag, $group_tags, TRUE)) {
          // Skip this entire condition group.
          $this->logVerbose('Skipping condition group with tag @tag for OR facet', [
            '@tag' => $exclude_tag,
          ]);
          return $item_ids;
        }
      }
    }

    $conjunction = $condition_group->getConjunction();
    $field_data_table = $this->schemaManager->getFieldDataTableName($index_id);

    // For OR conjunction, we collect all matching IDs and union them.
    $or_results = [];

    foreach ($condition_group->getConditions() as $condition) {
      // If we have no items left and using AND, no need to continue.
      if ($item_ids === [] && $conjunction === 'AND') {
        return [];
      }

      // Handle nested condition groups recursively.
      if ($condition instanceof ConditionGroupInterface) {
        $nested_results = $this->applyConditionGroup(
          $condition,
          $connection,
          $index,
          $index_id,
          $item_ids,
          $exclude_tags,
          $use_temp_table
        );

        if ($conjunction === 'OR') {
          $or_results = array_merge($or_results, $nested_results);
        }
        else {
          $item_ids = $nested_results;
        }

        continue;
      }

      // At this point, condition must be a ConditionInterface.
      // @phpstan-ignore instanceof.alwaysTrue
      if (!$condition instanceof ConditionInterface) {
        continue;
      }

      $field = $condition->getField();
      $value = $condition->getValue();
      $operator = $condition->getOperator();

      // Determine the correct column based on field type.
      $value_column = $this->getValueColumnForField($index, $field);

      $this->logVerbose('Applying condition: field=@field, value=@value, operator=@op, column=@column', [
        '@field' => $field,
        '@value' => is_array($value) ? implode(',', $value) : (string) $value,
        '@op' => $operator,
        '@column' => $value_column,
      ]);

      // Build a query to filter items - use temp table for large sets.
      $matching_ids = $use_temp_table && $this->currentTempTable !== NULL
        ? $this->executeConditionWithTempTable($index_id, $field_data_table, $field, $value_column, $value, $operator)
        : $this->executeConditionWithInClause($connection, $field_data_table, $field, $value_column, $value, $operator, $item_ids);

      $this->logVerbose('Condition matched @count items out of @total', [
        '@count' => count($matching_ids),
        '@total' => count($item_ids),
      ]);

      if ($conjunction === 'OR') {
        $or_results = array_merge($or_results, $matching_ids);
      }
      else {
        // AND conjunction: intersect with current results.
        $item_ids = array_intersect($item_ids, $matching_ids);
      }
    }

    // For OR conjunction, return unique union of all matches.
    if ($conjunction === 'OR' && $or_results !== []) {
      return array_values(array_unique($or_results));
    }

    return array_values($item_ids);
  }

  /**
   * Executes a condition query using temp table JOIN for large item sets.
   *
   * @param string $index_id
   *   The index ID.
   * @param string $field_data_table
   *   The field data table name.
   * @param string $field
   *   The field name.
   * @param string $value_column
   *   The value column name.
   * @param mixed $value
   *   The value to compare against.
   * @param string $operator
   *   The comparison operator.
   *
   * @return array<string>
   *   Matching item IDs.
   */
  private function executeConditionWithTempTable(
    string $index_id,
    string $field_data_table,
    string $field,
    string $value_column,
    mixed $value,
    string $operator,
  ): array {
    $pdo = $this->connectionManager->getPdo($index_id);
    $temp_table = $this->currentTempTable;

    // Build condition clause based on operator.
    $params = [];
    $condition_sql = $this->buildConditionSql($value_column, $operator, $value, $params);

    $sql = sprintf(
      'SELECT DISTINCT fd.item_id
       FROM %s fd
       INNER JOIN %s t ON fd.item_id = t.item_id
       WHERE fd.field_name = ? %s',
      $field_data_table,
      $temp_table,
      $condition_sql !== '' ? 'AND ' . $condition_sql : ''
    );

    array_unshift($params, $field);
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);

    return $stmt->fetchAll(\PDO::FETCH_COLUMN) ?: [];
  }

  /**
   * Executes a condition query using IN clause for small item sets.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param string $field_data_table
   *   The field data table name.
   * @param string $field
   *   The field name.
   * @param string $value_column
   *   The value column name.
   * @param mixed $value
   *   The value to compare against.
   * @param string $operator
   *   The comparison operator.
   * @param array<string> $item_ids
   *   The item IDs to filter.
   *
   * @return array<string>
   *   Matching item IDs.
   */
  private function executeConditionWithInClause(
    Connection $connection,
    string $field_data_table,
    string $field,
    string $value_column,
    mixed $value,
    string $operator,
    array $item_ids,
  ): array {
    $subquery = $connection->select($field_data_table, 'fd')
      ->fields('fd', ['item_id'])
      ->condition('fd.field_name', $field)
      ->condition('fd.item_id', $item_ids, 'IN');

    // Apply the condition based on operator.
    $this->applyOperatorCondition($subquery, $value_column, $value, $operator);

    $result = $subquery->execute();
    return $result !== NULL ? $result->fetchCol() : [];
  }

  /**
   * Builds SQL condition clause for PDO query.
   *
   * @param string $column
   *   The column name.
   * @param string $operator
   *   The comparison operator.
   * @param mixed $value
   *   The value to compare against.
   * @param array<int, mixed> $params
   *   Reference to params array to populate.
   *
   * @return string
   *   The SQL condition clause.
   */
  private function buildConditionSql(string $column, string $operator, mixed $value, array &$params): string {
    $params = [];

    switch ($operator) {
      case '=':
      case '<>':
      case '<':
      case '<=':
      case '>':
      case '>=':
        $params[] = $value;
        return sprintf('fd.%s %s ?', $column, $operator);

      case 'IN':
        $values = (array) $value;
        if ($values === []) {
          return '1 = 0';
        }

        $placeholders = implode(', ', array_fill(0, count($values), '?'));
        $params = array_values($values);
        return sprintf('fd.%s IN (%s)', $column, $placeholders);

      case 'NOT IN':
        $values = (array) $value;
        if ($values === []) {
          return '';
        }

        $placeholders = implode(', ', array_fill(0, count($values), '?'));
        $params = array_values($values);
        return sprintf('fd.%s NOT IN (%s)', $column, $placeholders);

      case 'BETWEEN':
        if (is_array($value) && count($value) >= 2) {
          $params = [$value[0], $value[1]];
          return sprintf('fd.%s >= ? AND fd.%s <= ?', $column, $column);
        }

        return '';

      default:
        return '';
    }
  }

  /**
   * Applies an operator condition to a query.
   *
   * @param \Drupal\Core\Database\Query\SelectInterface $query
   *   The query to modify.
   * @param string $column
   *   The column name.
   * @param mixed $value
   *   The value to compare against.
   * @param string $operator
   *   The comparison operator.
   */
  private function applyOperatorCondition(
    SelectInterface $query,
    string $column,
    mixed $value,
    string $operator,
  ): void {
    switch ($operator) {
      case '=':
      case '<>':
      case '<':
      case '<=':
      case '>':
      case '>=':
        $query->condition('fd.' . $column, $value, $operator);
        break;

      case 'IN':
        $query->condition('fd.' . $column, (array) $value, 'IN');
        break;

      case 'NOT IN':
        $query->condition('fd.' . $column, (array) $value, 'NOT IN');
        break;

      case 'BETWEEN':
        if (is_array($value) && count($value) >= 2) {
          $query->condition('fd.' . $column, $value[0], '>=');
          $query->condition('fd.' . $column, $value[1], '<=');
        }

        break;
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function getVerboseLogger(): LoggerInterface {
    return $this->logger;
  }

  /**
   * {@inheritdoc}
   */
  protected function getVerboseLoggerConfig(): array {
    return $this->backendConfig;
  }

}
