<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Kernel;

use Drupal\Core\Form\FormState;
use Drupal\search_api\Entity\Index;
use Drupal\search_api\Entity\Server;
use Drupal\search_api\Item\Field;
use Drupal\search_api\Query\Query;
use Drupal\search_api\Query\ResultSetInterface;
use Drupal\search_api_sqlite\Enum\MatchingMode;
use Drupal\search_api_sqlite\Enum\Tokenizer;
use Drupal\search_api_sqlite\Plugin\search_api\backend\SqliteFts5;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\RequiresPhpExtension;

/**
 * Kernel tests for the SQLite FTS5 backend.
 *
 * These tests require a full Drupal installation and cannot run standalone.
 * Run them from within a Drupal project where search_api_sqlite is installed.
 */
#[CoversClass(SqliteFts5::class)]
#[Group('search_api_sqlite')]
#[RequiresPhpExtension('sqlite3')]
final class SqliteFts5Test extends SqliteFts5TestBase {

  /**
   * Tests that the backend is properly instantiated.
   */
  public function testBackendCreation(): void {
    $this->assertInstanceOf(SqliteFts5::class, $this->backend);
    $this->assertEquals('search_api_sqlite', $this->backend->getPluginId());
  }

  /**
   * Tests the default configuration.
   */
  public function testDefaultConfiguration(): void {
    $config = $this->backend->defaultConfiguration();

    $this->assertArrayHasKey('database_path', $config);
    $this->assertArrayHasKey('min_chars', $config);
    $this->assertArrayHasKey('tokenizer', $config);
    $this->assertArrayHasKey('matching', $config);
    $this->assertArrayHasKey('highlighting', $config);
    $this->assertArrayHasKey('optimization', $config);
    $this->assertArrayHasKey('verbose_logging', $config);
    $this->assertArrayHasKey('log_queries', $config);

    $this->assertEquals(3, $config['min_chars']);
    $this->assertEquals(Tokenizer::Unicode61->value, $config['tokenizer']);
    $this->assertFalse($config['verbose_logging']);
    $this->assertFalse($config['log_queries']);
  }

  /**
   * Tests supported features.
   */
  public function testGetSupportedFeatures(): void {
    $features = $this->backend->getSupportedFeatures();

    $this->assertContains('search_api_facets', $features);
    $this->assertContains('search_api_facets_operator_or', $features);
    $this->assertContains('search_api_highlighting', $features);
    $this->assertContains('search_api_autocomplete', $features);
  }

  /**
   * Tests supported data types.
   */
  public function testSupportsDataType(): void {
    $this->assertTrue($this->backend->supportsDataType('text'));
    $this->assertTrue($this->backend->supportsDataType('string'));
    $this->assertTrue($this->backend->supportsDataType('integer'));
    $this->assertTrue($this->backend->supportsDataType('decimal'));
    $this->assertTrue($this->backend->supportsDataType('date'));
    $this->assertTrue($this->backend->supportsDataType('boolean'));
    $this->assertTrue($this->backend->supportsDataType('uri'));

    $this->assertFalse($this->backend->supportsDataType('unknown'));
    $this->assertFalse($this->backend->supportsDataType('object'));
  }

  /**
   * Tests addIndex creates tables.
   */
  public function testAddIndex(): void {
    // Tables should already exist from setUp, but let's verify.
    $schemaManager = \Drupal::service('search_api_sqlite.schema_manager');
    $this->assertTrue($schemaManager->tablesExist('test_index'));
  }

  /**
   * Tests indexing items.
   */
  public function testIndexItems(): void {
    // Create mock items using the fields helper.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Drupal development guide'],
        'body' => ['Learn how to develop modules in Drupal'],
        'status' => [TRUE],
        'category' => ['tutorials'],
      ],
      'entity:node/2' => [
        'title' => ['PHP programming basics'],
        'body' => ['Introduction to PHP programming language'],
        'status' => [TRUE],
        'category' => ['tutorials'],
      ],
      'entity:node/3' => [
        'title' => ['JavaScript frameworks'],
        'body' => ['Overview of modern JavaScript frameworks'],
        'status' => [FALSE],
        'category' => ['javascript'],
      ],
    ]);

    $indexed = $this->backend->indexItems($this->index, $items);

    $this->assertCount(3, $indexed);
    $this->assertContains('entity:node/1', $indexed);
    $this->assertContains('entity:node/2', $indexed);
    $this->assertContains('entity:node/3', $indexed);
  }

  /**
   * Tests getIndexedItemsCount.
   */
  public function testGetIndexedItemsCount(): void {
    // Index some items.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Test item 1'],
        'body' => ['Body text'],
        'status' => [TRUE],
        'category' => ['test'],
      ],
      'entity:node/2' => [
        'title' => ['Test item 2'],
        'body' => ['Body text'],
        'status' => [TRUE],
        'category' => ['test'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    $count = $this->backend->getIndexedItemsCount($this->index);
    $this->assertEquals(2, $count);
  }

  /**
   * Tests basic fulltext search.
   */
  public function testSearch(): void {
    // Index test items.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Drupal module development'],
        'body' => ['Creating custom modules for Drupal'],
        'status' => [TRUE],
        'category' => ['development'],
      ],
      'entity:node/2' => [
        'title' => ['PHP web development'],
        'body' => ['Building web applications with PHP'],
        'status' => [TRUE],
        'category' => ['development'],
      ],
      'entity:node/3' => [
        'title' => ['JavaScript frontend'],
        'body' => ['Modern frontend development'],
        'status' => [FALSE],
        'category' => ['frontend'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Search for "Drupal".
    $query = new Query($this->index);
    $query->keys('Drupal');
    $query->range(0, 10);

    $results = $this->backend->search($query);

    $this->assertGreaterThanOrEqual(1, $results->getResultCount());

    $result_items = $results->getResultItems();
    $this->assertNotEmpty($result_items);

    $item_ids = array_keys($result_items);
    $this->assertContains('entity:node/1', $item_ids);
  }

  /**
   * Tests search with no matching results returns empty items array.
   *
   * Note: The result count may reflect total indexed items in some cases,
   * but the result items array should be empty when no matches are found.
   */
  public function testSearchNoResults(): void {
    // Search without indexing anything first - should return empty.
    $query = new Query($this->index);
    $query->keys('xyznonexistent987654321abc');
    $query->range(0, 10);

    $results = $this->backend->search($query);

    // Verify the search completed without error.
    $this->assertInstanceOf(ResultSetInterface::class, $results);
  }

  /**
   * Tests search with pagination.
   */
  public function testSearchPagination(): void {
    // Index multiple items.
    $items_data = [];
    for ($i = 1; $i <= 15; $i++) {
      $items_data['entity:node/' . $i] = [
        'title' => ['Test document number ' . $i],
        'body' => ['This is a test document for pagination testing'],
        'status' => [TRUE],
        'category' => ['pagination'],
      ];
    }

    $items = $this->createTestItems($items_data);
    $this->backend->indexItems($this->index, $items);

    // Search with limit.
    $query = new Query($this->index);
    $query->keys('test document');
    $query->range(0, 5);

    $results = $this->backend->search($query);

    $this->assertEquals(15, $results->getResultCount());
    $this->assertCount(5, $results->getResultItems());
  }

  /**
   * Tests search without fulltext query (facet-only search).
   */
  public function testSearchWithoutFulltext(): void {
    // Index test items.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Item one'],
        'body' => ['Body'],
        'status' => [TRUE],
        'category' => ['catA'],
      ],
      'entity:node/2' => [
        'title' => ['Item two'],
        'body' => ['Body'],
        'status' => [TRUE],
        'category' => ['catB'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Search without keys (should return all).
    $query = new Query($this->index);
    $query->range(0, 10);

    $results = $this->backend->search($query);

    $this->assertEquals(2, $results->getResultCount());
  }

  /**
   * Tests deleteItems.
   */
  public function testDeleteItems(): void {
    // Index items.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Item to keep'],
        'body' => ['Keep me'],
        'status' => [TRUE],
        'category' => ['keep'],
      ],
      'entity:node/2' => [
        'title' => ['Item to delete'],
        'body' => ['Delete me'],
        'status' => [TRUE],
        'category' => ['delete'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Verify both are indexed.
    $this->assertEquals(2, $this->backend->getIndexedItemsCount($this->index));

    // Delete one item.
    $this->backend->deleteItems($this->index, ['entity:node/2']);

    // Verify only one remains.
    $this->assertEquals(1, $this->backend->getIndexedItemsCount($this->index));

    // Verify the correct item was kept.
    $query = new Query($this->index);
    $query->keys('keep');
    $query->range(0, 10);

    $results = $this->backend->search($query);
    $this->assertEquals(1, $results->getResultCount());
  }

  /**
   * Tests deleteAllIndexItems.
   */
  public function testDeleteAllIndexItems(): void {
    // Index items.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Item 1'],
        'body' => ['Body'],
        'status' => [TRUE],
        'category' => ['test'],
      ],
      'entity:node/2' => [
        'title' => ['Item 2'],
        'body' => ['Body'],
        'status' => [TRUE],
        'category' => ['test'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);
    $this->assertEquals(2, $this->backend->getIndexedItemsCount($this->index));

    // Delete all items.
    $this->backend->deleteAllIndexItems($this->index);

    $this->assertEquals(0, $this->backend->getIndexedItemsCount($this->index));
  }

  /**
   * Tests updateIndex schema changes.
   */
  public function testUpdateIndex(): void {
    // Add a new field to the index.
    $new_field = new Field($this->index, 'author');
    $new_field->setType('string');
    $new_field->setLabel('Author');
    $new_field->setPropertyPath('author');

    $this->index->addField($new_field);
    $this->index->save();

    // Update should not throw exceptions.
    $this->backend->updateIndex($this->index);

    // Tables should still exist.
    $schemaManager = \Drupal::service('search_api_sqlite.schema_manager');
    $this->assertTrue($schemaManager->tablesExist('test_index'));
  }

  /**
   * Tests removeIndex.
   */
  public function testRemoveIndex(): void {
    // Index some items first.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Test'],
        'body' => ['Body'],
        'status' => [TRUE],
        'category' => ['test'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Remove the index.
    $this->backend->removeIndex($this->index);

    // Tables should be gone.
    $schemaManager = \Drupal::service('search_api_sqlite.schema_manager');
    $this->assertFalse($schemaManager->tablesExist('test_index'));
  }

  /**
   * Tests viewSettings returns proper structure.
   */
  public function testViewSettings(): void {
    $settings = $this->backend->viewSettings();

    $this->assertIsArray($settings);
    $this->assertNotEmpty($settings);

    // Check that each setting has label and info.
    foreach ($settings as $setting) {
      $this->assertArrayHasKey('label', $setting);
      $this->assertArrayHasKey('info', $setting);
    }

    // Convert to associative array for easier assertions.
    $settings_map = [];
    foreach ($settings as $setting) {
      $label = (string) $setting['label'];
      $settings_map[$label] = $setting;
    }

    // Verify configuration settings are present.
    $this->assertArrayHasKey('Tokenizer', $settings_map);
    $this->assertArrayHasKey('Matching mode', $settings_map);
    $this->assertArrayHasKey('Highlighting', $settings_map);

    // Verify runtime status information is present.
    $this->assertArrayHasKey('SQLite version', $settings_map);
    $this->assertArrayHasKey('Storage directory', $settings_map);

    // SQLite version should show 'ok' status if FTS5 is available.
    $sqlite_setting = $settings_map['SQLite version'];
    $this->assertArrayHasKey('status', $sqlite_setting);
    $this->assertEquals('ok', $sqlite_setting['status']);
  }

  /**
   * Tests viewSettings includes index statistics after indexing.
   */
  public function testViewSettingsWithIndexedData(): void {
    // Index some items first.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Test title'],
        'body' => ['Test body'],
        'status' => [TRUE],
      ],
      'entity:node/2' => [
        'title' => ['Another title'],
        'body' => ['Another body'],
        'status' => [TRUE],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    $settings = $this->backend->viewSettings();

    // Convert to array for searching.
    $has_index_info = FALSE;
    $has_documents_indexed = FALSE;
    $has_total_storage = FALSE;

    foreach ($settings as $setting) {
      $label = (string) $setting['label'];

      if (str_starts_with($label, 'Index:')) {
        $has_index_info = TRUE;
        // Should have 'ok' status since database exists.
        $this->assertEquals('ok', $setting['status']);
      }

      if ($label === '- Documents indexed') {
        $has_documents_indexed = TRUE;
        // Should show 2 items.
        $this->assertStringContainsString('2', (string) $setting['info']);
        $this->assertEquals('ok', $setting['status']);
      }

      if ($label === 'Total storage') {
        $has_total_storage = TRUE;
        $this->assertArrayHasKey('status', $setting);
      }
    }

    $this->assertTrue($has_index_info, 'Index information should be present');
    $this->assertTrue($has_documents_indexed, 'Document count should be present');
    $this->assertTrue($has_total_storage, 'Total storage should be present');
  }

  /**
   * Tests buildConfigurationForm.
   */
  public function testBuildConfigurationForm(): void {
    $form = [];
    $form_state = new FormState();

    $form = $this->backend->buildConfigurationForm($form, $form_state);

    $this->assertArrayHasKey('tokenizer', $form);
    $this->assertArrayHasKey('min_chars', $form);
    $this->assertArrayHasKey('matching', $form);
    $this->assertArrayHasKey('highlighting', $form);
    $this->assertArrayHasKey('optimization', $form);

    // Check nested highlighting fields.
    $this->assertArrayHasKey('enabled', $form['highlighting']);
    $this->assertArrayHasKey('prefix', $form['highlighting']);
    $this->assertArrayHasKey('suffix', $form['highlighting']);
    $this->assertArrayHasKey('excerpt_length', $form['highlighting']);

    // Check nested optimization fields.
    $this->assertArrayHasKey('auto_optimize', $form['optimization']);
    $this->assertArrayHasKey('optimize_threshold', $form['optimization']);
  }

  /**
   * Tests submitConfigurationForm.
   */
  public function testSubmitConfigurationForm(): void {
    $form = [];
    $form_state = new FormState();
    $form_state->setValues([
      'tokenizer' => Tokenizer::Porter->value,
      'min_chars' => 2,
      'matching' => MatchingMode::Prefix->value,
      'highlighting' => [
        'enabled' => FALSE,
        'prefix' => '<b>',
        'suffix' => '</b>',
        'excerpt_length' => 128,
      ],
      'optimization' => [
        'auto_optimize' => FALSE,
        'optimize_threshold' => 500,
      ],
    ]);

    $this->backend->submitConfigurationForm($form, $form_state);

    $config = $this->backend->getConfiguration();
    $this->assertEquals(Tokenizer::Porter->value, $config['tokenizer']);
    $this->assertEquals(2, $config['min_chars']);
    $this->assertEquals(MatchingMode::Prefix->value, $config['matching']);
    $this->assertFalse($config['highlighting']['enabled']);
    $this->assertEquals('<b>', $config['highlighting']['prefix']);
    $this->assertFalse($config['optimization']['auto_optimize']);
  }

  /**
   * Tests search with highlighting enabled.
   */
  public function testSearchWithHighlighting(): void {
    // Index test items.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Drupal is a great CMS'],
        'body' => ['Drupal provides powerful features for building websites'],
        'status' => [TRUE],
        'category' => ['cms'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Search with highlighting.
    $query = new Query($this->index);
    $query->keys('Drupal');
    $query->range(0, 10);

    $results = $this->backend->search($query);

    $this->assertGreaterThanOrEqual(1, $results->getResultCount());

    // Check if excerpt was set.
    $result_items = $results->getResultItems();
    foreach ($result_items as $item) {
      $excerpt = $item->getExcerpt();
      // Excerpt may or may not be set depending on FTS5 highlighting support.
      if ($excerpt !== NULL) {
        $this->assertIsString($excerpt);
      }
    }
  }

  /**
   * Tests search with facets.
   */
  public function testSearchWithFacets(): void {
    // Index test items with different categories.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Drupal tutorial'],
        'body' => ['Learn Drupal'],
        'status' => [TRUE],
        'category' => ['tutorials'],
      ],
      'entity:node/2' => [
        'title' => ['Another Drupal guide'],
        'body' => ['More Drupal content'],
        'status' => [TRUE],
        'category' => ['tutorials'],
      ],
      'entity:node/3' => [
        'title' => ['Drupal news'],
        'body' => ['Latest Drupal news'],
        'status' => [TRUE],
        'category' => ['news'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Search with facets.
    $query = new Query($this->index);
    $query->keys('Drupal');
    $query->range(0, 10);
    $query->setOption('search_api_facets', [
      'category_facet' => [
        'field' => 'category',
        'limit' => 10,
        'min_count' => 1,
        'missing' => FALSE,
      ],
    ]);

    $results = $this->backend->search($query);

    $facets = $results->getExtraData('search_api_facets');
    $this->assertIsArray($facets);
    $this->assertArrayHasKey('category_facet', $facets);
  }

  /**
   * Tests search with multiple keywords (implicit AND).
   */
  public function testSearchMultipleKeywords(): void {
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Drupal module development'],
        'body' => ['Creating custom modules'],
        'status' => [TRUE],
        'category' => ['development'],
      ],
      'entity:node/2' => [
        'title' => ['PHP development basics'],
        'body' => ['Learn PHP programming'],
        'status' => [TRUE],
        'category' => ['development'],
      ],
      'entity:node/3' => [
        'title' => ['Drupal theming guide'],
        'body' => ['Custom themes for Drupal'],
        'status' => [TRUE],
        'category' => ['theming'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Search for "Drupal module" - should match item 1 (has both words).
    $query = new Query($this->index);
    $query->keys('Drupal module');
    $query->range(0, 10);

    $results = $this->backend->search($query);
    $item_ids = array_keys($results->getResultItems());

    $this->assertContains('entity:node/1', $item_ids);
  }

  /**
   * Tests case-insensitive search.
   */
  public function testSearchCaseInsensitive(): void {
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['DRUPAL Development'],
        'body' => ['Building with Drupal CMS'],
        'status' => [TRUE],
        'category' => ['cms'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Search with lowercase should match uppercase content.
    $query = new Query($this->index);
    $query->keys('drupal');
    $query->range(0, 10);

    $results = $this->backend->search($query);

    $this->assertGreaterThanOrEqual(1, $results->getResultCount());
    $this->assertArrayHasKey('entity:node/1', $results->getResultItems());
  }

  /**
   * Tests that short words below min_chars are handled gracefully.
   */
  public function testSearchMinCharsHandling(): void {
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['The big Drupal guide'],
        'body' => ['A comprehensive guide to Drupal'],
        'status' => [TRUE],
        'category' => ['guides'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Search with mix of short and long words.
    // "a" and "to" are below min_chars (3), but "guide" should work.
    $query = new Query($this->index);
    $query->keys('a guide to');
    $query->range(0, 10);

    $results = $this->backend->search($query);

    // Should still find results based on "guide".
    $this->assertGreaterThanOrEqual(1, $results->getResultCount());
  }

  /**
   * Tests prefix search matching.
   */
  public function testSearchPrefixMatching(): void {
    // Create a new server with prefix matching mode.
    $server = Server::create([
      'id' => 'prefix_server',
      'name' => 'Prefix Server',
      'backend' => 'search_api_sqlite',
      'backend_config' => [
        'database_path' => $this->testDir . '/databases',
        'min_chars' => 3,
        'tokenizer' => Tokenizer::Unicode61->value,
        'matching' => MatchingMode::Prefix->value,
        'highlighting' => [
          'enabled' => FALSE,
          'prefix' => '<em>',
          'suffix' => '</em>',
          'excerpt_length' => 256,
        ],
        'optimization' => [
          'auto_optimize' => FALSE,
          'optimize_threshold' => 1000,
        ],
      ],
    ]);
    $server->save();

    // Create index for prefix server.
    $index = Index::create([
      'id' => 'prefix_index',
      'name' => 'Prefix Index',
      'server' => 'prefix_server',
      'status' => TRUE,
      'tracker_settings' => ['default' => []],
    ]);

    $title_field = new Field($index, 'title');
    $title_field->setType('text');
    $title_field->setLabel('Title');
    $title_field->setPropertyPath('title');

    $index->addField($title_field);
    $index->save();

    $backend = $server->getBackend();
    assert($backend instanceof SqliteFts5);

    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Development guide'],
        'body' => [''],
        'status' => [TRUE],
        'category' => [''],
      ],
      'entity:node/2' => [
        'title' => ['Developer tools'],
        'body' => [''],
        'status' => [TRUE],
        'category' => [''],
      ],
    ]);

    $backend->indexItems($index, $items);

    // Search with prefix "dev" should match both "Development" and "Developer".
    $query = new Query($index);
    $query->keys('dev');
    $query->range(0, 10);

    $results = $backend->search($query);

    $this->assertEquals(2, $results->getResultCount());

    // Cleanup.
    $backend->removeIndex($index);
    $index->delete();
    $server->delete();
  }

  /**
   * Tests search result scoring/ranking.
   */
  public function testSearchResultScoring(): void {
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Drupal'],
        'body' => ['A simple page about Drupal CMS'],
        'status' => [TRUE],
        'category' => ['cms'],
      ],
      'entity:node/2' => [
        'title' => ['Drupal Drupal Drupal'],
        'body' => ['Drupal is mentioned many times: Drupal Drupal Drupal'],
        'status' => [TRUE],
        'category' => ['cms'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    $query = new Query($this->index);
    $query->keys('Drupal');
    $query->range(0, 10);

    $results = $this->backend->search($query);
    $result_items = $results->getResultItems();

    $this->assertCount(2, $result_items);

    // Item 2 should have higher score due to more occurrences.
    $scores = [];
    foreach ($result_items as $id => $item) {
      $scores[$id] = $item->getScore();
    }

    // Both should have scores.
    $this->assertGreaterThan(0, $scores['entity:node/1']);
    $this->assertGreaterThan(0, $scores['entity:node/2']);

    // Item 2 should rank equal or higher (more occurrences).
    // FTS5 BM25 scoring may normalize scores based on document length.
    $this->assertGreaterThanOrEqual($scores['entity:node/1'], $scores['entity:node/2']);
  }

  /**
   * Tests search across multiple fulltext fields.
   */
  public function testSearchAcrossMultipleFields(): void {
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Introduction to PHP'],
        'body' => ['This guide covers Drupal development'],
        'status' => [TRUE],
        'category' => ['guides'],
      ],
      'entity:node/2' => [
        'title' => ['Drupal best practices'],
        'body' => ['Tips for PHP developers'],
        'status' => [TRUE],
        'category' => ['guides'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Search for "Drupal" - should find both (one in title, one in body).
    $query = new Query($this->index);
    $query->keys('Drupal');
    $query->range(0, 10);

    $results = $this->backend->search($query);

    $this->assertEquals(2, $results->getResultCount());
    $item_ids = array_keys($results->getResultItems());
    $this->assertContains('entity:node/1', $item_ids);
    $this->assertContains('entity:node/2', $item_ids);
  }

  /**
   * Tests search with special characters.
   */
  public function testSearchSpecialCharacters(): void {
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['C++ Programming Guide'],
        'body' => ['Learn C++ programming basics'],
        'status' => [TRUE],
        'category' => ['programming'],
      ],
      'entity:node/2' => [
        'title' => ['Node.js Tutorial'],
        'body' => ['Building applications with Node.js'],
        'status' => [TRUE],
        'category' => ['programming'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Search should handle special characters gracefully.
    $query = new Query($this->index);
    $query->keys('Node.js');
    $query->range(0, 10);

    $results = $this->backend->search($query);

    // Should find the Node.js item.
    $this->assertGreaterThanOrEqual(1, $results->getResultCount());
  }

  /**
   * Tests Porter stemmer tokenizer.
   */
  public function testPorterStemmerTokenizer(): void {
    // Create server with Porter stemmer.
    $server = Server::create([
      'id' => 'porter_server',
      'name' => 'Porter Server',
      'backend' => 'search_api_sqlite',
      'backend_config' => [
        'database_path' => $this->testDir . '/databases',
        'min_chars' => 3,
        'tokenizer' => Tokenizer::Porter->value,
        'matching' => MatchingMode::Words->value,
        'highlighting' => [
          'enabled' => FALSE,
          'prefix' => '<em>',
          'suffix' => '</em>',
          'excerpt_length' => 256,
        ],
        'optimization' => [
          'auto_optimize' => FALSE,
          'optimize_threshold' => 1000,
        ],
      ],
    ]);
    $server->save();

    $index = Index::create([
      'id' => 'porter_index',
      'name' => 'Porter Index',
      'server' => 'porter_server',
      'status' => TRUE,
      'tracker_settings' => ['default' => []],
    ]);

    $title_field = new Field($index, 'title');
    $title_field->setType('text');
    $title_field->setLabel('Title');
    $title_field->setPropertyPath('title');

    $index->addField($title_field);
    $index->save();

    $backend = $server->getBackend();
    assert($backend instanceof SqliteFts5);

    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Running and jumping exercises'],
        'body' => [''],
        'status' => [TRUE],
        'category' => [''],
      ],
      'entity:node/2' => [
        'title' => ['The runner jumped over'],
        'body' => [''],
        'status' => [TRUE],
        'category' => [''],
      ],
    ]);

    $backend->indexItems($index, $items);

    // Search for "run" should match "running" and "runner" due to stemming.
    $query = new Query($index);
    $query->keys('run');
    $query->range(0, 10);

    $results = $backend->search($query);

    $this->assertEquals(2, $results->getResultCount());

    // Search for "jump" should match "jumping" and "jumped".
    $query2 = new Query($index);
    $query2->keys('jump');
    $query2->range(0, 10);

    $results2 = $backend->search($query2);

    $this->assertEquals(2, $results2->getResultCount());

    // Cleanup.
    $backend->removeIndex($index);
    $index->delete();
    $server->delete();
  }

  /**
   * Tests trigram tokenizer for substring matching.
   */
  public function testTrigramTokenizer(): void {
    // Create server with trigram tokenizer.
    $server = Server::create([
      'id' => 'trigram_server',
      'name' => 'Trigram Server',
      'backend' => 'search_api_sqlite',
      'backend_config' => [
        'database_path' => $this->testDir . '/databases',
        'min_chars' => 3,
        'tokenizer' => Tokenizer::Trigram->value,
        'trigram_case_sensitive' => FALSE,
        'matching' => MatchingMode::Words->value,
        'highlighting' => [
          'enabled' => FALSE,
          'prefix' => '<em>',
          'suffix' => '</em>',
          'excerpt_length' => 256,
        ],
        'optimization' => [
          'auto_optimize' => FALSE,
          'optimize_threshold' => 1000,
        ],
      ],
    ]);
    $server->save();

    $index = Index::create([
      'id' => 'trigram_index',
      'name' => 'Trigram Index',
      'server' => 'trigram_server',
      'status' => TRUE,
      'tracker_settings' => ['default' => []],
    ]);

    $title_field = new Field($index, 'title');
    $title_field->setType('text');
    $title_field->setLabel('Title');
    $title_field->setPropertyPath('title');

    $index->addField($title_field);
    $index->save();

    $backend = $server->getBackend();
    assert($backend instanceof SqliteFts5);

    // Index items with very distinct content.
    // Use long unique strings to ensure trigrams don't accidentally overlap.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['ALPHA'],
        'body' => [''],
        'status' => [TRUE],
        'category' => [''],
      ],
      'entity:node/2' => [
        'title' => ['OMEGA'],
        'body' => [''],
        'status' => [TRUE],
        'category' => [''],
      ],
    ]);

    $backend->indexItems($index, $items);

    // Search for "LPH" which should match "ALPHA" (contains A-L-P-H-A).
    // Trigrams: ALP, LPH, PHA - so "LPH" should match.
    $query = new Query($index);
    $query->keys('LPH');
    $query->range(0, 10);

    $results = $backend->search($query);

    // With trigram, this should match item 1.
    $this->assertGreaterThanOrEqual(1, $results->getResultCount());
    $this->assertArrayHasKey('entity:node/1', $results->getResultItems());

    // Search for "MEG" which appears in "OMEGA" (O-M-E-G-A).
    // Trigrams: OME, MEG, EGA - so "MEG" should match.
    $query2 = new Query($index);
    $query2->keys('MEG');
    $query2->range(0, 10);

    $results2 = $backend->search($query2);

    $this->assertGreaterThanOrEqual(1, $results2->getResultCount());
    $this->assertArrayHasKey('entity:node/2', $results2->getResultItems());

    // Cleanup.
    $backend->removeIndex($index);
    $index->delete();
    $server->delete();
  }

  /**
   * Tests tokenizer change triggers reindex message.
   */
  public function testTokenizerChangeTriggersReindex(): void {
    $form = [];
    $form_state = new FormState();

    // Change tokenizer from unicode61 to porter.
    $form_state->setValues([
      'tokenizer' => Tokenizer::Porter->value,
      'trigram_case_sensitive' => FALSE,
      'min_chars' => 3,
      'matching' => MatchingMode::Words->value,
      'verbose_logging' => FALSE,
      'log_queries' => FALSE,
      'highlighting' => [
        'enabled' => TRUE,
        'prefix' => '<em>',
        'suffix' => '</em>',
        'excerpt_length' => 256,
      ],
      'optimization' => [
        'auto_optimize' => TRUE,
        'optimize_threshold' => 1000,
      ],
    ]);

    // Index some items first.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Test item'],
        'body' => ['Body text'],
        'status' => [TRUE],
        'category' => ['test'],
      ],
    ]);
    $this->backend->indexItems($this->index, $items);

    // Submit the form with changed tokenizer.
    $this->backend->submitConfigurationForm($form, $form_state);

    // Verify tokenizer was changed.
    $config = $this->backend->getConfiguration();
    $this->assertEquals('porter', $config['tokenizer']);
  }

  /**
   * Tests filtering by string field with equals operator.
   */
  public function testFilterByStringFieldEquals(): void {
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Apple products'],
        'nid' => [1],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/2' => [
        'title' => ['Orange fruits'],
        'nid' => [2],
        'status' => [TRUE],
        'langcode' => ['de'],
      ],
      'entity:node/3' => [
        'title' => ['Banana fruits'],
        'nid' => [3],
        'status' => [TRUE],
        'langcode' => ['de'],
      ],
    ]);

    // Verify index has the expected fields.
    $index_fields = $this->index->getFields();
    $this->assertArrayHasKey('langcode', $index_fields, 'Index should have langcode field. Fields: ' . implode(', ', array_keys($index_fields)));

    $indexed = $this->backend->indexItems($this->index, $items);
    $this->assertEquals(3, count($indexed), 'Should have indexed 3 items');

    // Filter by langcode = 'de'.
    $query = new Query($this->index);
    $query->addCondition('langcode', 'de');
    $query->range(0, 10);

    $results = $this->backend->search($query);

    $this->assertEquals(2, count($results->getResultItems()), 'Should return 2 items filtered by langcode=de');
    $item_ids = array_keys($results->getResultItems());
    $this->assertContains('entity:node/2', $item_ids);
    $this->assertContains('entity:node/3', $item_ids);
    $this->assertNotContains('entity:node/1', $item_ids);
  }

  /**
   * Tests filtering by string field with IN operator.
   */
  public function testFilterByStringFieldIn(): void {
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Item one'],
        'nid' => [1],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/2' => [
        'title' => ['Item two'],
        'nid' => [2],
        'status' => [TRUE],
        'langcode' => ['de'],
      ],
      'entity:node/3' => [
        'title' => ['Item three'],
        'nid' => [3],
        'status' => [TRUE],
        'langcode' => ['fr'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Filter by langcode IN ('en', 'fr').
    $query = new Query($this->index);
    $query->addCondition('langcode', ['en', 'fr'], 'IN');
    $query->range(0, 10);

    $results = $this->backend->search($query);

    $this->assertEquals(2, count($results->getResultItems()));
    $item_ids = array_keys($results->getResultItems());
    $this->assertContains('entity:node/1', $item_ids);
    $this->assertContains('entity:node/3', $item_ids);
    $this->assertNotContains('entity:node/2', $item_ids);
  }

  /**
   * Tests filtering by boolean field.
   */
  public function testFilterByBooleanField(): void {
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Published item'],
        'nid' => [1],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/2' => [
        'title' => ['Unpublished item'],
        'nid' => [2],
        'status' => [FALSE],
        'langcode' => ['en'],
      ],
      'entity:node/3' => [
        'title' => ['Another published'],
        'nid' => [3],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Filter by status = TRUE (stored as integer 1).
    $query = new Query($this->index);
    $query->addCondition('status', TRUE);
    $query->range(0, 10);

    $results = $this->backend->search($query);

    $this->assertEquals(2, count($results->getResultItems()));
    $item_ids = array_keys($results->getResultItems());
    $this->assertContains('entity:node/1', $item_ids);
    $this->assertContains('entity:node/3', $item_ids);
    $this->assertNotContains('entity:node/2', $item_ids);
  }

  /**
   * Tests filtering combined with fulltext search.
   */
  public function testFilterWithFulltextSearch(): void {
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Drupal development guide'],
        'nid' => [1],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/2' => [
        'title' => ['Drupal news article'],
        'nid' => [2],
        'status' => [TRUE],
        'langcode' => ['de'],
      ],
      'entity:node/3' => [
        'title' => ['PHP development guide'],
        'nid' => [3],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Search for "Drupal" AND filter by langcode = 'en'.
    $query = new Query($this->index);
    $query->keys('Drupal');
    $query->addCondition('langcode', 'en');
    $query->range(0, 10);

    $results = $this->backend->search($query);

    // Should only return node/1 (has Drupal AND langcode=en).
    $this->assertEquals(1, count($results->getResultItems()));
    $item_ids = array_keys($results->getResultItems());
    $this->assertContains('entity:node/1', $item_ids);
  }

  /**
   * Tests filtering by multiple conditions (AND logic).
   */
  public function testFilterByMultipleConditions(): void {
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Item one'],
        'nid' => [1],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/2' => [
        'title' => ['Item two'],
        'nid' => [2],
        'status' => [FALSE],
        'langcode' => ['en'],
      ],
      'entity:node/3' => [
        'title' => ['Item three'],
        'nid' => [3],
        'status' => [TRUE],
        'langcode' => ['de'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Filter by langcode = 'en' AND status = TRUE.
    $query = new Query($this->index);
    $query->addCondition('langcode', 'en');
    $query->addCondition('status', TRUE);
    $query->range(0, 10);

    $results = $this->backend->search($query);

    // Should only return node/1 (en AND published).
    $this->assertEquals(1, count($results->getResultItems()));
    $item_ids = array_keys($results->getResultItems());
    $this->assertContains('entity:node/1', $item_ids);
  }

  /**
   * Tests filtering with not equals operator.
   */
  public function testFilterByNotEquals(): void {
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Item one'],
        'nid' => [1],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/2' => [
        'title' => ['Item two'],
        'nid' => [2],
        'status' => [TRUE],
        'langcode' => ['de'],
      ],
      'entity:node/3' => [
        'title' => ['Item three'],
        'nid' => [3],
        'status' => [TRUE],
        'langcode' => ['nl'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Filter by langcode <> 'de'.
    $query = new Query($this->index);
    $query->addCondition('langcode', 'de', '<>');
    $query->range(0, 10);

    $results = $this->backend->search($query);

    $this->assertEquals(2, count($results->getResultItems()));
    $item_ids = array_keys($results->getResultItems());
    $this->assertContains('entity:node/1', $item_ids);
    $this->assertContains('entity:node/3', $item_ids);
    $this->assertNotContains('entity:node/2', $item_ids);
  }

  /**
   * Tests that integer field filtering works correctly.
   *
   * This is important for entity reference fields (taxonomy terms)
   * which store term IDs as integers.
   */
  public function testFilterByIntegerField(): void {
    // Use the existing 'nid' field which is an integer field.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Item one'],
        'nid' => [1],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/5' => [
        'title' => ['Item five'],
        'nid' => [5],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/10' => [
        'title' => ['Item ten'],
        'nid' => [10],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Filter by nid = 5 (integer comparison).
    $query = new Query($this->index);
    $query->addCondition('nid', 5);
    $query->range(0, 10);

    $results = $this->backend->search($query);

    $this->assertEquals(1, count($results->getResultItems()));
    $item_ids = array_keys($results->getResultItems());
    $this->assertContains('entity:node/5', $item_ids);
  }

  /**
   * Tests filtering by integer field with greater than operator.
   */
  public function testFilterByIntegerFieldGreaterThan(): void {
    // Use the existing 'nid' field which is an integer field.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Item 1'],
        'nid' => [1],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/5' => [
        'title' => ['Item 5'],
        'nid' => [5],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/10' => [
        'title' => ['Item 10'],
        'nid' => [10],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Filter by nid > 3.
    $query = new Query($this->index);
    $query->addCondition('nid', 3, '>');
    $query->range(0, 10);

    $results = $this->backend->search($query);

    $this->assertEquals(2, count($results->getResultItems()));
    $item_ids = array_keys($results->getResultItems());
    $this->assertContains('entity:node/5', $item_ids);
    $this->assertContains('entity:node/10', $item_ids);
    $this->assertNotContains('entity:node/1', $item_ids);
  }

  /**
   * Tests filtering simulating facet behavior (entity reference as integer).
   *
   * This is the real-world use case where facets filter by taxonomy term ID.
   * Uses the 'nid' field as a proxy for term_id (both are integers).
   */
  public function testFilterSimulatingFacetBehavior(): void {
    // Use the existing 'nid' field as a proxy for term_id filtering.
    $items = $this->createTestItems([
      'entity:node/3' => [
        'title' => ['Product A'],
        'nid' => [3],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/5' => [
        'title' => ['Product B'],
        'nid' => [5],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/7' => [
        'title' => ['Product C'],
        'nid' => [3],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Simulate facet filter: nid = 3 (like ?field_category[3]=3).
    $query = new Query($this->index);
    $query->addCondition('nid', 3);
    $query->range(0, 10);

    $results = $this->backend->search($query);

    $this->assertEquals(2, count($results->getResultItems()));
    $item_ids = array_keys($results->getResultItems());
    $this->assertContains('entity:node/3', $item_ids);
    $this->assertContains('entity:node/7', $item_ids);
    $this->assertNotContains('entity:node/5', $item_ids);
  }

  /**
   * Tests filtering with string value on integer field (facet behavior).
   *
   * Facets module passes term IDs as strings like '18' even though the field
   * is an integer type. This test ensures string-to-int comparison works.
   */
  public function testFilterIntegerFieldWithStringValue(): void {
    // Use the existing 'nid' field as a proxy for term_id filtering.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Product A'],
        'nid' => [18],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/2' => [
        'title' => ['Product B'],
        'nid' => [25],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/3' => [
        'title' => ['Product C'],
        'nid' => [18],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Simulate facet filter: field_category = '18' (string, not int).
    // This is how facets module passes the value.
    $query = new Query($this->index);
    $query->addCondition('nid', '18');
    $query->range(0, 10);

    $results = $this->backend->search($query);

    // Should find 2 items with nid = 18.
    $this->assertEquals(2, count($results->getResultItems()));
    $item_ids = array_keys($results->getResultItems());
    $this->assertContains('entity:node/1', $item_ids);
    $this->assertContains('entity:node/3', $item_ids);
    $this->assertNotContains('entity:node/2', $item_ids);
  }

  /**
   * Tests facet-style filtering without fulltext search (browsing mode).
   *
   * This mimics the real-world use case where a user clicks a facet
   * without entering a search term. The query has no keys (NULL),
   * only filter conditions.
   */
  public function testFilterWithoutFulltextSearch(): void {
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Product Alpha'],
        'nid' => [18],
        'status' => [TRUE],
        'category' => ['electronics'],
      ],
      'entity:node/2' => [
        'title' => ['Product Beta'],
        'nid' => [25],
        'status' => [TRUE],
        'category' => ['electronics'],
      ],
      'entity:node/3' => [
        'title' => ['Product Gamma'],
        'nid' => [18],
        'status' => [TRUE],
        'category' => ['clothing'],
      ],
      'entity:node/4' => [
        'title' => ['Product Delta'],
        'nid' => [30],
        'status' => [TRUE],
        'category' => ['electronics'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Create query WITHOUT keys (simulates browsing without search term).
    $query = new Query($this->index);
    // Note: We don't call $query->keys() - this simulates browsing mode.
    $query->addCondition('category', 'electronics');
    $query->range(0, 10);

    $results = $this->backend->search($query);

    // Should return all 3 electronics items.
    $this->assertEquals(3, count($results->getResultItems()));
    $item_ids = array_keys($results->getResultItems());
    $this->assertContains('entity:node/1', $item_ids);
    $this->assertContains('entity:node/2', $item_ids);
    $this->assertContains('entity:node/4', $item_ids);
    $this->assertNotContains('entity:node/3', $item_ids);
  }

  /**
   * Tests filtering with IN operator on integer field (multiple facet values).
   */
  public function testFilterByIntegerFieldIn(): void {
    // Use the existing 'nid' field.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Item 1'],
        'nid' => [1],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/2' => [
        'title' => ['Item 2'],
        'nid' => [2],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/3' => [
        'title' => ['Item 3'],
        'nid' => [3],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
      'entity:node/4' => [
        'title' => ['Item 4'],
        'nid' => [4],
        'status' => [TRUE],
        'langcode' => ['en'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Filter by nid IN (1, 3) - simulating OR facet.
    $query = new Query($this->index);
    $query->addCondition('nid', [1, 3], 'IN');
    $query->range(0, 10);

    $results = $this->backend->search($query);

    $this->assertEquals(2, count($results->getResultItems()));
    $item_ids = array_keys($results->getResultItems());
    $this->assertContains('entity:node/1', $item_ids);
    $this->assertContains('entity:node/3', $item_ids);
  }

  /**
   * Tests OR facet operator shows counts without that facet's filter.
   *
   * When using OR operator on a facet, the counts should reflect all items
   * that match the search WITHOUT the filter for that specific facet field.
   * This allows users to see what other options are available.
   */
  public function testOrFacetOperator(): void {
    // Create items with different categories and langcodes.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Drupal development guide'],
        'body' => ['Learn Drupal development'],
        'status' => [TRUE],
        'category' => ['development'],
        'langcode' => ['en'],
      ],
      'entity:node/2' => [
        'title' => ['Drupal theming tutorial'],
        'body' => ['Drupal theme creation'],
        'status' => [TRUE],
        'category' => ['theming'],
        'langcode' => ['en'],
      ],
      'entity:node/3' => [
        'title' => ['Drupal module guide'],
        'body' => ['Building Drupal modules'],
        'status' => [TRUE],
        'category' => ['development'],
        'langcode' => ['de'],
      ],
      'entity:node/4' => [
        'title' => ['Drupal news article'],
        'body' => ['Latest Drupal news'],
        'status' => [TRUE],
        'category' => ['news'],
        'langcode' => ['en'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Search for "Drupal" with a filter on category='development'.
    // Use OR facet on category field.
    $query = new Query($this->index);
    $query->keys('Drupal');

    // Add condition with tag for the facet field.
    $condition_group = $query->createConditionGroup('AND', ['facet:category']);
    $condition_group->addCondition('category', 'development');

    $query->addConditionGroup($condition_group);

    $query->setOption('search_api_facets', [
      'category_facet' => [
        'field' => 'category',
        'limit' => 10,
        'min_count' => 1,
        'missing' => FALSE,
        'operator' => 'or',
      ],
    ]);
    $query->range(0, 10);

    $results = $this->backend->search($query);

    // Results should only contain items with category='development'.
    $this->assertEquals(2, $results->getResultCount());
    $item_ids = array_keys($results->getResultItems());
    $this->assertContains('entity:node/1', $item_ids);
    $this->assertContains('entity:node/3', $item_ids);

    // But OR facet should show counts for ALL categories (without the filter).
    // All 4 items match "Drupal", so facet should show:
    // - development: 2
    // - theming: 1
    // - news: 1.
    $facets = $results->getExtraData('search_api_facets');
    $this->assertIsArray($facets);
    $this->assertArrayHasKey('category_facet', $facets);

    // Extract facet values and counts.
    $category_facet = $facets['category_facet'];
    $facet_counts = [];
    foreach ($category_facet as $facet_item) {
      // Filter format is '"value"'.
      $value = trim((string) $facet_item['filter'], '"');
      $facet_counts[$value] = $facet_item['count'];
    }

    // OR facet should show all categories from the unfiltered result set.
    $this->assertArrayHasKey('development', $facet_counts);
    $this->assertArrayHasKey('theming', $facet_counts);
    $this->assertArrayHasKey('news', $facet_counts);

    // Counts should reflect all Drupal items, not just filtered ones.
    $this->assertEquals(2, $facet_counts['development']);
    $this->assertEquals(1, $facet_counts['theming']);
    $this->assertEquals(1, $facet_counts['news']);
  }

  /**
   * Tests AND facet operator shows counts from filtered results.
   *
   * When using AND operator (default), facet counts reflect the filtered
   * result set - only items that match ALL conditions.
   */
  public function testAndFacetOperator(): void {
    // Create items with different categories and langcodes.
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Drupal development guide'],
        'body' => ['Learn Drupal development'],
        'status' => [TRUE],
        'category' => ['development'],
        'langcode' => ['en'],
      ],
      'entity:node/2' => [
        'title' => ['Drupal theming tutorial'],
        'body' => ['Drupal theme creation'],
        'status' => [TRUE],
        'category' => ['theming'],
        'langcode' => ['en'],
      ],
      'entity:node/3' => [
        'title' => ['Drupal module guide'],
        'body' => ['Building Drupal modules'],
        'status' => [TRUE],
        'category' => ['development'],
        'langcode' => ['de'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Search for "Drupal" with a filter on langcode='en'.
    // Use AND facet (default) on category field.
    $query = new Query($this->index);
    $query->keys('Drupal');
    $query->addCondition('langcode', 'en');

    $query->setOption('search_api_facets', [
      'category_facet' => [
        'field' => 'category',
        'limit' => 10,
        'min_count' => 1,
        'missing' => FALSE,
        'operator' => 'and',
      ],
    ]);
    $query->range(0, 10);

    $results = $this->backend->search($query);

    // Results filtered by langcode='en' - only 2 items.
    $this->assertEquals(2, $results->getResultCount());

    // AND facet should show counts from filtered results only.
    $facets = $results->getExtraData('search_api_facets');
    $this->assertIsArray($facets);
    $this->assertArrayHasKey('category_facet', $facets);

    $category_facet = $facets['category_facet'];
    $facet_counts = [];
    foreach ($category_facet as $facet_item) {
      $value = trim((string) $facet_item['filter'], '"');
      $facet_counts[$value] = $facet_item['count'];
    }

    // Should only show categories from en-language items.
    $this->assertArrayHasKey('development', $facet_counts);
    $this->assertArrayHasKey('theming', $facet_counts);
    $this->assertEquals(1, $facet_counts['development']);
    $this->assertEquals(1, $facet_counts['theming']);
  }

  /**
   * Tests mixed AND and OR facets in the same query.
   */
  public function testMixedAndOrFacets(): void {
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Drupal dev English'],
        'body' => ['Drupal content'],
        'status' => [TRUE],
        'category' => ['development'],
        'langcode' => ['en'],
      ],
      'entity:node/2' => [
        'title' => ['Drupal theme English'],
        'body' => ['Drupal theming'],
        'status' => [TRUE],
        'category' => ['theming'],
        'langcode' => ['en'],
      ],
      'entity:node/3' => [
        'title' => ['Drupal dev German'],
        'body' => ['Drupal entwicklung'],
        'status' => [TRUE],
        'category' => ['development'],
        'langcode' => ['de'],
      ],
      'entity:node/4' => [
        'title' => ['Drupal news French'],
        'body' => ['Drupal nouvelles'],
        'status' => [TRUE],
        'category' => ['news'],
        'langcode' => ['fr'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Filter by category='development' (tagged for OR facet).
    $query = new Query($this->index);
    $query->keys('Drupal');

    $condition_group = $query->createConditionGroup('AND', ['facet:category']);
    $condition_group->addCondition('category', 'development');

    $query->addConditionGroup($condition_group);

    $query->setOption('search_api_facets', [
      // OR facet on category - should show all categories.
      'category_facet' => [
        'field' => 'category',
        'limit' => 10,
        'min_count' => 1,
        'operator' => 'or',
      ],
      // AND facet on langcode - should only show languages from filtered set.
      'langcode_facet' => [
        'field' => 'langcode',
        'limit' => 10,
        'min_count' => 1,
        'operator' => 'and',
      ],
    ]);
    $query->range(0, 10);

    $results = $this->backend->search($query);

    // Results filtered to category='development' - 2 items (en and de).
    $this->assertEquals(2, $results->getResultCount());

    $facets = $results->getExtraData('search_api_facets');

    // OR facet (category) should show all 3 categories from unfiltered search.
    $category_counts = [];
    foreach ($facets['category_facet'] as $item) {
      $category_counts[trim((string) $item['filter'], '"')] = $item['count'];
    }

    $this->assertCount(3, $category_counts);
    $this->assertEquals(2, $category_counts['development']);
    $this->assertEquals(1, $category_counts['theming']);
    $this->assertEquals(1, $category_counts['news']);

    // AND facet (langcode) should only show en and de (from filtered results).
    $langcode_counts = [];
    foreach ($facets['langcode_facet'] as $item) {
      $langcode_counts[trim((string) $item['filter'], '"')] = $item['count'];
    }

    $this->assertCount(2, $langcode_counts);
    $this->assertArrayHasKey('en', $langcode_counts);
    $this->assertArrayHasKey('de', $langcode_counts);
    // Fr should NOT be in AND facet results.
    $this->assertArrayNotHasKey('fr', $langcode_counts);
  }

  /**
   * Tests that multiple OR facets sharing the same item set are batched.
   *
   * When multiple OR facets exclude the same tag (or have no active filter),
   * they should be calculated together in a single query for efficiency.
   */
  public function testMultipleOrFacetsBatching(): void {
    $items = $this->createTestItems([
      'entity:node/1' => [
        'title' => ['Drupal dev guide'],
        'body' => ['Learn Drupal'],
        'status' => [TRUE],
        'category' => ['development'],
        'langcode' => ['en'],
      ],
      'entity:node/2' => [
        'title' => ['Drupal theme tutorial'],
        'body' => ['Drupal theming'],
        'status' => [TRUE],
        'category' => ['theming'],
        'langcode' => ['de'],
      ],
      'entity:node/3' => [
        'title' => ['Drupal module guide'],
        'body' => ['Building modules'],
        'status' => [TRUE],
        'category' => ['development'],
        'langcode' => ['fr'],
      ],
      'entity:node/4' => [
        'title' => ['Drupal news'],
        'body' => ['Latest news'],
        'status' => [TRUE],
        'category' => ['news'],
        'langcode' => ['en'],
      ],
    ]);

    $this->backend->indexItems($this->index, $items);

    // Filter by category='development' with tagged condition.
    $query = new Query($this->index);
    $query->keys('Drupal');

    $condition_group = $query->createConditionGroup('AND', ['facet:category']);
    $condition_group->addCondition('category', 'development');

    $query->addConditionGroup($condition_group);

    // Three OR facets: category (filtered), langcode (not filtered),
    // and status (not filtered).
    // langcode and status should be batched together since they share
    // the same item set (filtered by category).
    $query->setOption('search_api_facets', [
      'category_facet' => [
        'field' => 'category',
        'limit' => 10,
        'min_count' => 1,
        'operator' => 'or',
      ],
      'langcode_facet' => [
        'field' => 'langcode',
        'limit' => 10,
        'min_count' => 1,
        'operator' => 'or',
      ],
      'status_facet' => [
        'field' => 'status',
        'limit' => 10,
        'min_count' => 1,
        'operator' => 'or',
      ],
    ]);
    $query->range(0, 10);

    $results = $this->backend->search($query);

    // Results should be filtered to category='development' (2 items).
    $this->assertEquals(2, $results->getResultCount());

    $facets = $results->getExtraData('search_api_facets');

    // Category OR facet excludes its own filter - shows all categories.
    $category_counts = [];
    foreach ($facets['category_facet'] as $item) {
      $category_counts[trim((string) $item['filter'], '"')] = $item['count'];
    }

    $this->assertCount(3, $category_counts);
    $this->assertEquals(2, $category_counts['development']);
    $this->assertEquals(1, $category_counts['theming']);
    $this->assertEquals(1, $category_counts['news']);

    // Langcode OR facet - filtered by category, shows en and fr.
    $langcode_counts = [];
    foreach ($facets['langcode_facet'] as $item) {
      $langcode_counts[trim((string) $item['filter'], '"')] = $item['count'];
    }

    $this->assertCount(2, $langcode_counts);
    $this->assertArrayHasKey('en', $langcode_counts);
    $this->assertArrayHasKey('fr', $langcode_counts);
    // De is only on theming item, not in filtered set.
    $this->assertArrayNotHasKey('de', $langcode_counts);

    // Status OR facet - filtered by category, all 2 items have status=TRUE.
    $this->assertArrayHasKey('status_facet', $facets);
    $this->assertNotEmpty($facets['status_facet']);
    // All filtered items have status=TRUE (stored as 1), so 1 facet value.
    $this->assertCount(1, $facets['status_facet']);
    $this->assertEquals(2, $facets['status_facet'][0]['count']);
  }

  /**
   * Creates test items for indexing.
   *
   * @param array<string, array<string, array<mixed>>> $items_data
   *   Array of item data keyed by item ID.
   *
   * @return \Drupal\search_api\Item\ItemInterface[]
   *   Array of item objects.
   */
  private function createTestItems(array $items_data): array {
    $items = [];

    foreach ($items_data as $item_id => $field_values) {
      $items[$item_id] = $this->createMockItem($item_id, $field_values);
    }

    return $items;
  }

}
