<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Database;

use Drupal\search_api_sqlite\Database\QueryLogger;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Log\LoggerInterface;

/**
 * Tests the QueryLogger service.
 */
#[CoversClass(QueryLogger::class)]
#[Group('search_api_sqlite')]
final class QueryLoggerTest extends UnitTestCase {

  /**
   * The query logger under test.
   */
  private QueryLogger $queryLogger;

  /**
   * The mocked PSR logger.
   */
  private MockObject $logger;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->logger = $this->createMock(LoggerInterface::class);
    $this->queryLogger = new QueryLogger($this->logger);
  }

  /**
   * Tests that logging is disabled by default.
   */
  public function testLoggingDisabledByDefault(): void {
    $this->assertFalse($this->queryLogger->isEnabled());
  }

  /**
   * Tests enabling and disabling logging.
   */
  public function testSetEnabled(): void {
    $this->queryLogger->setEnabled(TRUE);
    $this->assertTrue($this->queryLogger->isEnabled());

    $this->queryLogger->setEnabled(FALSE);
    $this->assertFalse($this->queryLogger->isEnabled());
  }

  /**
   * Tests that log() does nothing when disabled.
   */
  public function testLogDoesNothingWhenDisabled(): void {
    $this->logger->expects($this->never())
      ->method('debug');

    $this->queryLogger->log(
      'SELECT * FROM test',
      [':id' => 1],
      1.5,
      'TestSource'
    );
  }

  /**
   * Tests that log() logs when enabled.
   */
  public function testLogLogsWhenEnabled(): void {
    $this->queryLogger->setEnabled(TRUE);

    $this->logger->expects($this->once())
      ->method('debug')
      ->with(
        '[@source] @sql | Params: @params | Duration: @duration ms',
        $this->callback(fn(array $context): bool => $context['@source'] === 'TestSource'
          && str_contains((string) $context['@sql'], 'SELECT * FROM test')
          && str_contains((string) $context['@params'], ':id')
          && $context['@duration'] === '1.50')
      );

    $this->queryLogger->log(
      'SELECT * FROM test',
      [':id' => 1],
      1.5,
      'TestSource'
    );
  }

  /**
   * Tests that log() normalizes SQL whitespace.
   */
  public function testLogNormalizesWhitespace(): void {
    $this->queryLogger->setEnabled(TRUE);

    $this->logger->expects($this->once())
      ->method('debug')
      ->with(
        $this->anything(),
        $this->callback(fn(array $context): bool =>
            // SQL should have normalized whitespace.
            $context['@sql'] === 'SELECT * FROM test WHERE id = 1')
      );

    $this->queryLogger->log(
      "SELECT *\n  FROM test\n  WHERE id = 1",
      [],
      1.0,
      'Test'
    );
  }

  /**
   * Tests that log() truncates long parameter values.
   */
  public function testLogTruncatesLongParams(): void {
    $this->queryLogger->setEnabled(TRUE);

    $longValue = str_repeat('a', 150);

    $this->logger->expects($this->once())
      ->method('debug')
      ->with(
        $this->anything(),
        $this->callback(fn(array $context): bool =>
            // Long values should be truncated to 100 chars + "...".
            str_contains((string) $context['@params'], '...')
          && !str_contains((string) $context['@params'], $longValue))
      );

    $this->queryLogger->log(
      'SELECT * FROM test',
      [':content' => $longValue],
      1.0,
      'Test'
    );
  }

  /**
   * Tests that log() shows "(none)" for empty params.
   */
  public function testLogShowsNoneForEmptyParams(): void {
    $this->queryLogger->setEnabled(TRUE);

    $this->logger->expects($this->once())
      ->method('debug')
      ->with(
        $this->anything(),
        $this->callback(fn(array $context): bool => $context['@params'] === '(none)')
      );

    $this->queryLogger->log(
      'SELECT * FROM test',
      [],
      1.0,
      'Test'
    );
  }

  /**
   * Tests startTimer() returns a float.
   */
  public function testStartTimerReturnsFloat(): void {
    $start = $this->queryLogger->startTimer();
    $this->assertIsFloat($start);
    $this->assertGreaterThan(0, $start);
  }

  /**
   * Tests endTimer() calculates elapsed time.
   */
  public function testEndTimerCalculatesElapsedTime(): void {
    $start = $this->queryLogger->startTimer();

    // Small delay to ensure measurable time passes.
    usleep(1000);

    $elapsed = $this->queryLogger->endTimer($start);

    $this->assertIsFloat($elapsed);
    $this->assertGreaterThan(0, $elapsed);
  }

  /**
   * Tests timer accuracy is reasonable.
   */
  public function testTimerAccuracy(): void {
    $start = $this->queryLogger->startTimer();

    // Sleep for ~10ms.
    usleep(10000);

    $elapsed = $this->queryLogger->endTimer($start);

    // Should be roughly 10ms, allow for some variance (5-50ms).
    $this->assertGreaterThan(5, $elapsed);
    $this->assertLessThan(50, $elapsed);
  }

}
