<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Search;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\Select;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\QueryLoggerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Search\FacetBuilder;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the FacetBuilder service.
 */
#[CoversClass(FacetBuilder::class)]
#[Group('search_api_sqlite')]
class FacetBuilderTest extends UnitTestCase {

  /**
   * The connection manager mock.
   *
   * @var \Drupal\search_api_sqlite\Database\ConnectionManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected ConnectionManagerInterface $connectionManager;

  /**
   * The schema manager mock.
   *
   * @var \Drupal\search_api_sqlite\Database\SchemaManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected SchemaManagerInterface $schemaManager;

  /**
   * The query logger mock.
   *
   * @var \Drupal\search_api_sqlite\Database\QueryLoggerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected QueryLoggerInterface $queryLogger;

  /**
   * The FacetBuilder instance.
   */
  protected FacetBuilder $facetBuilder;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->connectionManager = $this->createMock(ConnectionManagerInterface::class);
    $this->schemaManager = $this->createMock(SchemaManagerInterface::class);
    $this->queryLogger = $this->createMock(QueryLoggerInterface::class);

    // Set up default schema manager behavior.
    $this->schemaManager->method('getFieldDataTableName')
      ->willReturnCallback(fn($index_id): string => 'search_api_sqlite_field_data_' . $index_id);

    // Set up query logger default behavior.
    $this->queryLogger->method('startTimer')->willReturn(0.0);
    $this->queryLogger->method('endTimer')->willReturn(0.001);

    // Default shouldUseTempTable to FALSE (tests use small item sets).
    $this->connectionManager->method('shouldUseTempTable')
      ->willReturnCallback(fn(int $count): bool => $count >= 500);

    $this->facetBuilder = new FacetBuilder(
      $this->connectionManager,
      $this->schemaManager,
      $this->queryLogger,
    );
  }

  /**
   * Creates a mock PDO with prepared statement support.
   *
   * @param array<int, object> $facetResults
   *   The facet results to return from the query.
   *
   * @return \PDO|\PHPUnit\Framework\MockObject\MockObject
   *   The mocked PDO object.
   */
  protected function createMockPdo(array $facetResults = []): \PDO {
    $execCalls = [];

    $mockStatement = $this->createMock(\PDOStatement::class);
    $mockStatement->method('execute')->willReturn(TRUE);
    $mockStatement->method('fetchAll')
      ->with(\PDO::FETCH_OBJ)
      ->willReturn($facetResults);

    $mockPdo = $this->createMock(\PDO::class);
    $mockPdo->method('prepare')->willReturn($mockStatement);
    $mockPdo->method('exec')->willReturnCallback(function ($sql) use (&$execCalls): int {
      $execCalls[] = $sql;
      return 1;
    });

    return $mockPdo;
  }

  /**
   * Creates a mock Drupal database Connection for IN() clause queries.
   *
   * @param array<int, object> $facetResults
   *   The facet results to return from the query.
   *
   * @return \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   *   The mocked Connection object.
   */
  protected function createMockConnection(array $facetResults = []): Connection {
    // Create an anonymous class that implements Traversable.
    // phpcs:ignore Drupal.Files.LineLength.TooLong
    $mockStatement = new readonly class ($facetResults) implements \IteratorAggregate {

      /**
       * Constructs the mock statement.
       *
       * @param array<int, object> $results
       *   The results to return.
       */
      public function __construct(
        private array $results,
      ) {}

      /**
       * Returns an iterator for the results.
       *
       * @return \Traversable<int, object>
       *   The results iterator.
       */
      public function getIterator(): \Traversable {
        return new \ArrayIterator($this->results);
      }

    };

    $mockSelect = $this->createMock(Select::class);
    $mockSelect->method('addField')->willReturnSelf();
    $mockSelect->method('addExpression')->willReturnSelf();
    $mockSelect->method('condition')->willReturnSelf();
    $mockSelect->method('groupBy')->willReturnSelf();
    $mockSelect->method('orderBy')->willReturnSelf();
    $mockSelect->method('execute')->willReturn($mockStatement);
    $mockSelect->method('__toString')->willReturn('SELECT ... FROM field_data');

    $mockConnection = $this->createMock(Connection::class);
    $mockConnection->method('select')->willReturn($mockSelect);

    return $mockConnection;
  }

  /**
   * Tests calculateFacets returns empty array for empty item_ids.
   */
  public function testCalculateFacetsWithEmptyItemIds(): void {
    $result = $this->facetBuilder->calculateFacets('test_index', 'category', []);

    $this->assertSame([], $result);
  }

  /**
   * Tests calculateMultipleFacets returns empty arrays for empty item_ids.
   */
  public function testCalculateMultipleFacetsWithEmptyItemIds(): void {
    $result = $this->facetBuilder->calculateMultipleFacets(
      'test_index',
      [
        'category' => ['limit' => 10],
        'type' => ['limit' => 5],
      ],
      []
    );

    $this->assertSame([
      'category' => [],
      'type' => [],
    ], $result);
  }

  /**
   * Tests single facet calculation.
   */
  public function testCalculateSingleFacet(): void {
    $facetResults = [
      (object) ['field_name' => 'category', 'value' => 'Electronics', 'count' => 10],
      (object) ['field_name' => 'category', 'value' => 'Books', 'count' => 5],
      (object) ['field_name' => 'category', 'value' => 'Clothing', 'count' => 3],
    ];

    // Small item count uses IN() clause path.
    $mockConnection = $this->createMockConnection($facetResults);
    $this->connectionManager->method('getConnection')->willReturn($mockConnection);

    $result = $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      ['item_1', 'item_2', 'item_3'],
      ['limit' => 10]
    );

    $this->assertCount(3, $result);
    $this->assertSame('Electronics', $result[0]['value']);
    $this->assertSame(10, $result[0]['count']);
    $this->assertSame('Books', $result[1]['value']);
    $this->assertSame(5, $result[1]['count']);
    $this->assertSame('Clothing', $result[2]['value']);
    $this->assertSame(3, $result[2]['count']);
  }

  /**
   * Tests multiple facets calculation.
   */
  public function testCalculateMultipleFacets(): void {
    $facetResults = [
      (object) ['field_name' => 'category', 'value' => 'Electronics', 'count' => 10],
      (object) ['field_name' => 'category', 'value' => 'Books', 'count' => 5],
      (object) ['field_name' => 'brand', 'value' => 'Sony', 'count' => 8],
      (object) ['field_name' => 'brand', 'value' => 'Samsung', 'count' => 6],
      (object) ['field_name' => 'brand', 'value' => 'Apple', 'count' => 4],
    ];

    // Small item count uses IN() clause path.
    $mockConnection = $this->createMockConnection($facetResults);
    $this->connectionManager->method('getConnection')->willReturn($mockConnection);

    $result = $this->facetBuilder->calculateMultipleFacets(
      'test_index',
      [
        'category' => ['limit' => 10],
        'brand' => ['limit' => 10],
      ],
      ['item_1', 'item_2', 'item_3']
    );

    $this->assertArrayHasKey('category', $result);
    $this->assertArrayHasKey('brand', $result);
    $this->assertCount(2, $result['category']);
    $this->assertCount(3, $result['brand']);
  }

  /**
   * Tests facet limit is respected.
   */
  public function testFacetLimitIsRespected(): void {
    $facetResults = [
      (object) ['field_name' => 'category', 'value' => 'Electronics', 'count' => 10],
      (object) ['field_name' => 'category', 'value' => 'Books', 'count' => 8],
      (object) ['field_name' => 'category', 'value' => 'Clothing', 'count' => 6],
      (object) ['field_name' => 'category', 'value' => 'Home', 'count' => 4],
      (object) ['field_name' => 'category', 'value' => 'Garden', 'count' => 2],
    ];

    // Small item count uses IN() clause path.
    $mockConnection = $this->createMockConnection($facetResults);
    $this->connectionManager->method('getConnection')->willReturn($mockConnection);

    $result = $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      ['item_1', 'item_2'],
      ['limit' => 3]
    );

    $this->assertCount(3, $result);
    $this->assertSame('Electronics', $result[0]['value']);
    $this->assertSame('Books', $result[1]['value']);
    $this->assertSame('Clothing', $result[2]['value']);
  }

  /**
   * Tests min_count filter is applied.
   */
  public function testMinCountFilterIsApplied(): void {
    $facetResults = [
      (object) ['field_name' => 'category', 'value' => 'Electronics', 'count' => 10],
      (object) ['field_name' => 'category', 'value' => 'Books', 'count' => 5],
      (object) ['field_name' => 'category', 'value' => 'Clothing', 'count' => 2],
      (object) ['field_name' => 'category', 'value' => 'Home', 'count' => 1],
    ];

    // Small item count uses IN() clause path.
    $mockConnection = $this->createMockConnection($facetResults);
    $this->connectionManager->method('getConnection')->willReturn($mockConnection);

    $result = $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      ['item_1', 'item_2'],
      ['limit' => 10, 'min_count' => 3]
    );

    $this->assertCount(2, $result);
    $this->assertSame('Electronics', $result[0]['value']);
    $this->assertSame('Books', $result[1]['value']);
  }

  /**
   * Tests default min_count of 1 is applied.
   */
  public function testDefaultMinCountIsOne(): void {
    $facetResults = [
      (object) ['field_name' => 'category', 'value' => 'Electronics', 'count' => 2],
      (object) ['field_name' => 'category', 'value' => 'Books', 'count' => 1],
      (object) ['field_name' => 'category', 'value' => 'Clothing', 'count' => 0],
    ];

    // Small item count uses IN() clause path.
    $mockConnection = $this->createMockConnection($facetResults);
    $this->connectionManager->method('getConnection')->willReturn($mockConnection);

    $result = $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      ['item_1', 'item_2'],
      ['limit' => 10]
    );

    // Only items with count >= 1 should be returned.
    $this->assertCount(2, $result);
    $this->assertSame('Electronics', $result[0]['value']);
    $this->assertSame('Books', $result[1]['value']);
  }

  /**
   * Tests different options per facet field.
   */
  public function testDifferentOptionsPerFacetField(): void {
    $facetResults = [
      (object) ['field_name' => 'category', 'value' => 'A', 'count' => 10],
      (object) ['field_name' => 'category', 'value' => 'B', 'count' => 8],
      (object) ['field_name' => 'category', 'value' => 'C', 'count' => 1],
      (object) ['field_name' => 'brand', 'value' => 'X', 'count' => 15],
      (object) ['field_name' => 'brand', 'value' => 'Y', 'count' => 3],
    ];

    // Small item count uses IN() clause path.
    $mockConnection = $this->createMockConnection($facetResults);
    $this->connectionManager->method('getConnection')->willReturn($mockConnection);

    $result = $this->facetBuilder->calculateMultipleFacets(
      'test_index',
      [
        'category' => ['limit' => 2, 'min_count' => 5],
        'brand' => ['limit' => 10, 'min_count' => 1],
      ],
      ['item_1', 'item_2']
    );

    // Category: limit 2, min_count 5, so only A(10) and B(8).
    $this->assertCount(2, $result['category']);

    // Brand: limit 10, min_count 1, so X(15) and Y(3).
    $this->assertCount(2, $result['brand']);
  }

  /**
   * Tests that missing field returns empty array.
   */
  public function testMissingFieldReturnsEmptyArray(): void {
    // Return results only for 'category', not 'nonexistent'.
    $facetResults = [
      (object) ['field_name' => 'category', 'value' => 'Electronics', 'count' => 10],
    ];

    // Small item count uses IN() clause path.
    $mockConnection = $this->createMockConnection($facetResults);
    $this->connectionManager->method('getConnection')->willReturn($mockConnection);

    $result = $this->facetBuilder->calculateMultipleFacets(
      'test_index',
      [
        'category' => ['limit' => 10],
        'nonexistent' => ['limit' => 10],
      ],
      ['item_1', 'item_2']
    );

    $this->assertCount(1, $result['category']);
    $this->assertSame([], $result['nonexistent']);
  }

  /**
   * Tests that query logger is called during facet calculation with IN clause.
   */
  public function testQueryLoggerIsCalled(): void {
    $facetResults = [
      (object) ['field_name' => 'category', 'value' => 'Test', 'count' => 5],
    ];

    // Small item count uses IN() clause path.
    $mockConnection = $this->createMockConnection($facetResults);
    $this->connectionManager->method('getConnection')->willReturn($mockConnection);

    $this->queryLogger->expects($this->once())->method('startTimer');
    $this->queryLogger->expects($this->once())->method('endTimer');
    $this->queryLogger->expects($this->once())
      ->method('log')
      ->with(
        $this->stringContains('SELECT'),
        $this->arrayHasKey('field_names'),
        $this->anything(),
        $this->equalTo('FacetBuilder::calculateFacetsWithInClause')
      );

    $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      ['item_1', 'item_2'],
      ['limit' => 10]
    );
  }

  /**
   * Tests setQueryLoggingEnabled delegates to query logger.
   */
  public function testSetQueryLoggingEnabled(): void {
    $this->queryLogger->expects($this->once())
      ->method('setEnabled')
      ->with(TRUE);

    $this->facetBuilder->setQueryLoggingEnabled(TRUE);
  }

  /**
   * Tests that temp table is created and cleaned up for large item sets.
   */
  public function testTempTableCreatedAndCleanedUp(): void {
    $mockStatement = $this->createMock(\PDOStatement::class);
    $mockStatement->method('execute')->willReturn(TRUE);
    $mockStatement->method('fetchAll')
      ->with(\PDO::FETCH_OBJ)
      ->willReturn([]);

    $mockPdo = $this->createMock(\PDO::class);
    $mockPdo->method('prepare')->willReturn($mockStatement);

    $this->connectionManager->method('getPdo')->willReturn($mockPdo);

    // Track calls to createTempItemsTable and dropTempTable.
    $createCalled = FALSE;
    $dropCalled = FALSE;

    $this->connectionManager->method('createTempItemsTable')
      ->willReturnCallback(function () use (&$createCalled): string {
        $createCalled = TRUE;
        return 'temp_facet_results_test_index';
      });

    $this->connectionManager->method('dropTempTable')
      ->willReturnCallback(function () use (&$dropCalled): void {
        $dropCalled = TRUE;
      });

    // Use >= 500 items to trigger temp table path.
    $itemIds = [];
    for ($i = 1; $i <= 500; $i++) {
      $itemIds[] = 'item_' . $i;
    }

    $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      $itemIds,
      ['limit' => 10]
    );

    $this->assertTrue($createCalled, 'createTempItemsTable should be called');
    $this->assertTrue($dropCalled, 'dropTempTable should be called for cleanup');
  }

  /**
   * Tests that large item sets trigger the temp table path.
   *
   * The actual batching is now handled by ConnectionManager.
   * This test verifies FacetBuilder correctly delegates to ConnectionManager.
   */
  public function testLargeItemSetsBatched(): void {
    $mockStatement = $this->createMock(\PDOStatement::class);
    $mockStatement->method('execute')->willReturn(TRUE);
    $mockStatement->method('fetchAll')
      ->with(\PDO::FETCH_OBJ)
      ->willReturn([]);

    $mockPdo = $this->createMock(\PDO::class);
    $mockPdo->method('prepare')->willReturn($mockStatement);

    $this->connectionManager->method('getPdo')->willReturn($mockPdo);

    // Track what item IDs are passed to createTempItemsTable.
    $receivedItemIds = [];

    $this->connectionManager->method('createTempItemsTable')
      ->willReturnCallback(function ($index_id, $item_ids) use (&$receivedItemIds): string {
        $receivedItemIds = $item_ids;
        return 'temp_facet_results_test_index';
      });

    $this->connectionManager->method('dropTempTable');

    // Create 1200 item IDs.
    $itemIds = [];
    for ($i = 1; $i <= 1200; $i++) {
      $itemIds[] = 'item_' . $i;
    }

    $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      $itemIds,
      ['limit' => 10]
    );

    // Verify all 1200 items were passed to createTempItemsTable.
    $this->assertCount(1200, $receivedItemIds, 'All 1200 items should be passed to createTempItemsTable');
  }

  /**
   * Tests facet results with numeric values.
   */
  public function testFacetResultsWithNumericValues(): void {
    $facetResults = [
      (object) ['field_name' => 'price_range', 'value' => '100', 'count' => 10],
      (object) ['field_name' => 'price_range', 'value' => '200', 'count' => 5],
      (object) ['field_name' => 'price_range', 'value' => '300', 'count' => 3],
    ];

    // Small item count uses IN() clause path.
    $mockConnection = $this->createMockConnection($facetResults);
    $this->connectionManager->method('getConnection')->willReturn($mockConnection);

    $result = $this->facetBuilder->calculateFacets(
      'test_index',
      'price_range',
      ['item_1', 'item_2'],
      ['limit' => 10]
    );

    $this->assertCount(3, $result);
    $this->assertSame('100', $result[0]['value']);
    $this->assertSame('200', $result[1]['value']);
    $this->assertSame('300', $result[2]['value']);
  }

  /**
   * Tests zero limit means unlimited.
   */
  public function testZeroLimitMeansUnlimited(): void {
    $facetResults = [];
    for ($i = 1; $i <= 20; $i++) {
      $facetResults[] = (object) [
        'field_name' => 'category',
        'value' => 'Value_' . $i,
        'count' => 21 - $i,
      ];
    }

    // Small item count uses IN() clause path.
    $mockConnection = $this->createMockConnection($facetResults);
    $this->connectionManager->method('getConnection')->willReturn($mockConnection);

    $result = $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      ['item_1', 'item_2'],
      ['limit' => 0]
    );

    // All 20 results should be returned when limit is 0.
    $this->assertCount(20, $result);
  }

  /**
   * Tests default limit of 10 is applied.
   */
  public function testDefaultLimitIsTen(): void {
    $facetResults = [];
    for ($i = 1; $i <= 15; $i++) {
      $facetResults[] = (object) [
        'field_name' => 'category',
        'value' => 'Value_' . $i,
        'count' => 16 - $i,
      ];
    }

    // Small item count uses IN() clause path.
    $mockConnection = $this->createMockConnection($facetResults);
    $this->connectionManager->method('getConnection')->willReturn($mockConnection);

    $result = $this->facetBuilder->calculateFacets(
      'test_index',
      'category',
      ['item_1', 'item_2'],
    // No options, should use default limit of 10.
      []
    );

    // Default limit of 10 should be applied.
    $this->assertCount(10, $result);
  }

}
