<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Utility;

use Drupal\search_api_sqlite\Utility\ColumnNameHelper;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the ColumnNameHelper utility class.
 */
#[CoversClass(ColumnNameHelper::class)]
#[Group('search_api_sqlite')]
final class ColumnNameHelperTest extends UnitTestCase {

  /**
   * Tests sanitize() with various field names.
   *
   * @param string $field_name
   *   The field name to sanitize.
   * @param string $expected
   *   The expected sanitized column name.
   */
  #[DataProvider('sanitizeDataProvider')]
  public function testSanitize(string $field_name, string $expected): void {
    $result = ColumnNameHelper::sanitize($field_name);
    $this->assertEquals($expected, $result);
  }

  /**
   * Data provider for testSanitize().
   *
   * @return array<string, array{string, string}>
   *   Test cases.
   */
  public static function sanitizeDataProvider(): array {
    return [
      'simple field name' => ['title', 'title'],
      'field with underscore' => ['field_title', 'field_title'],
      'field with colon' => ['entity:title', 'entity_title'],
      'field with multiple special chars' => ['field.name:value', 'field_name_value'],
      'field starting with number' => ['123field', 'f_123field'],
      'field with only numbers' => ['12345', 'f_12345'],
      'field with spaces' => ['field name', 'field_name'],
      'field with hyphens' => ['field-name', 'field_name'],
      'uppercase field' => ['TITLE', 'TITLE'],
      'mixed case field' => ['MyFieldName', 'MyFieldName'],
      'empty string' => ['', ''],
      'search api field format' => ['search_api_language', 'search_api_language'],
      'drupal entity field' => ['entity:node/title', 'entity_node_title'],
    ];
  }

  /**
   * Tests that sanitize() returns consistent results.
   */
  public function testSanitizeIsIdempotent(): void {
    $original = 'entity:field:name';
    $first_sanitize = ColumnNameHelper::sanitize($original);
    $second_sanitize = ColumnNameHelper::sanitize($first_sanitize);

    $this->assertEquals($first_sanitize, $second_sanitize);
  }

  /**
   * Tests that sanitize() produces SQLite-safe column names.
   */
  public function testSanitizeProducesSqliteSafeNames(): void {
    $dangerous_names = [
      'DROP TABLE',
      'SELECT * FROM',
      '; DELETE FROM',
      "field'name",
      'field"name',
      'field`name',
    ];

    foreach ($dangerous_names as $name) {
      $result = ColumnNameHelper::sanitize($name);
      // Should only contain alphanumeric and underscore.
      $this->assertMatchesRegularExpression(
        '/^[a-zA-Z_]\w*$|^f_\d\w*$|^$/',
        $result,
        sprintf("Sanitized name '%s' from '%s' is not SQLite-safe", $result, $name)
      );
    }
  }

}
