<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\search_api\IndexInterface;

/**
 * Access check for SQLite-specific index operations.
 *
 * Ensures the index uses the SQLite FTS5 backend before granting access
 * to SQLite-specific operations pages.
 */
final class SqliteIndexAccessCheck implements AccessInterface {

  /**
   * The SQLite backend plugin ID.
   */
  private const string BACKEND_PLUGIN_ID = 'search_api_sqlite';

  /**
   * Checks access for SQLite index operations.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The current user account.
   * @param \Drupal\search_api\IndexInterface|null $search_api_index
   *   The Search API index.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(AccountInterface $account, ?IndexInterface $search_api_index = NULL): AccessResultInterface {
    // No index provided.
    if (!$search_api_index instanceof IndexInterface) {
      return AccessResult::forbidden('No index provided.');
    }

    // Check if user has permission to administer Search API.
    $has_permission = $account->hasPermission('administer search_api');
    if (!$has_permission) {
      return AccessResult::forbidden('User lacks required permission.')
        ->addCacheableDependency($search_api_index)
        ->cachePerPermissions();
    }

    // Check if index has a server.
    if (!$search_api_index->hasValidServer()) {
      return AccessResult::forbidden('Index has no valid server.')
        ->addCacheableDependency($search_api_index);
    }

    // Check if the server uses SQLite backend.
    try {
      $server = $search_api_index->getServerInstance();
      $backend_id = $server->getBackendId();

      if ($backend_id === self::BACKEND_PLUGIN_ID) {
        return AccessResult::allowed()
          ->addCacheableDependency($search_api_index)
          ->addCacheableDependency($server)
          ->cachePerPermissions();
      }

      return AccessResult::forbidden('Index does not use SQLite backend.')
        ->addCacheableDependency($search_api_index)
        ->addCacheableDependency($server);
    }
    catch (\Exception) {
      return AccessResult::forbidden('Could not determine backend.')
        ->addCacheableDependency($search_api_index);
    }
  }

}
