<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Database;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Database;
use Drupal\Core\File\FileSystemInterface;
use Psr\Log\LoggerInterface;

/**
 * Manages SQLite database connections for Search API indexes.
 */
final class ConnectionManager implements ConnectionManagerInterface {

  /**
   * The connection key prefix.
   */
  private const string CONNECTION_PREFIX = 'search_api_sqlite_';

  /**
   * Maximum number of item IDs per INSERT batch.
   *
   * Used when populating the temp table with item IDs.
   * Higher values reduce INSERT overhead but use more memory.
   */
  private const int INSERT_BATCH_SIZE = 500;

  /**
   * Threshold for switching from IN() clause to temp table approach.
   *
   * Below this threshold, use IN() clause (faster for small sets).
   * At or above this threshold, use temp table (more efficient for large sets).
   * SQLite has a default SQLITE_MAX_VARIABLE_NUMBER of 999.
   */
  private const int TEMP_TABLE_THRESHOLD = 500;

  /**
   * Cache of PDO connections.
   *
   * @var array<string, \PDO>
   */
  private array $pdoConnections = [];

  /**
   * Constructs a ConnectionManager instance.
   *
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    private readonly FileSystemInterface $fileSystem,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getConnection(string $index_id): Connection {
    $connection_key = $this->getConnectionKey($index_id);

    // Check if connection already exists.
    if (Database::getConnectionInfo($connection_key)) {
      return Database::getConnection('default', $connection_key);
    }

    // Ensure the database directory exists.
    $this->ensureDatabaseDirectory();

    // Add the connection info.
    $database_path = $this->getDatabasePath($index_id);
    Database::addConnectionInfo($connection_key, 'default', [
      'driver' => 'sqlite',
      'database' => $database_path,
    ]);

    $connection = Database::getConnection('default', $connection_key);

    // Apply performance PRAGMAs via PDO.
    $this->applyPragmas($index_id);

    return $connection;
  }

  /**
   * {@inheritdoc}
   */
  public function getPdo(string $index_id): \PDO {
    if (isset($this->pdoConnections[$index_id])) {
      return $this->pdoConnections[$index_id];
    }

    // Ensure the database directory exists.
    $this->ensureDatabaseDirectory();

    $database_path = $this->getDatabasePath($index_id);
    $pdo = new \PDO('sqlite:' . $database_path);
    $pdo->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);

    // Apply FTS5-specific PRAGMAs for write operations.
    $pdo->exec('PRAGMA journal_mode = WAL');
    $pdo->exec('PRAGMA synchronous = NORMAL');
    $pdo->exec('PRAGMA cache_size = -131072');
    $pdo->exec('PRAGMA temp_store = MEMORY');
    $pdo->exec('PRAGMA mmap_size = 268435456');
    $pdo->exec('PRAGMA busy_timeout = 10000');

    $this->pdoConnections[$index_id] = $pdo;

    return $pdo;
  }

  /**
   * {@inheritdoc}
   */
  public function getReadOnlyPdo(string $index_id, int $busy_timeout = 10000): \PDO {
    $readonly_key = $index_id . '_readonly';

    if (isset($this->pdoConnections[$readonly_key])) {
      return $this->pdoConnections[$readonly_key];
    }

    $database_path = $this->getDatabasePath($index_id);

    if (!file_exists($database_path)) {
      throw new \RuntimeException(sprintf('Database file does not exist: %s', $database_path));
    }

    // Open in read-only mode for better concurrency.
    $pdo = new \PDO('sqlite:' . $database_path, '', '', [
      \PDO::SQLITE_ATTR_OPEN_FLAGS => \PDO::SQLITE_OPEN_READONLY,
    ]);
    $pdo->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);

    // Apply read-optimized PRAGMAs.
    $pdo->exec('PRAGMA query_only = ON');
    $pdo->exec('PRAGMA cache_size = -65536');
    $pdo->exec('PRAGMA temp_store = MEMORY');
    $pdo->exec('PRAGMA mmap_size = 268435456');
    $pdo->exec(sprintf('PRAGMA busy_timeout = %d', $busy_timeout));

    $this->pdoConnections[$readonly_key] = $pdo;

    return $pdo;
  }

  /**
   * {@inheritdoc}
   */
  public function closeConnection(string $index_id): void {
    $connection_key = $this->getConnectionKey($index_id);

    // Close PDO connections (both read-write and read-only).
    if (isset($this->pdoConnections[$index_id])) {
      unset($this->pdoConnections[$index_id]);
    }

    $readonly_key = $index_id . '_readonly';
    if (isset($this->pdoConnections[$readonly_key])) {
      unset($this->pdoConnections[$readonly_key]);
    }

    // Remove Drupal connection.
    if (Database::getConnectionInfo($connection_key)) {
      Database::closeConnection('default', $connection_key);
      Database::removeConnection($connection_key);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getDatabasePath(string $index_id): string {
    $config = $this->configFactory->get('search_api_sqlite.settings');
    $base_path = $config->get('database_path') ?? 'private://search_api_sqlite';

    $real_path = $this->fileSystem->realpath($base_path);
    if (!$real_path) {
      // Fall back to constructing the path manually.
      $private_path = $this->fileSystem->realpath('private://');
      if (!$private_path) {
        throw new \RuntimeException('Private file system is not configured.');
      }

      $real_path = $private_path . '/search_api_sqlite';
    }

    return $real_path . '/' . $index_id . '.sqlite';
  }

  /**
   * {@inheritdoc}
   */
  public function databaseExists(string $index_id): bool {
    $path = $this->getDatabasePath($index_id);
    return file_exists($path);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteDatabase(string $index_id): bool {
    // Close connections first.
    $this->closeConnection($index_id);

    $path = $this->getDatabasePath($index_id);

    $deleted = TRUE;

    // Delete main database file.
    if (file_exists($path)) {
      $deleted = unlink($path);
    }

    // Delete WAL file if it exists.
    $wal_path = $path . '-wal';
    if (file_exists($wal_path)) {
      unlink($wal_path);
    }

    // Delete SHM file if it exists.
    $shm_path = $path . '-shm';
    if (file_exists($shm_path)) {
      unlink($shm_path);
    }

    if ($deleted) {
      $this->logger->info('Deleted SQLite database for index @index', [
        '@index' => $index_id,
      ]);
    }

    return $deleted;
  }

  /**
   * Gets the connection key for an index.
   *
   * @param string $index_id
   *   The Search API index ID.
   *
   * @return string
   *   The connection key.
   */
  private function getConnectionKey(string $index_id): string {
    return self::CONNECTION_PREFIX . $index_id;
  }

  /**
   * Ensures the database directory exists.
   *
   * @throws \RuntimeException
   *   If the directory cannot be created.
   */
  private function ensureDatabaseDirectory(): void {
    $config = $this->configFactory->get('search_api_sqlite.settings');
    $base_path = $config->get('database_path') ?? 'private://search_api_sqlite';

    $real_path = $this->fileSystem->realpath($base_path);
    if ($real_path && is_dir($real_path)) {
      return;
    }

    // Try to create the directory.
    $private_path = $this->fileSystem->realpath('private://');
    if (!$private_path) {
      throw new \RuntimeException('Private file system is not configured. Please configure the private file path in settings.php.');
    }

    $directory = $private_path . '/search_api_sqlite';
    if (!is_dir($directory) && (!mkdir($directory, 0775, TRUE) && !is_dir($directory))) {
      throw new \RuntimeException(sprintf('Failed to create database directory: %s', $directory));
    }
  }

  /**
   * Applies performance PRAGMAs to a connection.
   *
   * @param string $index_id
   *   The Search API index ID.
   */
  private function applyPragmas(string $index_id): void {
    $connection_key = $this->getConnectionKey($index_id);
    $connection = Database::getConnection('default', $connection_key);

    // WAL mode is usually set by Drupal core for SQLite, but we ensure it.
    // Use query() for PRAGMA statements as they don't work with prepared
    // statements in some contexts.
    $connection->query('PRAGMA synchronous = NORMAL');
    $connection->query('PRAGMA cache_size = -131072');
    $connection->query('PRAGMA temp_store = MEMORY');
    $connection->query('PRAGMA mmap_size = 268435456');
    $connection->query('PRAGMA busy_timeout = 5000');
  }

  /**
   * {@inheritdoc}
   */
  public function createTempItemsTable(string $index_id, array $item_ids, string $purpose = 'items'): string {
    $pdo = $this->getPdo($index_id);

    // Use index_id and purpose in table name to avoid conflicts.
    $safe_index = preg_replace('/[^a-z0-9_]/', '_', $index_id);
    $safe_purpose = preg_replace('/[^a-z0-9_]/', '_', $purpose);
    $temp_table = 'temp_' . $safe_purpose . '_' . $safe_index;

    // Create temp table (will be auto-dropped when connection closes).
    $pdo->exec(sprintf(
      'CREATE TEMP TABLE IF NOT EXISTS %s (item_id TEXT PRIMARY KEY)',
      $temp_table
    ));

    // Clear any existing data (in case of reuse within same connection).
    $pdo->exec(sprintf('DELETE FROM %s', $temp_table));

    // Batch insert item IDs for efficiency.
    $batches = array_chunk($item_ids, self::INSERT_BATCH_SIZE);

    foreach ($batches as $batch) {
      $placeholders = implode(', ', array_fill(0, count($batch), '(?)'));
      $sql = sprintf('INSERT OR IGNORE INTO %s (item_id) VALUES %s', $temp_table, $placeholders);
      $stmt = $pdo->prepare($sql);
      $stmt->execute($batch);
    }

    return $temp_table;
  }

  /**
   * {@inheritdoc}
   */
  public function dropTempTable(string $index_id, string $table_name): void {
    $pdo = $this->getPdo($index_id);
    $pdo->exec(sprintf('DROP TABLE IF EXISTS %s', $table_name));
  }

  /**
   * {@inheritdoc}
   */
  public function shouldUseTempTable(int $count): bool {
    return $count >= self::TEMP_TABLE_THRESHOLD;
  }

  /**
   * {@inheritdoc}
   */
  public function getTempTableThreshold(): int {
    return self::TEMP_TABLE_THRESHOLD;
  }

}
