<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\search_api\IndexInterface;
use Drupal\search_api\Query\QueryInterface;

/**
 * Interface for handling search query conditions.
 *
 * Processes Search API query conditions and filters item IDs accordingly.
 * Supports nested condition groups with AND/OR conjunctions.
 */
interface ConditionHandlerInterface {

  /**
   * Applies query conditions to filter item IDs.
   *
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The search query containing conditions.
   * @param string $index_id
   *   The index ID.
   * @param array<string> $item_ids
   *   The item IDs to filter.
   * @param array<string, mixed> $backend_config
   *   The backend configuration.
   *
   * @return array<string>
   *   The filtered item IDs.
   */
  public function applyConditions(QueryInterface $query, string $index_id, array $item_ids, array $backend_config = []): array;

  /**
   * Gets the value column name for a field.
   *
   * Determines which column (value_string, value_integer, value_decimal)
   * should be used for querying based on the field type.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The search index.
   * @param string $field_name
   *   The field name.
   *
   * @return string
   *   The value column name.
   */
  public function getValueColumnForField(IndexInterface $index, string $field_name): string;

  /**
   * Applies query conditions while excluding groups with specific tags.
   *
   * This is used for OR facets, where we need to calculate facet counts
   * without applying the filter for that specific facet field.
   *
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The search query containing conditions.
   * @param string $index_id
   *   The index ID.
   * @param array<string> $item_ids
   *   The item IDs to filter.
   * @param array<string> $exclude_tags
   *   Tags of condition groups to exclude from filtering.
   * @param array<string, mixed> $backend_config
   *   The backend configuration.
   *
   * @return array<string>
   *   The filtered item IDs.
   */
  public function applyConditionsExcludingTags(
    QueryInterface $query,
    string $index_id,
    array $item_ids,
    array $exclude_tags,
    array $backend_config = [],
  ): array;

}
