<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;

/**
 * Provides search result highlighting using FTS5 functions.
 */
final readonly class Highlighter implements HighlighterInterface {

  /**
   * Default highlight options.
   *
   * @var array<string, mixed>
   */
  private const array DEFAULT_OPTIONS = [
    'prefix' => '<mark>',
    'suffix' => '</mark>',
    'excerpt_length' => 64,
  ];

  /**
   * Constructs a Highlighter instance.
   *
   * @param \Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface $fts5QueryRunner
   *   The FTS5 query runner.
   * @param \Drupal\search_api_sqlite\Database\SchemaManagerInterface $schemaManager
   *   The schema manager.
   */
  public function __construct(
    private Fts5QueryRunnerInterface $fts5QueryRunner,
    private SchemaManagerInterface $schemaManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getExcerpts(
    string $index_id,
    string $match_query,
    array $item_ids,
    array $columns,
    array $options = [],
  ): array {
    if ($item_ids === [] || $columns === [] || ($match_query === '' || $match_query === '0')) {
      return [];
    }

    $options = array_merge(self::DEFAULT_OPTIONS, $options);
    $fts_table = $this->schemaManager->getFtsTableName($index_id);

    $results = [];

    foreach ($columns as $column) {
      $snippets = $this->fts5QueryRunner->snippet(
        $index_id,
        $fts_table,
        $match_query,
        $column,
        $item_ids,
        $options['prefix'],
        $options['suffix'],
        $options['excerpt_length']
      );

      foreach ($snippets as $item_id => $snippet) {
        if (!isset($results[$item_id])) {
          $results[$item_id] = [];
        }

        $results[$item_id][$column] = $snippet;
      }
    }

    return $results;
  }

}
