<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Index;

use Drupal\Core\State\StateInterface;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Item\FieldInterface;
use Drupal\search_api\Item\ItemInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Index\FieldTypeMapperInterface;
use Drupal\search_api_sqlite\Index\Indexer;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Log\LoggerInterface;

/**
 * Tests the Indexer service.
 */
#[CoversClass(Indexer::class)]
#[Group('search_api_sqlite')]
final class IndexerTest extends UnitTestCase {

  /**
   * The indexer under test.
   */
  private Indexer $indexer;

  /**
   * The mocked connection manager.
   */
  private ConnectionManagerInterface&MockObject $connectionManager;

  /**
   * The mocked schema manager.
   */
  private SchemaManagerInterface&MockObject $schemaManager;

  /**
   * The mocked field type mapper.
   */
  private FieldTypeMapperInterface&MockObject $fieldTypeMapper;

  /**
   * The mocked FTS5 query runner.
   */
  private Fts5QueryRunnerInterface&MockObject $fts5QueryRunner;

  /**
   * The mocked state service.
   */
  private StateInterface&MockObject $state;

  /**
   * The mocked logger.
   */
  private LoggerInterface&MockObject $logger;

  /**
   * The mocked PDO connection.
   */
  private \PDO&MockObject $pdo;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->connectionManager = $this->createMock(ConnectionManagerInterface::class);
    $this->schemaManager = $this->createMock(SchemaManagerInterface::class);
    $this->fieldTypeMapper = $this->createMock(FieldTypeMapperInterface::class);
    $this->fts5QueryRunner = $this->createMock(Fts5QueryRunnerInterface::class);
    $this->state = $this->createMock(StateInterface::class);
    $this->logger = $this->createMock(LoggerInterface::class);

    $this->pdo = $this->createMock(\PDO::class);

    $this->connectionManager->method('getPdo')
      ->willReturn($this->pdo);

    $this->schemaManager->method('getFtsTableName')
      ->willReturnCallback(fn(string $id): string => $id . '_fts');
    $this->schemaManager->method('getFieldDataTableName')
      ->willReturnCallback(fn(string $id): string => $id . '_field_data');
    $this->schemaManager->method('getItemsTableName')
      ->willReturnCallback(fn(string $id): string => $id . '_items');

    $this->indexer = new Indexer(
      $this->connectionManager,
      $this->schemaManager,
      $this->fieldTypeMapper,
      $this->fts5QueryRunner,
      $this->state,
      $this->logger
    );
  }

  /**
   * Provides data for sanitizeColumnName tests.
   *
   * @return array<string, array{string, string}>
   *   Test data with input and expected output.
   */
  public static function sanitizeColumnNameDataProvider(): array {
    return [
      'simple field name' => ['title', 'title'],
      'field with dots' => ['field.name', 'field_name'],
      'field with dashes' => ['field-name', 'field_name'],
      'field starting with number' => ['1field', 'f_1field'],
      'field with special chars' => ['field@name!test', 'field_name_test'],
      'mixed case preserved' => ['MyField', 'MyField'],
      'underscores preserved' => ['my_field_name', 'my_field_name'],
    ];
  }

  /**
   * Tests indexItems() returns empty array for empty input.
   */
  public function testIndexItemsReturnsEmptyForEmptyInput(): void {
    $index = $this->createMock(IndexInterface::class);

    $result = $this->indexer->indexItems($index, [], []);

    $this->assertSame([], $result);
  }

  /**
   * Tests indexItems() indexes fulltext and filter fields correctly.
   */
  public function testIndexItemsIndexesFieldsCorrectly(): void {
    $index = $this->createMockIndex();
    $item = $this->createMockItem('entity:node/1', 'entity:node', 'en');

    // Set up PDO exec for transaction.
    $this->pdo->expects($this->exactly(2))
      ->method('exec')
      ->willReturnCallback(function (string $sql): int|false {
        // Accept BEGIN and COMMIT.
        if (str_contains($sql, 'BEGIN') || str_contains($sql, 'COMMIT')) {
          return 0;
        }

        return FALSE;
      });

    // Mock all prepared statements.
    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')->willReturn($stmt);

    // Track change count.
    $this->state->expects($this->once())
      ->method('set')
      ->with(
        'search_api_sqlite.changes.test_index',
        $this->callback(fn($v): bool => $v >= 1)
      );

    $result = $this->indexer->indexItems($index, ['entity:node/1' => $item], []);

    $this->assertSame(['entity:node/1'], $result);
  }

  /**
   * Tests deleteItems() with empty array does nothing.
   */
  public function testDeleteItemsWithEmptyArrayDoesNothing(): void {
    $index = $this->createMock(IndexInterface::class);

    $this->pdo->expects($this->never())->method('exec');

    $this->indexer->deleteItems($index, []);
  }

  /**
   * Tests deleteItems() deletes from all tables.
   */
  public function testDeleteItemsDeletesFromAllTables(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    // Mock transaction.
    $this->pdo->expects($this->exactly(2))
      ->method('exec')
      ->willReturn(0);

    // Mock delete statements.
    $deleteStmt = $this->createMock(\PDOStatement::class);
    $deleteStmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')
      ->with($this->stringContains('DELETE FROM'))
      ->willReturn($deleteStmt);

    $this->state->expects($this->once())
      ->method('set')
      ->with(
        'search_api_sqlite.changes.test_index',
        $this->greaterThanOrEqual(2)
      );

    $this->indexer->deleteItems($index, ['item_1', 'item_2']);
  }

  /**
   * Tests deleteAllItems() without datasource deletes all tables.
   */
  public function testDeleteAllItemsDeletesAllTables(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    // Expect BEGIN, 3x DELETE, COMMIT.
    $this->pdo->expects($this->exactly(5))
      ->method('exec')
      ->willReturn(0);

    $this->state->expects($this->once())
      ->method('delete')
      ->with('search_api_sqlite.changes.test_index');

    $this->indexer->deleteAllItems($index);
  }

  /**
   * Tests deleteAllItems() with datasource only deletes matching items.
   */
  public function testDeleteAllItemsWithDatasourceDeletesMatchingItems(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    // Mock select query to get item IDs by datasource.
    $selectStmt = $this->createMock(\PDOStatement::class);
    $selectStmt->method('execute')->willReturn(TRUE);
    $selectStmt->method('fetchAll')->willReturn(['item_1', 'item_2']);

    // Mock delete statements.
    $deleteStmt = $this->createMock(\PDOStatement::class);
    $deleteStmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')
      ->willReturnCallback(function (string $sql) use ($selectStmt, $deleteStmt): MockObject {
        if (str_contains($sql, 'SELECT')) {
          return $selectStmt;
        }

        return $deleteStmt;
      });

    // Expect BEGIN and COMMIT.
    $this->pdo->expects($this->exactly(2))
      ->method('exec')
      ->willReturn(0);

    $this->indexer->deleteAllItems($index, 'entity:node');
  }

  /**
   * Tests getIndexedItemsCount() returns correct count.
   */
  public function testGetIndexedItemsCountReturnsCorrectCount(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $result = $this->createMock(\PDOStatement::class);
    $result->method('fetchColumn')->willReturn('123');

    $this->pdo->method('query')
      ->with($this->stringContains('SELECT COUNT(*)'))
      ->willReturn($result);

    $count = $this->indexer->getIndexedItemsCount($index);

    $this->assertSame(123, $count);
  }

  /**
   * Tests getIndexedItemsCount() returns zero on query failure.
   */
  public function testGetIndexedItemsCountReturnsZeroOnFailure(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $this->pdo->method('query')->willReturn(FALSE);

    $count = $this->indexer->getIndexedItemsCount($index);

    $this->assertSame(0, $count);
  }

  /**
   * Tests that indexing rolls back transaction on error.
   */
  public function testIndexItemsRollsBackOnError(): void {
    $index = $this->createMockIndex();
    $item = $this->createMockItem('entity:node/1', 'entity:node', 'en');

    // Expect BEGIN, then ROLLBACK on error.
    $execCalls = [];
    $this->pdo->method('exec')
      ->willReturnCallback(function (string $sql) use (&$execCalls): int {
        $execCalls[] = $sql;
        return 0;
      });

    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willThrowException(new \Exception('DB Error'));

    $this->pdo->method('prepare')->willReturn($stmt);

    // Non-locking errors are thrown immediately without logging.
    $this->expectException(\Exception::class);
    $this->expectExceptionMessage('DB Error');

    $this->indexer->indexItems($index, ['entity:node/1' => $item], []);
  }

  /**
   * Tests indexItems retries on database locked error.
   */
  public function testIndexItemsRetriesOnDatabaseLocked(): void {
    $index = $this->createMockIndex();
    $item = $this->createMockItem('entity:node/1', 'entity:node', 'en');

    $this->pdo->method('exec')->willReturn(0);

    $callCount = 0;
    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')
      ->willReturnCallback(function () use (&$callCount): bool {
        $callCount++;
        if ($callCount < 3) {
          throw new \Exception('database is locked');
        }

        return TRUE;
      });

    $this->pdo->method('prepare')->willReturn($stmt);

    // Logger should log warning for retries.
    $this->logger->expects($this->atLeastOnce())
      ->method('warning')
      ->with($this->stringContains('Database locked'));

    $result = $this->indexer->indexItems($index, ['entity:node/1' => $item], []);

    $this->assertSame(['entity:node/1'], $result);
  }

  /**
   * Tests indexItems fails after max retries on database locked.
   */
  public function testIndexItemsFailsAfterMaxRetries(): void {
    $index = $this->createMockIndex();
    $item = $this->createMockItem('entity:node/1', 'entity:node', 'en');

    $this->pdo->method('exec')->willReturn(0);

    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')
      ->willThrowException(new \Exception('database is locked'));

    $this->pdo->method('prepare')->willReturn($stmt);

    // Warning is logged for retries (max_retries - 1 times = 2 for config).
    // Pass config with max_retries = 3 for predictable test behavior.
    $this->logger->expects($this->exactly(2))
      ->method('warning');

    $this->expectException(\Exception::class);
    $this->expectExceptionMessage('database is locked');

    $this->indexer->indexItems($index, ['entity:node/1' => $item], [
      'concurrency' => [
        'max_retries' => 3,
        'retry_delay' => 10,
      ],
    ]);
  }

  /**
   * Tests indexItems with SQLITE_BUSY error triggers retry.
   */
  public function testIndexItemsRetriesOnSqliteBusy(): void {
    $index = $this->createMockIndex();
    $item = $this->createMockItem('entity:node/1', 'entity:node', 'en');

    $this->pdo->method('exec')->willReturn(0);

    $callCount = 0;
    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')
      ->willReturnCallback(function () use (&$callCount): bool {
        $callCount++;
        if ($callCount === 1) {
          throw new \Exception('SQLITE_BUSY');
        }

        return TRUE;
      });

    $this->pdo->method('prepare')->willReturn($stmt);

    $result = $this->indexer->indexItems($index, ['entity:node/1' => $item], []);

    $this->assertSame(['entity:node/1'], $result);
  }

  /**
   * Tests getIndexedItemsCount returns zero on exception.
   */
  public function testGetIndexedItemsCountReturnsZeroOnException(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $this->pdo->method('query')->willThrowException(new \Exception('Query failed'));

    $this->logger->expects($this->once())
      ->method('error')
      ->with($this->stringContains('Error getting indexed items count'));

    $count = $this->indexer->getIndexedItemsCount($index);

    $this->assertSame(0, $count);
  }

  /**
   * Tests indexItems with multiple items indexes all.
   */
  public function testIndexItemsWithMultipleItems(): void {
    $index = $this->createMockIndex();
    $item1 = $this->createMockItem('entity:node/1', 'entity:node', 'en');
    $item2 = $this->createMockItem('entity:node/2', 'entity:node', 'en');
    $item3 = $this->createMockItem('entity:node/3', 'entity:node', 'fr');

    $this->pdo->method('exec')->willReturn(0);

    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')->willReturn($stmt);

    $result = $this->indexer->indexItems($index, [
      'entity:node/1' => $item1,
      'entity:node/2' => $item2,
      'entity:node/3' => $item3,
    ], []);

    $this->assertCount(3, $result);
    $this->assertContains('entity:node/1', $result);
    $this->assertContains('entity:node/2', $result);
    $this->assertContains('entity:node/3', $result);
  }

  /**
   * Tests indexItems tracks changes with state.
   */
  public function testIndexItemsTracksChangesWithState(): void {
    $index = $this->createMockIndex();
    $item = $this->createMockItem('entity:node/1', 'entity:node', 'en');

    $this->pdo->method('exec')->willReturn(0);

    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')->willReturn($stmt);

    $this->state->expects($this->once())
      ->method('get')
      ->with('search_api_sqlite.changes.test_index', 0)
      ->willReturn(5);

    $this->state->expects($this->once())
      ->method('set')
      ->with('search_api_sqlite.changes.test_index', 6);

    $this->indexer->indexItems($index, ['entity:node/1' => $item], []);
  }

  /**
   * Tests deleteItems tracks changes correctly.
   */
  public function testDeleteItemsTracksChanges(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $this->pdo->method('exec')->willReturn(0);

    $deleteStmt = $this->createMock(\PDOStatement::class);
    $deleteStmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')
      ->willReturn($deleteStmt);

    $this->state->expects($this->once())
      ->method('get')
      ->with('search_api_sqlite.changes.test_index', 0)
      ->willReturn(10);

    $this->state->expects($this->once())
      ->method('set')
      ->with('search_api_sqlite.changes.test_index', 13);

    $this->indexer->deleteItems($index, ['item_1', 'item_2', 'item_3']);
  }

  /**
   * Tests deleteItems rolls back on error.
   */
  public function testDeleteItemsRollsBackOnError(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $execCalls = [];
    $this->pdo->method('exec')
      ->willReturnCallback(function (string $sql) use (&$execCalls): int {
        $execCalls[] = $sql;
        return 0;
      });

    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willThrowException(new \Exception('Delete failed'));

    $this->pdo->method('prepare')->willReturn($stmt);

    $this->expectException(\Exception::class);
    $this->expectExceptionMessage('Delete failed');

    $this->indexer->deleteItems($index, ['item_1']);

    $this->assertContains('ROLLBACK', $execCalls);
  }

  /**
   * Tests deleteAllItems rolls back on error.
   */
  public function testDeleteAllItemsRollsBackOnError(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $callCount = 0;
    $this->pdo->method('exec')
      ->willReturnCallback(function (string $sql) use (&$callCount): int {
        $callCount++;
        // BEGIN succeeds, then DELETE fails.
        if ($callCount === 1) {
          return 0;
        }

        throw new \Exception('Delete failed');
      });

    $this->expectException(\Exception::class);
    $this->expectExceptionMessage('Delete failed');

    $this->indexer->deleteAllItems($index);
  }

  /**
   * Tests indexItems with item that has no field returns null.
   */
  public function testIndexItemsWithMissingField(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $statusField = $this->createMock(FieldInterface::class);
    $statusField->method('getType')->willReturn('boolean');

    $index->method('getFields')->willReturn([
      'status' => $statusField,
    ]);

    $this->fieldTypeMapper->method('isFulltextType')
      ->willReturn(FALSE);

    $item = $this->createMock(ItemInterface::class);
    $item->method('getId')->willReturn('entity:node/1');
    $item->method('getDatasourceId')->willReturn('entity:node');
    $item->method('getLanguage')->willReturn('en');
    // Return NULL for field lookup - field doesn't exist on item.
    $item->method('getField')->willReturn(NULL);

    $this->pdo->method('exec')->willReturn(0);

    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')->willReturn($stmt);

    $result = $this->indexer->indexItems($index, ['entity:node/1' => $item], []);

    $this->assertSame(['entity:node/1'], $result);
  }

  /**
   * Tests indexItems logs success message when verbose logging is enabled.
   */
  public function testIndexItemsLogsSuccess(): void {
    $index = $this->createMockIndex();
    $item = $this->createMockItem('entity:node/1', 'entity:node', 'en');

    $this->pdo->method('exec')->willReturn(0);

    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')->willReturn($stmt);

    $this->logger->expects($this->once())
      ->method('info')
      ->with(
        'Indexed @count items for index @index',
        $this->callback(fn($args): bool => $args['@count'] === 1 && $args['@index'] === 'test_index')
      );

    $this->indexer->indexItems($index, ['entity:node/1' => $item], ['verbose_logging' => TRUE]);
  }

  /**
   * Tests indexItems does not log when verbose logging is disabled.
   */
  public function testIndexItemsDoesNotLogWhenVerboseDisabled(): void {
    $index = $this->createMockIndex();
    $item = $this->createMockItem('entity:node/1', 'entity:node', 'en');

    $this->pdo->method('exec')->willReturn(0);

    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')->willReturn($stmt);

    $this->logger->expects($this->never())
      ->method('info');

    $this->indexer->indexItems($index, ['entity:node/1' => $item], ['verbose_logging' => FALSE]);
  }

  /**
   * Tests deleteItems logs success message when verbose logging is enabled.
   *
   * Note: Backend config is set via indexItems call first.
   */
  public function testDeleteItemsLogsSuccess(): void {
    $index = $this->createMockIndex();
    $item = $this->createMockItem('entity:node/1', 'entity:node', 'en');

    $this->pdo->method('exec')->willReturn(0);

    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')->willReturn($stmt);

    // Expect 2 info logs: one for indexing, one for deleting.
    $this->logger->expects($this->exactly(2))
      ->method('info');

    // First call indexItems to set the backend config with verbose_logging.
    $this->indexer->indexItems($index, ['entity:node/1' => $item], ['verbose_logging' => TRUE]);

    $this->indexer->deleteItems($index, ['item_1', 'item_2']);
  }

  /**
   * Tests deleteAllItems logs success message when verbose logging is enabled.
   *
   * Note: Backend config is set via indexItems call first.
   */
  public function testDeleteAllItemsLogsSuccess(): void {
    $index = $this->createMockIndex();
    $item = $this->createMockItem('entity:node/1', 'entity:node', 'en');

    $this->pdo->method('exec')->willReturn(0);

    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')->willReturn($stmt);

    // Expect 2 info logs: one for indexing, one for deleting all.
    $this->logger->expects($this->exactly(2))
      ->method('info');

    // First call indexItems to set the backend config with verbose_logging.
    $this->indexer->indexItems($index, ['entity:node/1' => $item], ['verbose_logging' => TRUE]);

    $this->indexer->deleteAllItems($index);
  }

  /**
   * Tests deleteAllItems with datasource logs datasource.
   *
   * Note: Backend config is set via indexItems call first.
   */
  public function testDeleteAllItemsWithDatasourceLogsCorrectly(): void {
    $index = $this->createMockIndex();
    $item = $this->createMockItem('entity:node/1', 'entity:node', 'en');

    $selectStmt = $this->createMock(\PDOStatement::class);
    $selectStmt->method('execute')->willReturn(TRUE);
    $selectStmt->method('fetchAll')->willReturn([]);

    $this->pdo->method('prepare')
      ->willReturn($selectStmt);

    $this->pdo->method('exec')->willReturn(0);

    // Expect 2 info logs: one for indexing, one for deleting all.
    $this->logger->expects($this->exactly(2))
      ->method('info');

    // First call indexItems to set the backend config with verbose_logging.
    $this->indexer->indexItems($index, ['entity:node/1' => $item], ['verbose_logging' => TRUE]);

    $this->indexer->deleteAllItems($index, 'entity:node');
  }

  /**
   * Tests indexItems with string storage type.
   */
  public function testIndexItemsWithStringStorageType(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $urlField = $this->createMock(FieldInterface::class);
    $urlField->method('getType')->willReturn('uri');

    $index->method('getFields')->willReturn([
      'url' => $urlField,
    ]);

    $this->fieldTypeMapper->method('isFulltextType')
      ->willReturn(FALSE);
    $this->fieldTypeMapper->method('getStorageType')
      ->willReturn('string');
    $this->fieldTypeMapper->method('getFieldDataColumn')
      ->willReturn('value_string');

    $itemField = $this->createMock(FieldInterface::class);

    $item = $this->createMock(ItemInterface::class);
    $item->method('getId')->willReturn('entity:node/1');
    $item->method('getDatasourceId')->willReturn('entity:node');
    $item->method('getLanguage')->willReturn('en');
    $item->method('getField')
      ->willReturnCallback(fn($id): ?MockObject => $id === 'url' ? $itemField : NULL);

    $this->fieldTypeMapper->method('extractFieldValues')
      ->willReturn(['https://example.com']);

    $this->pdo->method('exec')->willReturn(0);

    $executedValues = [];
    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')
      ->willReturnCallback(function ($params) use (&$executedValues): bool {
        $executedValues[] = $params;
        return TRUE;
      });

    $this->pdo->method('prepare')->willReturn($stmt);

    $result = $this->indexer->indexItems($index, ['entity:node/1' => $item], []);

    $this->assertSame(['entity:node/1'], $result);
  }

  /**
   * Tests indexItems with decimal storage type.
   */
  public function testIndexItemsWithDecimalStorageType(): void {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $priceField = $this->createMock(FieldInterface::class);
    $priceField->method('getType')->willReturn('decimal');

    $index->method('getFields')->willReturn([
      'price' => $priceField,
    ]);

    $this->fieldTypeMapper->method('isFulltextType')
      ->willReturn(FALSE);
    $this->fieldTypeMapper->method('getStorageType')
      ->willReturn('decimal');
    $this->fieldTypeMapper->method('getFieldDataColumn')
      ->willReturn('value_decimal');

    $itemField = $this->createMock(FieldInterface::class);

    $item = $this->createMock(ItemInterface::class);
    $item->method('getId')->willReturn('entity:node/1');
    $item->method('getDatasourceId')->willReturn('entity:node');
    $item->method('getLanguage')->willReturn('en');
    $item->method('getField')
      ->willReturnCallback(fn($id): ?MockObject => $id === 'price' ? $itemField : NULL);

    $this->fieldTypeMapper->method('extractFieldValues')
      ->willReturn([19.99]);

    $this->pdo->method('exec')->willReturn(0);

    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')->willReturn($stmt);

    $result = $this->indexer->indexItems($index, ['entity:node/1' => $item], []);

    $this->assertSame(['entity:node/1'], $result);
  }

  /**
   * Tests that auto-optimization triggers when threshold is reached.
   */
  public function testAutoOptimizationTriggersAtThreshold(): void {
    $index = $this->createMockIndex();
    $item = $this->createMockItem('entity:node/1', 'entity:node', 'en');

    $this->pdo->method('exec')->willReturn(0);

    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')->willReturn($stmt);

    // Simulate 999 previous changes (threshold is 1000).
    $this->state->expects($this->atLeastOnce())
      ->method('get')
      ->with('search_api_sqlite.changes.test_index', 0)
      ->willReturn(999);

    // Expect the state to be set to 1000 then reset to 0.
    $this->state->expects($this->exactly(2))
      ->method('set')
      ->willReturnCallback(function (string $key, int $value): void {
        static $call = 0;
        $call++;
        if ($call === 1) {
          // First call: increment to 1000.
          $this->assertEquals(1000, $value);
        }
        else {
          // Second call: reset to 0 after optimization.
          $this->assertEquals(0, $value);
        }
      });

    // Expect optimization to be called.
    $this->fts5QueryRunner->expects($this->once())
      ->method('optimize')
      ->with('test_index', 'test_index_fts');

    // Expect info log for optimization (2 logs: indexing + optimization).
    $this->logger->expects($this->exactly(2))
      ->method('info');

    $backend_config = [
      'verbose_logging' => TRUE,
      'optimization' => [
        'auto_optimize' => TRUE,
        'optimize_threshold' => 1000,
      ],
    ];

    $this->indexer->indexItems($index, ['entity:node/1' => $item], $backend_config);
  }

  /**
   * Tests that auto-optimization does not trigger when disabled.
   */
  public function testAutoOptimizationDisabled(): void {
    $index = $this->createMockIndex();
    $item = $this->createMockItem('entity:node/1', 'entity:node', 'en');

    $this->pdo->method('exec')->willReturn(0);

    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')->willReturn($stmt);

    // Simulate many previous changes.
    $this->state->method('get')
      ->with('search_api_sqlite.changes.test_index', 0)
      ->willReturn(9999);

    // Optimization should NOT be called.
    $this->fts5QueryRunner->expects($this->never())
      ->method('optimize');

    $backend_config = [
      'optimization' => [
        'auto_optimize' => FALSE,
        'optimize_threshold' => 1000,
      ],
    ];

    $this->indexer->indexItems($index, ['entity:node/1' => $item], $backend_config);
  }

  /**
   * Tests that auto-optimization does not trigger below threshold.
   */
  public function testAutoOptimizationBelowThreshold(): void {
    $index = $this->createMockIndex();
    $item = $this->createMockItem('entity:node/1', 'entity:node', 'en');

    $this->pdo->method('exec')->willReturn(0);

    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')->willReturn($stmt);

    // Simulate 500 previous changes (below 1000 threshold).
    $this->state->method('get')
      ->with('search_api_sqlite.changes.test_index', 0)
      ->willReturn(500);

    // Optimization should NOT be called (501 < 1000).
    $this->fts5QueryRunner->expects($this->never())
      ->method('optimize');

    $backend_config = [
      'optimization' => [
        'auto_optimize' => TRUE,
        'optimize_threshold' => 1000,
      ],
    ];

    $this->indexer->indexItems($index, ['entity:node/1' => $item], $backend_config);
  }

  /**
   * Tests that optimization failure is handled gracefully.
   */
  public function testAutoOptimizationFailureIsHandled(): void {
    $index = $this->createMockIndex();
    $item = $this->createMockItem('entity:node/1', 'entity:node', 'en');

    $this->pdo->method('exec')->willReturn(0);

    $stmt = $this->createMock(\PDOStatement::class);
    $stmt->method('execute')->willReturn(TRUE);

    $this->pdo->method('prepare')->willReturn($stmt);

    // Simulate threshold reached.
    $this->state->method('get')
      ->with('search_api_sqlite.changes.test_index', 0)
      ->willReturn(999);

    // Optimization throws an exception.
    $this->fts5QueryRunner->expects($this->once())
      ->method('optimize')
      ->willThrowException(new \RuntimeException('Optimization failed'));

    // Expect warning log.
    $this->logger->expects($this->once())
      ->method('warning');

    $backend_config = [
      'optimization' => [
        'auto_optimize' => TRUE,
        'optimize_threshold' => 1000,
      ],
    ];

    // Should not throw, failure is handled gracefully.
    $result = $this->indexer->indexItems($index, ['entity:node/1' => $item], $backend_config);
    $this->assertSame(['entity:node/1'], $result);
  }

  /**
   * Creates a mock index with fields.
   *
   * @return \Drupal\search_api\IndexInterface&\PHPUnit\Framework\MockObject\MockObject
   *   The mock index.
   */
  private function createMockIndex(): IndexInterface&MockObject {
    $index = $this->createMock(IndexInterface::class);
    $index->method('id')->willReturn('test_index');

    $titleField = $this->createMock(FieldInterface::class);
    $titleField->method('getType')->willReturn('text');

    $statusField = $this->createMock(FieldInterface::class);
    $statusField->method('getType')->willReturn('boolean');

    $index->method('getFields')->willReturn([
      'title' => $titleField,
      'status' => $statusField,
    ]);

    $this->fieldTypeMapper->method('isFulltextType')
      ->willReturnCallback(fn(string $type): bool => $type === 'text');

    $this->fieldTypeMapper->method('getStorageType')
      ->willReturn('integer');

    $this->fieldTypeMapper->method('getFieldDataColumn')
      ->willReturn('value_integer');

    return $index;
  }

  /**
   * Creates a mock search API item.
   *
   * @param string $item_id
   *   The item ID.
   * @param string $datasource
   *   The datasource ID.
   * @param string $language
   *   The language code.
   *
   * @return \Drupal\search_api\Item\ItemInterface&\PHPUnit\Framework\MockObject\MockObject
   *   The mock item.
   */
  private function createMockItem(
    string $item_id,
    string $datasource,
    string $language,
  ): ItemInterface&MockObject {
    $item = $this->createMock(ItemInterface::class);
    $item->method('getId')->willReturn($item_id);
    $item->method('getDatasourceId')->willReturn($datasource);
    $item->method('getLanguage')->willReturn($language);

    $titleItemField = $this->createMock(FieldInterface::class);
    $statusItemField = $this->createMock(FieldInterface::class);

    $item->method('getField')
      ->willReturnCallback(function (string $field_id) use ($titleItemField, $statusItemField): ?MockObject {
        if ($field_id === 'title') {
          return $titleItemField;
        }

        if ($field_id === 'status') {
          return $statusItemField;
        }

        return NULL;
      });

    $this->fieldTypeMapper->method('extractFieldValues')
      ->willReturnCallback(function ($field) use ($titleItemField): array {
        if ($field === $titleItemField) {
          return ['Test Title'];
        }

        return [1];
      });

    return $item;
  }

}
