<?php

declare(strict_types=1);

namespace Drupal\Tests\search_api_sqlite\Unit\Search;

use Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Search\Highlighter;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests the Highlighter service.
 */
#[CoversClass(Highlighter::class)]
#[Group('search_api_sqlite')]
final class HighlighterTest extends UnitTestCase {

  /**
   * The highlighter under test.
   */
  private Highlighter $highlighter;

  /**
   * The mocked FTS5 query runner.
   */
  private Fts5QueryRunnerInterface&MockObject $fts5QueryRunner;

  /**
   * The mocked schema manager.
   */
  private SchemaManagerInterface&MockObject $schemaManager;

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();

    $this->fts5QueryRunner = $this->createMock(Fts5QueryRunnerInterface::class);
    $this->schemaManager = $this->createMock(SchemaManagerInterface::class);

    // Configure schema manager to return expected table names.
    $this->schemaManager->method('getFtsTableName')
      ->willReturnCallback(fn(string $index_id): string => $index_id . '_fts');

    $this->highlighter = new Highlighter($this->fts5QueryRunner, $this->schemaManager);
  }

  /**
   * Tests getExcerpts() returns empty array for empty item IDs.
   */
  public function testGetExcerptsReturnsEmptyForEmptyItemIds(): void {
    $result = $this->highlighter->getExcerpts(
      'test_index',
      'search term',
      [],
      ['title', 'body'],
      []
    );

    $this->assertSame([], $result);
  }

  /**
   * Tests getExcerpts() returns empty array for empty columns.
   */
  public function testGetExcerptsReturnsEmptyForEmptyColumns(): void {
    $result = $this->highlighter->getExcerpts(
      'test_index',
      'search term',
      ['item_1', 'item_2'],
      [],
      []
    );

    $this->assertSame([], $result);
  }

  /**
   * Tests getExcerpts() returns empty array for empty query.
   */
  public function testGetExcerptsReturnsEmptyForEmptyQuery(): void {
    $result = $this->highlighter->getExcerpts(
      'test_index',
      '',
      ['item_1', 'item_2'],
      ['title'],
      []
    );

    $this->assertSame([], $result);
  }

  /**
   * Tests getExcerpts() returns excerpts for single column.
   */
  public function testGetExcerptsReturnsSingleColumnExcerpts(): void {
    $this->fts5QueryRunner->expects($this->once())
      ->method('snippet')
      ->with(
        'test_index',
        'test_index_fts',
        'search term',
        'title',
        ['item_1', 'item_2'],
        '<mark>',
        '</mark>',
        64
      )
      ->willReturn([
        'item_1' => '...found <mark>search term</mark> here...',
        'item_2' => '...<mark>search term</mark> in title...',
      ]);

    $result = $this->highlighter->getExcerpts(
      'test_index',
      'search term',
      ['item_1', 'item_2'],
      ['title'],
      []
    );

    $this->assertCount(2, $result);
    $this->assertArrayHasKey('item_1', $result);
    $this->assertArrayHasKey('item_2', $result);
    $this->assertArrayHasKey('title', $result['item_1']);
    $this->assertStringContainsString('<mark>search term</mark>', $result['item_1']['title']);
  }

  /**
   * Tests getExcerpts() returns excerpts for multiple columns.
   */
  public function testGetExcerptsReturnsMultipleColumnExcerpts(): void {
    $this->fts5QueryRunner->expects($this->exactly(2))
      ->method('snippet')
      ->willReturnCallback(function (
        string $index_id,
        string $table,
        string $query,
        string $column,
        array $item_ids,
      ): array {
        if ($column === 'title') {
          return [
            'item_1' => '<mark>test</mark> title',
          ];
        }

        return [
          'item_1' => '<mark>test</mark> body content',
        ];
      });

    $result = $this->highlighter->getExcerpts(
      'test_index',
      'test',
      ['item_1'],
      ['title', 'body'],
      []
    );

    $this->assertArrayHasKey('item_1', $result);
    $this->assertArrayHasKey('title', $result['item_1']);
    $this->assertArrayHasKey('body', $result['item_1']);
    $this->assertSame('<mark>test</mark> title', $result['item_1']['title']);
    $this->assertSame('<mark>test</mark> body content', $result['item_1']['body']);
  }

  /**
   * Tests getExcerpts() uses custom prefix and suffix.
   */
  public function testGetExcerptsUsesCustomPrefixAndSuffix(): void {
    $this->fts5QueryRunner->expects($this->once())
      ->method('snippet')
      ->with(
        $this->anything(),
        $this->anything(),
        $this->anything(),
        $this->anything(),
        $this->anything(),
        '<strong>',
        '</strong>',
        $this->anything()
      )
      ->willReturn([]);

    $this->highlighter->getExcerpts(
      'test_index',
      'query',
      ['item_1'],
      ['title'],
      ['prefix' => '<strong>', 'suffix' => '</strong>']
    );
  }

  /**
   * Tests getExcerpts() uses custom excerpt length.
   */
  public function testGetExcerptsUsesCustomExcerptLength(): void {
    $this->fts5QueryRunner->expects($this->once())
      ->method('snippet')
      ->with(
        $this->anything(),
        $this->anything(),
        $this->anything(),
        $this->anything(),
        $this->anything(),
        $this->anything(),
        $this->anything(),
        128
      )
      ->willReturn([]);

    $this->highlighter->getExcerpts(
      'test_index',
      'query',
      ['item_1'],
      ['title'],
      ['excerpt_length' => 128]
    );
  }

  /**
   * Tests getExcerpts() uses default options.
   */
  public function testGetExcerptsUsesDefaultOptions(): void {
    $this->fts5QueryRunner->expects($this->once())
      ->method('snippet')
      ->with(
        'test_index',
        'test_index_fts',
        'query',
        'title',
        ['item_1'],
        '<mark>',
        '</mark>',
        64
      )
      ->willReturn([]);

    $this->highlighter->getExcerpts(
      'test_index',
      'query',
      ['item_1'],
      ['title'],
      []
    );
  }

  /**
   * Tests getExcerpts() handles partial results from columns.
   */
  public function testGetExcerptsHandlesPartialResults(): void {
    // First column returns result for item_1 only.
    // Second column returns result for item_2 only.
    $callCount = 0;
    $this->fts5QueryRunner->method('snippet')
      ->willReturnCallback(function () use (&$callCount): array {
        $callCount++;
        if ($callCount === 1) {
          return ['item_1' => 'title excerpt'];
        }

        return ['item_2' => 'body excerpt'];
      });

    $result = $this->highlighter->getExcerpts(
      'test_index',
      'query',
      ['item_1', 'item_2'],
      ['title', 'body'],
      []
    );

    $this->assertArrayHasKey('item_1', $result);
    $this->assertArrayHasKey('item_2', $result);
    $this->assertArrayHasKey('title', $result['item_1']);
    $this->assertArrayNotHasKey('body', $result['item_1']);
    $this->assertArrayHasKey('body', $result['item_2']);
    $this->assertArrayNotHasKey('title', $result['item_2']);
  }

  /**
   * Tests getExcerpts() handles empty snippets from FTS5.
   */
  public function testGetExcerptsHandlesEmptySnippets(): void {
    $this->fts5QueryRunner->method('snippet')
      ->willReturn([]);

    $result = $this->highlighter->getExcerpts(
      'test_index',
      'query',
      ['item_1'],
      ['title'],
      []
    );

    $this->assertSame([], $result);
  }

  /**
   * Tests getExcerpts() uses SchemaManager for table name.
   */
  public function testGetExcerptsUsesSchemaManagerForTableName(): void {
    // Create a fresh mock to verify schemaManager is called.
    $schemaManager = $this->createMock(SchemaManagerInterface::class);
    $schemaManager->expects($this->once())
      ->method('getFtsTableName')
      ->with('products')
      ->willReturn('products_fts');

    $highlighter = new Highlighter($this->fts5QueryRunner, $schemaManager);

    $this->fts5QueryRunner->expects($this->once())
      ->method('snippet')
      ->with(
        'products',
        'products_fts',
        $this->anything(),
        $this->anything(),
        $this->anything(),
        $this->anything(),
        $this->anything(),
        $this->anything()
      )
      ->willReturn([]);

    $highlighter->getExcerpts(
      'products',
      'query',
      ['item_1'],
      ['title'],
      []
    );
  }

}
