<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Database;

/**
 * Interface for logging database queries.
 *
 * Provides timing utilities and logging for SQL queries executed by the
 * Search API SQLite backend. Useful for debugging and performance analysis.
 */
interface QueryLoggerInterface {

  /**
   * Logs a database query.
   *
   * @param string $sql
   *   The SQL query string.
   * @param array<string, mixed> $params
   *   The query parameters.
   * @param float $duration_ms
   *   The query execution duration in milliseconds.
   * @param string $source
   *   The source identifier (e.g., 'FTS5::search', 'FacetBuilder').
   */
  public function log(string $sql, array $params, float $duration_ms, string $source): void;

  /**
   * Starts a timer for measuring query execution time.
   *
   * @return float
   *   The start timestamp in microseconds.
   */
  public function startTimer(): float;

  /**
   * Calculates elapsed time since the timer started.
   *
   * @param float $start
   *   The start timestamp from startTimer().
   *
   * @return float
   *   The elapsed time in milliseconds.
   */
  public function endTimer(float $start): float;

  /**
   * Enables or disables query logging.
   *
   * @param bool $enabled
   *   TRUE to enable logging, FALSE to disable.
   */
  public function setEnabled(bool $enabled): void;

  /**
   * Checks if query logging is enabled.
   *
   * @return bool
   *   TRUE if logging is enabled.
   */
  public function isEnabled(): bool;

}
