<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Hook;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\search_api_sqlite\Enum\MatchingMode;
use Drupal\search_api_sqlite\Enum\Tokenizer;
use Drupal\search_api_sqlite\Utility\IndexSettings;

/**
 * Hook implementations for index form alterations.
 */
final class IndexFormHooks {

  use StringTranslationTrait;

  /**
   * The SQLite backend plugin ID.
   */
  private const string BACKEND_PLUGIN_ID = 'search_api_sqlite';

  /**
   * Constructs an IndexFormHooks instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * Implements hook_form_search_api_index_form_alter().
   *
   * Adds SQLite-specific settings to the index form as third-party settings.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $form_id
   *   The form ID.
   */
  #[Hook('form_search_api_index_form_alter')]
  public function formSearchApiIndexFormAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    // Only apply to index add/edit forms.
    if (!in_array($form_id, ['search_api_index_form', 'search_api_index_edit_form'], TRUE)) {
      return;
    }

    // Get visibility conditions for SQLite servers.
    $visibility = $this->getSqliteServerVisibility();
    if ($visibility === []) {
      // No SQLite servers configured, nothing to add.
      return;
    }

    // Get current settings from the index.
    $settings = [];
    /** @var \Drupal\Core\Entity\EntityFormInterface $form_object */
    $form_object = $form_state->getFormObject();
    /** @var \Drupal\search_api\IndexInterface $index */
    $index = $form_object->getEntity();
    if (!$index->isNew()) {
      $settings = $index->getThirdPartySettings(IndexSettings::MODULE_NAME);
    }

    $settings = IndexSettings::mergeDefaults($settings);

    // Add third-party settings container.
    $form['third_party_settings'][IndexSettings::MODULE_NAME] = [
      '#type' => 'details',
      '#title' => $this->t('SQLite FTS5 index options'),
      '#open' => TRUE,
      '#tree' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="server"]' => $visibility,
        ],
      ],
    ];

    $container = &$form['third_party_settings'][IndexSettings::MODULE_NAME];

    // Tokenizer settings.
    $container['tokenizer'] = [
      '#type' => 'select',
      '#title' => $this->t('Tokenizer'),
      '#description' => $this->t('<strong>Unicode61</strong>: Best for most languages. <strong>Porter Stemmer</strong>: English with word stemming. <strong>ASCII</strong>: Basic English. <strong>Trigram</strong>: Substring matching for codes/SKUs.'),
      '#options' => Tokenizer::formOptions(),
      '#default_value' => $settings['tokenizer'],
    ];

    $container['trigram_case_sensitive'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Case-sensitive matching'),
      '#description' => $this->t('Enable case-sensitive matching for trigram tokenizer.'),
      '#default_value' => $settings['trigram_case_sensitive'],
      '#states' => [
        'visible' => [
          ':input[name="third_party_settings[search_api_sqlite][tokenizer]"]' => ['value' => Tokenizer::Trigram->value],
        ],
      ],
    ];

    $container['min_chars'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum word length'),
      '#description' => $this->t('Minimum number of characters for a word to be indexed.'),
      '#default_value' => $settings['min_chars'],
      '#min' => 1,
      '#max' => 10,
      '#states' => [
        'invisible' => [
          ':input[name="third_party_settings[search_api_sqlite][tokenizer]"]' => ['value' => Tokenizer::Trigram->value],
        ],
      ],
    ];

    $container['matching'] = [
      '#type' => 'select',
      '#title' => $this->t('Default matching mode'),
      '#description' => $this->t('<strong>Match all words</strong>: All terms must appear. <strong>Prefix</strong>: Ideal for autocomplete. <strong>Phrase</strong>: Exact word sequence.'),
      '#options' => MatchingMode::formOptions(),
      '#default_value' => $settings['matching'],
    ];

    // Highlighting settings.
    $container['highlighting'] = [
      '#type' => 'details',
      '#title' => $this->t('Highlighting'),
      '#open' => FALSE,
    ];

    $container['highlighting']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable native FTS5 highlighting'),
      '#description' => $this->t('Use SQLite FTS5 built-in highlighting. Disable if using the Search API Highlighting processor.'),
      '#default_value' => $settings['highlighting']['enabled'],
    ];

    $container['highlighting']['prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Highlight prefix'),
      '#default_value' => $settings['highlighting']['prefix'],
      '#states' => [
        'visible' => [
          ':input[name="third_party_settings[search_api_sqlite][highlighting][enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $container['highlighting']['suffix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Highlight suffix'),
      '#default_value' => $settings['highlighting']['suffix'],
      '#states' => [
        'visible' => [
          ':input[name="third_party_settings[search_api_sqlite][highlighting][enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $container['highlighting']['excerpt_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Excerpt length'),
      '#description' => $this->t('Maximum number of tokens in excerpts.'),
      '#default_value' => $settings['highlighting']['excerpt_length'],
      '#min' => 32,
      '#max' => 512,
      '#states' => [
        'visible' => [
          ':input[name="third_party_settings[search_api_sqlite][highlighting][enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Optimization settings.
    $container['optimization'] = [
      '#type' => 'details',
      '#title' => $this->t('Optimization'),
      '#open' => FALSE,
    ];

    $container['optimization']['auto_optimize'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable automatic optimization'),
      '#description' => $this->t('Automatically optimize the FTS5 index after a threshold of changes.'),
      '#default_value' => $settings['optimization']['auto_optimize'],
    ];

    $container['optimization']['optimize_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Optimization threshold'),
      '#description' => $this->t('Number of changes before automatic optimization.'),
      '#default_value' => $settings['optimization']['optimize_threshold'],
      '#min' => 100,
      '#max' => 10000,
      '#states' => [
        'visible' => [
          ':input[name="third_party_settings[search_api_sqlite][optimization][auto_optimize]"]' => ['checked' => TRUE],
        ],
      ],
    ];
  }

  /**
   * Gets visibility conditions for servers using the SQLite backend.
   *
   * @return array<int, array<string, string>>
   *   Array of visibility conditions for #states.
   */
  private function getSqliteServerVisibility(): array {
    $visibility = [];

    try {
      /** @var \Drupal\search_api\ServerInterface[] $servers */
      $servers = $this->entityTypeManager
        ->getStorage('search_api_server')
        ->loadMultiple();

      foreach ($servers as $server) {
        if ($server->getBackendId() === self::BACKEND_PLUGIN_ID) {
          $visibility[] = ['value' => $server->id()];
        }
      }
    }
    catch (\Exception) {
      // Ignore exceptions during server loading.
    }

    return $visibility;
  }

}
