<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Index;

use Drupal\Core\State\StateInterface;
use Drupal\search_api\IndexInterface;
use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;
use Drupal\search_api_sqlite\Utility\ColumnNameHelper;
use Drupal\search_api_sqlite\Utility\VerboseLoggerTrait;
use Psr\Log\LoggerInterface;

/**
 * Indexes items into SQLite FTS5 and field data tables.
 */
final class Indexer implements IndexerInterface {

  use VerboseLoggerTrait;

  /**
   * Default maximum retry attempts for locked database.
   */
  private const int DEFAULT_MAX_RETRIES = 5;

  /**
   * Default retry delay in milliseconds.
   */
  private const int DEFAULT_RETRY_DELAY = 200;

  /**
   * The current backend configuration.
   *
   * @var array<string, mixed>|null
   */
  private ?array $backendConfig = NULL;

  /**
   * Constructs an Indexer instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The connection manager.
   * @param \Drupal\search_api_sqlite\Database\SchemaManagerInterface $schemaManager
   *   The schema manager.
   * @param \Drupal\search_api_sqlite\Index\FieldTypeMapperInterface $fieldTypeMapper
   *   The field type mapper.
   * @param \Drupal\search_api_sqlite\Database\Fts5QueryRunnerInterface $fts5QueryRunner
   *   The FTS5 query runner.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    private readonly ConnectionManagerInterface $connectionManager,
    private readonly SchemaManagerInterface $schemaManager,
    private readonly FieldTypeMapperInterface $fieldTypeMapper,
    private readonly Fts5QueryRunnerInterface $fts5QueryRunner,
    private readonly StateInterface $state,
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The index.
   * @param \Drupal\search_api\Item\ItemInterface[] $items
   *   The items to index.
   * @param array<string, mixed> $backend_config
   *   Backend configuration.
   *
   * @return array<int, string>
   *   Successfully indexed item IDs.
   */
  public function indexItems(IndexInterface $index, array $items, array $backend_config): array {
    if ($items === []) {
      return [];
    }

    // Store backend config for use in trackChanges.
    $this->backendConfig = $backend_config;
    $index_id = (string) $index->id();

    return $this->executeWithRetry(function () use ($index, $items, $index_id): array {
      $indexed_ids = [];
      $pdo = $this->connectionManager->getPdo($index_id);

      $fts_table = $this->schemaManager->getFtsTableName($index_id);
      $field_data_table = $this->schemaManager->getFieldDataTableName($index_id);
      $items_table = $this->schemaManager->getItemsTableName($index_id);

      // Get field definitions.
      $fields = $index->getFields();
      $fulltext_fields = [];
      $filter_fields = [];

      foreach ($fields as $field_id => $field) {
        if ($this->fieldTypeMapper->isFulltextType($field->getType())) {
          $fulltext_fields[$field_id] = $field;
        }
        else {
          $filter_fields[$field_id] = $field;
        }
      }

      // Build column list for FTS5 insert.
      $fts_columns = ['item_id'];
      foreach (array_keys($fulltext_fields) as $field_id) {
        $fts_columns[] = ColumnNameHelper::sanitize($field_id);
      }

      // Start transaction - use IMMEDIATE to acquire write lock early.
      $pdo->exec('BEGIN IMMEDIATE TRANSACTION');

      try {
        // Prepare statements.
        $fts_placeholders = array_fill(0, count($fts_columns), '?');
        $fts_sql = sprintf(
          'INSERT OR REPLACE INTO %s (%s) VALUES (%s)',
          $fts_table,
          implode(', ', $fts_columns),
          implode(', ', $fts_placeholders)
        );
        $fts_stmt = $pdo->prepare($fts_sql);

        // Prepare delete statements.
        $delete_fts_stmt = $pdo->prepare(
          sprintf('DELETE FROM %s WHERE item_id = ?', $fts_table)
        );
        $delete_field_data_stmt = $pdo->prepare(
          sprintf('DELETE FROM %s WHERE item_id = ?', $field_data_table)
        );
        $delete_items_stmt = $pdo->prepare(
          sprintf('DELETE FROM %s WHERE item_id = ?', $items_table)
        );

        // Prepare insert statement for field_data.
        $insert_field_data_stmt = $pdo->prepare(sprintf(
          'INSERT INTO %s (item_id, field_name, value_string, value_integer, value_decimal) VALUES (?, ?, ?, ?, ?)',
          $field_data_table
        ));

        // Prepare upsert statement for items tracking.
        $upsert_items_stmt = $pdo->prepare(sprintf(
          'INSERT OR REPLACE INTO %s (item_id, datasource, language, indexed_at) VALUES (?, ?, ?, ?)',
          $items_table
        ));

        foreach ($items as $item_id => $item) {
          // Delete existing data for this item.
          $delete_fts_stmt->execute([$item_id]);
          $delete_field_data_stmt->execute([$item_id]);
          $delete_items_stmt->execute([$item_id]);

          // Index fulltext fields in FTS5.
          $fts_values = [$item_id];
          foreach ($fulltext_fields as $field_id => $field_definition) {
            $item_field = $item->getField($field_id);
            $text = '';
            if ($item_field) {
              $values = $this->fieldTypeMapper->extractFieldValues($item_field);
              $text = implode(' ', $values);
            }

            $fts_values[] = $text;
          }

          $fts_stmt->execute($fts_values);

          // Index filter fields in field_data table.
          foreach ($filter_fields as $field_id => $field_definition) {
            $item_field = $item->getField($field_id);
            if (!$item_field) {
              continue;
            }

            $values = $this->fieldTypeMapper->extractFieldValues($item_field);
            $storage_type = $this->fieldTypeMapper->getStorageType($field_definition->getType());
            $column = $this->fieldTypeMapper->getFieldDataColumn($storage_type);

            foreach ($values as $value) {
              $params = [
                $item_id,
                $field_id,
                $column === 'value_string' ? $value : NULL,
                $column === 'value_integer' ? $value : NULL,
                $column === 'value_decimal' ? $value : NULL,
              ];
              $insert_field_data_stmt->execute($params);
            }
          }

          // Track the indexed item.
          $upsert_items_stmt->execute([
            $item_id,
            $item->getDatasourceId(),
            $item->getLanguage(),
            time(),
          ]);

          $indexed_ids[] = $item_id;
        }

        $pdo->exec('COMMIT');

        // Track changes for auto-optimization.
        $this->trackChanges($index_id, count($indexed_ids));

        $this->logVerbose('Indexed @count items for index @index', [
          '@count' => count($indexed_ids),
          '@index' => $index_id,
        ]);

        return $indexed_ids;
      }
      catch (\Exception $exception) {
        $pdo->exec('ROLLBACK');
        throw $exception;
      }
    }, $index_id);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteItems(IndexInterface $index, array $item_ids): void {
    if ($item_ids === []) {
      return;
    }

    $index_id = (string) $index->id();

    $this->executeWithRetry(function () use ($item_ids, $index_id): void {
      $pdo = $this->connectionManager->getPdo($index_id);

      $fts_table = $this->schemaManager->getFtsTableName($index_id);
      $field_data_table = $this->schemaManager->getFieldDataTableName($index_id);
      $items_table = $this->schemaManager->getItemsTableName($index_id);

      $pdo->exec('BEGIN IMMEDIATE TRANSACTION');

      try {
        $delete_fts_stmt = $pdo->prepare(
          sprintf('DELETE FROM %s WHERE item_id = ?', $fts_table)
        );
        $delete_field_data_stmt = $pdo->prepare(
          sprintf('DELETE FROM %s WHERE item_id = ?', $field_data_table)
        );
        $delete_items_stmt = $pdo->prepare(
          sprintf('DELETE FROM %s WHERE item_id = ?', $items_table)
        );

        foreach ($item_ids as $item_id) {
          $delete_fts_stmt->execute([(string) $item_id]);
          $delete_field_data_stmt->execute([(string) $item_id]);
          $delete_items_stmt->execute([(string) $item_id]);
        }

        $pdo->exec('COMMIT');

        $this->trackChanges($index_id, count($item_ids));

        $this->logVerbose('Deleted @count items from index @index', [
          '@count' => count($item_ids),
          '@index' => $index_id,
        ]);
      }
      catch (\Exception $exception) {
        $pdo->exec('ROLLBACK');
        throw $exception;
      }
    }, $index_id);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAllItems(IndexInterface $index, ?string $datasource_id = NULL): void {
    $index_id = (string) $index->id();

    $this->executeWithRetry(function () use ($index_id, $datasource_id): void {
      $pdo = $this->connectionManager->getPdo($index_id);

      $fts_table = $this->schemaManager->getFtsTableName($index_id);
      $field_data_table = $this->schemaManager->getFieldDataTableName($index_id);
      $items_table = $this->schemaManager->getItemsTableName($index_id);

      $pdo->exec('BEGIN IMMEDIATE TRANSACTION');

      try {
        if ($datasource_id === NULL) {
          // Delete all items.
          $pdo->exec(sprintf('DELETE FROM %s', $fts_table));
          $pdo->exec(sprintf('DELETE FROM %s', $field_data_table));
          $pdo->exec(sprintf('DELETE FROM %s', $items_table));
        }
        else {
          // Get item IDs for this datasource.
          $stmt = $pdo->prepare(
            sprintf('SELECT item_id FROM %s WHERE datasource = ?', $items_table)
          );
          $stmt->execute([$datasource_id]);
          $item_ids = $stmt->fetchAll(\PDO::FETCH_COLUMN);

          $delete_fts_stmt = $pdo->prepare(
            sprintf('DELETE FROM %s WHERE item_id = ?', $fts_table)
          );
          $delete_field_data_stmt = $pdo->prepare(
            sprintf('DELETE FROM %s WHERE item_id = ?', $field_data_table)
          );
          $delete_items_stmt = $pdo->prepare(
            sprintf('DELETE FROM %s WHERE item_id = ?', $items_table)
          );

          foreach ($item_ids as $item_id) {
            $delete_fts_stmt->execute([$item_id]);
            $delete_field_data_stmt->execute([$item_id]);
            $delete_items_stmt->execute([$item_id]);
          }
        }

        $pdo->exec('COMMIT');

        // Reset change counter.
        $this->state->delete('search_api_sqlite.changes.' . $index_id);

        $this->logVerbose('Deleted all items from index @index (datasource: @datasource)', [
          '@index' => $index_id,
          '@datasource' => $datasource_id ?? 'all',
        ]);
      }
      catch (\Exception $exception) {
        $pdo->exec('ROLLBACK');
        throw $exception;
      }
    }, $index_id);
  }

  /**
   * {@inheritdoc}
   */
  public function getIndexedItemsCount(IndexInterface $index): int {
    $index_id = (string) $index->id();

    try {
      $pdo = $this->connectionManager->getPdo($index_id);
      $items_table = $this->schemaManager->getItemsTableName($index_id);

      $result = $pdo->query(sprintf('SELECT COUNT(*) FROM %s', $items_table));

      if ($result === FALSE) {
        return 0;
      }

      return (int) $result->fetchColumn();
    }
    catch (\Exception $exception) {
      $this->logger->error('Error getting indexed items count for @index: @message', [
        '@index' => $index_id,
        '@message' => $exception->getMessage(),
      ]);
      return 0;
    }
  }

  /**
   * Executes a callback with retry logic for database locking.
   *
   * @param callable $callback
   *   The callback to execute.
   * @param string $index_id
   *   The index ID for logging.
   *
   * @return mixed
   *   The callback return value.
   *
   * @throws \Exception
   *   If all retries fail.
   */
  private function executeWithRetry(callable $callback, string $index_id): mixed {
    $attempts = 0;
    $last_exception = NULL;

    // Get retry settings from configuration or use defaults.
    $max_retries = $this->backendConfig['concurrency']['max_retries'] ?? self::DEFAULT_MAX_RETRIES;
    $retry_delay_ms = $this->backendConfig['concurrency']['retry_delay'] ?? self::DEFAULT_RETRY_DELAY;
    // Convert milliseconds to microseconds for usleep().
    $retry_delay = $retry_delay_ms * 1000;

    while ($attempts < $max_retries) {
      try {
        return $callback();
      }
      catch (\Exception $e) {
        $last_exception = $e;

        // Check if this is a database locked error.
        if (str_contains($e->getMessage(), 'database is locked') ||
            str_contains($e->getMessage(), 'SQLITE_BUSY')) {
          $attempts++;

          if ($attempts < $max_retries) {
            $this->logger->warning('Database locked for index @index, retry @attempt of @max', [
              '@index' => $index_id,
              '@attempt' => $attempts,
              '@max' => $max_retries,
            ]);

            // Wait before retrying with exponential backoff.
            usleep($retry_delay * $attempts);
            continue;
          }
        }

        // Not a locking error or max retries reached.
        throw $e;
      }
    }

    $this->logger->error('Failed after @max retries for index @index: @error', [
      '@max' => $max_retries,
      '@index' => $index_id,
      '@error' => $last_exception?->getMessage() ?? 'Unknown error',
    ]);

    throw $last_exception ?? new \RuntimeException('Unknown error during indexing');
  }

  /**
   * Tracks changes for auto-optimization.
   *
   * Increments the change counter and triggers optimization when the
   * configured threshold is exceeded.
   *
   * @param string $index_id
   *   The index ID.
   * @param int $count
   *   The number of changes.
   */
  private function trackChanges(string $index_id, int $count): void {
    $key = 'search_api_sqlite.changes.' . $index_id;
    $current = (int) $this->state->get($key, 0);
    $new_count = $current + $count;
    $this->state->set($key, $new_count);

    // Check if auto-optimization is enabled and threshold is reached.
    $auto_optimize = $this->backendConfig['optimization']['auto_optimize'] ?? FALSE;
    $threshold = $this->backendConfig['optimization']['optimize_threshold'] ?? 1000;

    if ($auto_optimize && $new_count >= $threshold) {
      $this->runOptimization($index_id);
    }
  }

  /**
   * Runs FTS5 optimization and resets the change counter.
   *
   * @param string $index_id
   *   The index ID.
   */
  private function runOptimization(string $index_id): void {
    try {
      $fts_table = $this->schemaManager->getFtsTableName($index_id);
      $this->fts5QueryRunner->optimize($index_id, $fts_table);

      // Reset the change counter.
      $key = 'search_api_sqlite.changes.' . $index_id;
      $this->state->set($key, 0);

      $this->logVerbose('Auto-optimized FTS5 index for @index', [
        '@index' => $index_id,
      ]);
    }
    catch (\Exception $exception) {
      $this->logger->warning('Auto-optimization failed for index @index: @error', [
        '@index' => $index_id,
        '@error' => $exception->getMessage(),
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function getVerboseLogger(): LoggerInterface {
    return $this->logger;
  }

  /**
   * {@inheritdoc}
   */
  protected function getVerboseLoggerConfig(): array {
    return $this->backendConfig ?? [];
  }

}
