<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Index;

use Drupal\search_api\IndexInterface;

/**
 * Defines the interface for indexing items into SQLite.
 */
interface IndexerInterface {

  /**
   * Indexes items into the SQLite database.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   * @param \Drupal\search_api\Item\ItemInterface[] $items
   *   The items to index, keyed by item ID.
   * @param array<string, mixed> $backend_config
   *   The backend configuration.
   *
   * @return array<int, string>
   *   Array of successfully indexed item IDs.
   */
  public function indexItems(IndexInterface $index, array $items, array $backend_config): array;

  /**
   * Deletes items from the index.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   * @param array<string> $item_ids
   *   The item IDs to delete.
   *
   * @throws \Drupal\search_api_sqlite\Exception\IndexingException
   *   If deletion fails.
   */
  public function deleteItems(IndexInterface $index, array $item_ids): void;

  /**
   * Deletes all items from the index.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   * @param string|null $datasource_id
   *   Optional datasource ID to limit deletion.
   *
   * @throws \Drupal\search_api_sqlite\Exception\IndexingException
   *   If deletion fails.
   */
  public function deleteAllItems(IndexInterface $index, ?string $datasource_id = NULL): void;

  /**
   * Gets the number of indexed items.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index.
   *
   * @return int
   *   The number of indexed items.
   */
  public function getIndexedItemsCount(IndexInterface $index): int;

}
