<?php

declare(strict_types=1);

namespace Drupal\search_api_sqlite\Search;

use Drupal\search_api_sqlite\Database\ConnectionManagerInterface;
use Drupal\search_api_sqlite\Database\QueryLoggerInterface;
use Drupal\search_api_sqlite\Database\SchemaManagerInterface;

/**
 * Builds facet queries and calculates facet counts.
 */
final readonly class FacetBuilder implements FacetBuilderInterface {

  /**
   * Constructs a FacetBuilder instance.
   *
   * @param \Drupal\search_api_sqlite\Database\ConnectionManagerInterface $connectionManager
   *   The connection manager.
   * @param \Drupal\search_api_sqlite\Database\SchemaManagerInterface $schemaManager
   *   The schema manager.
   * @param \Drupal\search_api_sqlite\Database\QueryLoggerInterface $queryLogger
   *   The query logger.
   */
  public function __construct(
    private ConnectionManagerInterface $connectionManager,
    private SchemaManagerInterface $schemaManager,
    private QueryLoggerInterface $queryLogger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function calculateFacets(
    string $index_id,
    string $field_name,
    array $item_ids,
    array $options = [],
  ): array {
    // Delegate to calculateMultipleFacets for consistent temp table approach.
    $results = $this->calculateMultipleFacets(
      $index_id,
      [$field_name => $options],
      $item_ids
    );

    return $results[$field_name] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function calculateMultipleFacets(
    string $index_id,
    array $field_options,
    array $item_ids,
  ): array {
    if ($item_ids === [] || $field_options === []) {
      // Return empty arrays for each requested facet.
      return array_fill_keys(array_keys($field_options), []);
    }

    // Choose strategy based on item count.
    // Small result sets: IN() clause is faster (no temp table overhead).
    // Large result sets: Temp table JOIN is more efficient.
    if (!$this->connectionManager->shouldUseTempTable(count($item_ids))) {
      return $this->calculateFacetsWithInClause($index_id, $field_options, $item_ids);
    }

    return $this->calculateFacetsWithTempTable($index_id, $field_options, $item_ids);
  }

  /**
   * Calculates facets using IN() clause for small result sets.
   *
   * This approach avoids the overhead of creating/populating/dropping
   * a temp table, which is beneficial for small result sets.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param array<string, array<string, mixed>> $field_options
   *   Array of field options keyed by field name.
   * @param array<int, string> $item_ids
   *   The item IDs to calculate facets for.
   *
   * @return array<string, array<int, array{value: mixed, count: int}>>
   *   Array of facet results keyed by field name.
   */
  private function calculateFacetsWithInClause(
    string $index_id,
    array $field_options,
    array $item_ids,
  ): array {
    $connection = $this->connectionManager->getConnection($index_id);
    $field_data_table = $this->schemaManager->getFieldDataTableName($index_id);
    $field_names = array_keys($field_options);

    $query = $connection->select($field_data_table, 'fd');
    $query->addField('fd', 'field_name');
    $query->addExpression(
      "COALESCE(fd.value_string, CAST(fd.value_integer AS TEXT), CAST(fd.value_decimal AS TEXT))",
      'value'
    );
    $query->addExpression('COUNT(DISTINCT fd.item_id)', 'count');
    $query->condition('fd.field_name', $field_names, 'IN');
    $query->condition('fd.item_id', $item_ids, 'IN');
    $query->groupBy('fd.field_name');
    $query->groupBy('value');
    $query->orderBy('fd.field_name');
    $query->orderBy('count', 'DESC');

    $start = $this->queryLogger->startTimer();
    $result = $query->execute();
    $duration = $this->queryLogger->endTimer($start);

    $this->queryLogger->log((string) $query, [
      'field_names' => $field_names,
      'item_ids_count' => count($item_ids),
    ], $duration, 'FacetBuilder::calculateFacetsWithInClause');

    // Convert to standard row format.
    $rows = [];
    if ($result) {
      foreach ($result as $row) {
        $rows[] = (object) [
          'field_name' => $row->field_name,
          'value' => $row->value,
          'count' => (int) $row->count,
        ];
      }
    }

    return $this->groupAndFilterResults($rows, $field_options);
  }

  /**
   * Calculates multiple facets using a temp table for efficiency.
   *
   * This method creates a temporary table with the search result item IDs,
   * then executes a single query to calculate all facets at once using a JOIN.
   * This is significantly faster than running separate queries for each facet.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param array<string, array<string, mixed>> $field_options
   *   Array of field options keyed by field name.
   * @param array<int, string> $item_ids
   *   The item IDs to calculate facets for.
   *
   * @return array<string, array<int, array{value: mixed, count: int}>>
   *   Array of facet results keyed by field name.
   */
  private function calculateFacetsWithTempTable(
    string $index_id,
    array $field_options,
    array $item_ids,
  ): array {
    $pdo = $this->connectionManager->getPdo($index_id);
    $field_data_table = $this->schemaManager->getFieldDataTableName($index_id);

    // Create and populate temp table with search result item IDs.
    $temp_table = $this->connectionManager->createTempItemsTable($index_id, $item_ids, 'facet_results');

    // Build single query to get all facet counts.
    $field_names = array_keys($field_options);
    $placeholders = implode(', ', array_fill(0, count($field_names), '?'));

    $sql = sprintf(
      'SELECT
        fd.field_name,
        COALESCE(fd.value_string, CAST(fd.value_integer AS TEXT), CAST(fd.value_decimal AS TEXT)) AS value,
        COUNT(DISTINCT fd.item_id) AS count
      FROM %s fd
      INNER JOIN %s sr ON fd.item_id = sr.item_id
      WHERE fd.field_name IN (%s)
      GROUP BY fd.field_name, value
      ORDER BY fd.field_name, count DESC',
      $field_data_table,
      $temp_table,
      $placeholders
    );

    $start = $this->queryLogger->startTimer();
    $stmt = $pdo->prepare($sql);
    $stmt->execute($field_names);

    $rows = $stmt->fetchAll(\PDO::FETCH_OBJ);
    $duration = $this->queryLogger->endTimer($start);

    $this->queryLogger->log($sql, [
      'field_names' => $field_names,
      'item_ids_count' => count($item_ids),
    ], $duration, 'FacetBuilder::calculateFacetsWithTempTable');

    // Clean up temp table.
    $this->connectionManager->dropTempTable($index_id, $temp_table);

    // Group results by field and apply per-facet limits.
    return $this->groupAndFilterResults($rows, $field_options);
  }

  /**
   * Groups raw facet results by field and applies per-facet limits.
   *
   * @param array<int, object> $rows
   *   Raw result rows with field_name, value, and count properties.
   * @param array<string, array<string, mixed>> $field_options
   *   Array of field options keyed by field name.
   *
   * @return array<string, array<int, array{value: mixed, count: int}>>
   *   Array of facet results keyed by field name.
   */
  private function groupAndFilterResults(array $rows, array $field_options): array {
    // Initialize results with empty arrays for all requested fields.
    $results = array_fill_keys(array_keys($field_options), []);

    // Group rows by field name.
    $grouped = [];
    foreach ($rows as $row) {
      $field_name = $row->field_name;
      if (!isset($grouped[$field_name])) {
        $grouped[$field_name] = [];
      }

      $grouped[$field_name][] = [
        'value' => $row->value,
        'count' => (int) $row->count,
      ];
    }

    // Apply per-facet limit and min_count.
    foreach ($field_options as $field_name => $options) {
      if (!isset($grouped[$field_name])) {
        continue;
      }

      $limit = $options['limit'] ?? 10;
      $min_count = $options['min_count'] ?? 1;

      $filtered = [];
      foreach ($grouped[$field_name] as $facet) {
        if ($facet['count'] < $min_count) {
          continue;
        }

        $filtered[] = $facet;
        if ($limit > 0 && count($filtered) >= $limit) {
          break;
        }
      }

      $results[$field_name] = $filtered;
    }

    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function setQueryLoggingEnabled(bool $enabled): void {
    $this->queryLogger->setEnabled($enabled);
  }

}
